﻿//-----------------------------------------------------------------------------
// <copyright file="UsbI2C.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// UsbI2C
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.IO;
    using System.Text;
    using System.Threading;

    /// <summary>
    /// Wrapper for a connection to a USB-I2C.
    /// </summary>
    public class UsbI2C
    {
        /// <summary>
        /// Data frame reader.
        /// </summary>
        private StreamReader reader;

        /// <summary>
        /// Data frame writer.
        /// </summary>
        private StreamWriter writer;

        /// <summary>
        /// Base stream of the device.
        /// </summary>
        private Stream baseStream;

        /// <summary>
        /// Initializes a new instance of the UsbI2C class.
        /// </summary>
        /// <param name="baseStream">Base stream of the device.</param>
        public UsbI2C(Stream baseStream)
        {
            this.baseStream = baseStream;
            this.baseStream.ReadTimeout = 50;
            this.reader = new StreamReader(this.baseStream, new ASCIIEncoding());
            this.writer = new StreamWriter(this.baseStream, new ASCIIEncoding());
        }

        /// <summary>
        /// Gets base stream of the device.
        /// </summary>
        /// <value>Base stream of the device.</value>
        public Stream BaseStream
        {
            get { return this.baseStream; }
        }

        /// <summary>
        /// Reads the remaining buffer.
        /// </summary>
        /// <returns>Read values.</returns>
        public string ReadBuffer()
        {
            string status = string.Empty;

            try
            {
                status = this.reader.ReadToEnd();
                return status;
            }
            catch (Exception)
            {
                return status;
            }
        }

        /// <summary>
        /// Returns current status of USB-I2C.
        /// </summary>
        /// <returns>Current status of USB-I2C.</returns>
        public string GetStatus()
        {
            try
            {
                this.writer.WriteLine("?");
                this.writer.Flush();
                string status = this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                return status;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Resets the y-parameter.
        /// </summary>
        /// <returns>Returns current status string.</returns>
        public string ResetY()
        {
            try
            {
                this.writer.WriteLine("y00 y10 y20 y30 y40 y50 y60 y70 ?");
                this.writer.Flush();
                Thread.Sleep(500);
                string status = this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                status += this.reader.ReadLine() + "\r\n";
                return status;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Resets the USB-I2C.
        /// </summary>
        /// <returns>Returns current status string.</returns>
        public string Reset()
        {
            try
            {
                this.writer.WriteLine("z4b");
                this.writer.Flush();
                Thread.Sleep(2000);
                string result = this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                return result;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Resets completely the USB-I2C.
        /// </summary>
        /// <returns>Returns current status string.</returns>
        public string FullReset()
        {
            try
            {
                this.writer.WriteLine("zAA");
                this.writer.Flush();
                Thread.Sleep(3000);
                string result = this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                return result;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Gets current macro from the USB-I2C.
        /// </summary>
        /// <returns>Returns read macro.</returns>
        public string GetMakro()
        {
            try
            {
                this.writer.WriteLine("u");
                this.writer.Flush();
                string result = this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                return result;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Clears macro memory of the USB-I2C.
        /// </summary>
        /// <returns>Returns current status string.</returns>
        public string DeleteMakro()
        {
            try
            {
                this.writer.WriteLine("u v00{} u");
                this.writer.Flush();
                string result = this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                result += this.reader.ReadLine() + "\r\n";
                return result;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Sends a command to the I2C-interface.
        /// </summary>
        /// <param name="command">Command string.</param>
        /// <returns>Result string.</returns>
        public string SendReceiveCommand(string command)
        {
            try
            {
                this.ReadBuffer();
                this.writer.WriteLine(command);
                this.writer.Flush();
                string result = this.reader.ReadLine();

                return result;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        /// <summary>
        /// Sends a command to the I2C-interface securely.
        /// </summary>
        /// <param name="command">Command string.</param>
        /// <returns>Result string.</returns>
        public string SendTryReceiveCommand(string command)
        {
            string result = string.Empty;

            try
            {
                this.ReadBuffer();
                this.writer.WriteLine(command);
                this.writer.Flush();

                while (!this.reader.EndOfStream)
                {
                    result += this.reader.ReadLine() + "\r\n";
                }

                return result;
            }
            catch (TimeoutException)
            {
                return result;
            }
            catch (Exception)
            {
                return null;
            }
        }

        /// <summary>
        /// Sends a command to the I2C-interface with receiving a result.
        /// </summary>
        /// <param name="command">Command string.</param>
        /// <returns>Result string.</returns>
        public int SendReceiveByteCommand(string command)
        {
            try
            {
                this.ReadBuffer();
                this.writer.WriteLine(command);
                this.writer.Flush();
                char highnibble = (char)this.reader.Read();
                char lownibble = (char)this.reader.Read();
                StringBuilder hexstring = new StringBuilder();
                hexstring.Append(highnibble);
                hexstring.Append(lownibble);

                // int result = this.reader.Read();
                return int.Parse(hexstring.ToString(), System.Globalization.NumberStyles.HexNumber);
            }
            catch (Exception)
            {
                return -1;
            }
        }

        /// <summary>
        /// Sends a command to the I2C-interface without receiving a result.
        /// </summary>
        /// <param name="command">Command string.</param>
        /// <returns>Result string.</returns>
        public bool SendCommand(string command)
        {
            try
            {
                this.ReadBuffer();
                this.writer.WriteLine(command);
                this.writer.Flush();
                return true;
            }
            catch (Exception)
            {
                return false;
            }
        }
    }
}
