﻿//-----------------------------------------------------------------------------
// <copyright file="MainForm.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Funktionalitaet des Hauptfensters
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Drawing;
    using System.IO;
    using System.IO.Ports;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;
    using System.Windows.Forms.DataVisualization.Charting;

    /// <summary>
    /// Main form of the application.
    /// </summary>
    public partial class MainForm : Form
    {
        #region Konstanten der Verbindung
        /// <summary>
        /// Baud-rate (serial port).
        /// </summary>
        private const int BAUDRATE = 115200;

        /// <summary>
        /// Data-bits (serial port).
        /// </summary>
        private const int DATABITS = 8;

        /// <summary>
        /// Parity (serial port).
        /// </summary>
        private const Parity PARITY = Parity.None;

        /// <summary>
        /// Stopp-bits (serial port).
        /// </summary>
        private const StopBits STOPBITS = StopBits.One;

        /// <summary>
        /// Read-out timeout (serial port).
        /// </summary>
        private const int READTIMEOUT = 1000; // 1 Sekunden
        #endregion

        /// <summary>
        /// USB I2C Interface.
        /// </summary>
        private Eq3.misc.USBI2C.UsbI2C usbI2C;

        /// <summary>
        /// Flag for connection with I2C.
        /// </summary>
        private bool isConnected;

        /// <summary>
        /// Serial port.
        /// </summary>
        private SerialPort serialPort;

        /// <summary>
        /// Port name of connection.
        /// </summary>
        private string portName = string.Empty;

        /// <summary>
        /// Current position for list iteration.
        /// </summary>
        private int listPosition = 0;

        /// <summary>
        /// Amount of expected ticks for list iteration.
        /// </summary>
        private int waitcounter = 0;

        /// <summary>
        /// Current temperature of MCP9801 in Celsius.
        /// </summary>
        private double temperatureCelsius = 0.0;

        /// <summary>
        /// Current temperature of MCP9801 in Fahrenheit.
        /// </summary>
        private double temperatureFahrenheit = 0.0;

        /// <summary>
        /// Current temperature of MCP9801 in Kelvin.
        /// </summary>
        private double temperatureKelvin = 0.0;

        /// <summary>
        /// Flag for connection with I2C-4DLED-MCP9801.
        /// </summary>
        private bool connectedWithMCP9801 = false;

        /// <summary>
        /// Current alarm value of MCP9801'.
        /// </summary>
        private double alert = 0.0;

        /// <summary>
        /// Timer for triggering the SAA1064 after a specific intervall.
        /// </summary>
        private Timer timerTriggerSaa1064;

        /// <summary>
        /// Timer for triggering the MCP9801 after a specific intervall.
        /// </summary>
        private Timer timerTriggerMcp9801;

        /// <summary>
        /// Timer for showing the current time on I2C-4DLED-SAA1064.
        /// </summary>
        private Timer timerShowTimeOnSaa1064;
        
        /// <summary>
        /// Timer for refreshing the chart of the linear acceleration sensor of the 6D-BS.
        /// </summary>
        private Timer timerTriggerChart6dbsLa;

        /// <summary>
        /// Timer for refreshing the chart of the angular rate sensor of the 6D-BS.
        /// </summary>
        private Timer timerTriggerChart6dbsAr;

        /// <summary>
        /// List of different cut-off frequencies which depends on 6D-BS' ODR.
        /// </summary>
        private List<List<string>> odrList;

        /// <summary>
        /// Initializes a new instance of the MainForm class.
        /// </summary>
        public MainForm()
        {
            this.InitializeComponent();
            this.Scale(new System.Drawing.SizeF((this.CreateGraphics().DpiX - ((this.CreateGraphics().DpiX - 96) / 2)) / 96, (this.CreateGraphics().DpiY - ((this.CreateGraphics().DpiY - 96) / 2)) / 96));
            
            this.isConnected = false;
            this.Disable();

            this.UpdateComList();
        }

        /// <summary>
        /// Finalizes an instance of the MainForm class.
        /// </summary>
        ~MainForm()
        {
            this.Disconnect();
        }

        /// <summary>
        /// Delegate for updating the selected index of ListBoxLEDsequence.
        /// </summary>
        private delegate void DelegateSetSelectedIndexOfListBoxLEDsequence();

        /// <summary>
        /// Delegate for adding a command to the list of send commands.
        /// </summary>
        /// <param name="command">Command to add.</param>
        private delegate void DelegateAddCommandToListBoxOutput(string command);

        /// <summary>
        /// Gets or sets wrapper for I2C-4DLED.
        /// </summary>
        /// <value>Wrapper for I2C-4DLED.</value>
        public I2C4DLed I2C4DLed { get; set; }

        /// <summary>
        /// Gets or sets wrapper for I2C-LCD.
        /// </summary>
        /// <value>Wrapper for I2C-LCD.</value>
        public LCD Lcd { get; set; }

        /// <summary>
        /// Gets or sets wrapper for I2C-6D-BS.
        /// </summary>
        /// <value>Wrapper for I2C-I2C-6D-BS.</value>
        public I2C6DBS I2C6Dbs { get; set; }

        /// <summary>
        /// EventHandler for ListBox for copying.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ListBox_KeyDown(object sender, KeyEventArgs e)
        {
            // copy
            if (e.KeyCode == Keys.C && e.Modifiers == Keys.Control)
            {
                string markedText = string.Empty;
                ListBox tempBox = (ListBox)sender;

                for (int i = 0; i < tempBox.SelectedItems.Count; i++)
                {
                    markedText += tempBox.SelectedItems[i].ToString() + "\r\n";
                }

                Clipboard.SetText(markedText);
            }
        }

        /// <summary>
        /// EventHandler for ListBox for copy and paste.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ListBoxList_KeyDown(object sender, KeyEventArgs e)
        {
            // copy
            if (e.KeyCode == Keys.C && e.Modifiers == Keys.Control)
            {
                string markedText = string.Empty;
                ListBox tempBox = (ListBox)sender;

                for (int i = 0; i < tempBox.SelectedItems.Count; i++)
                {
                    markedText += tempBox.SelectedItems[i].ToString() + "\r\n";
                }

                Clipboard.SetText(markedText);
            } 
            
            // paste
            if (e.KeyCode == Keys.V && e.Modifiers == Keys.Control)
            {
                ListBox tempBox = (ListBox)sender;
                string content = Clipboard.GetText();

                tempBox.Items.AddRange(content.Trim().Split('\r'));
            }

            // remove
            if (e.KeyCode == Keys.Delete)
            {
                ListBox tempBox = (ListBox)sender;
                
                while (tempBox.SelectedItems.Count != 0)
                {
                    tempBox.Items.Remove(tempBox.SelectedItem);
                }
            }
        }

        /// <summary>
        /// Shows an specific error.
        /// </summary>
        /// <param name="errorID">Specific error id.</param>
        private void ShowError(int errorID)
        {
            this.listBoxInput.Items.Add(this.usbI2C.ReadBuffer());
            this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

            switch (errorID)
            {
                // Allgemeiner Fehler
                case 0:
                    MessageBox.Show("Der Befehl konnte nicht gesendet werden oder es kam keine Antwort. Überprüfen Sie die Verbindung zum Gerät");
                    this.Disconnect();
                    break;

                // iSMT
                case 1:
                    MessageBox.Show("Der Status konnte nicht gelesen werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;
                case 2:
                    MessageBox.Show("Die MotorParameter konnten nicht gesetzt werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;
                case 3:
                    MessageBox.Show("Die StallParameter konnten nicht gesetzt werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;
                case 4:
                    MessageBox.Show("Der Befehl konnte nicht gesendet werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;
                case 5:
                    MessageBox.Show("Der OTP konnte nicht gesetzt werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;
                case 6:
                    MessageBox.Show("Fehler beim Abarbeiten der Liste. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    this.timerSequenz.Enabled = false;
                    break;
                case 7:
                    MessageBox.Show("Fehler beim Speichern der Liste. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen iSMT und USB-I2C");
                    break;

                // 3D-BS
                case 10:
                    MessageBox.Show("Der Status konnte nicht gelesen werden. Bitte überprüfen Sie die Verbindung zwischen 3D-BS und USB-I2C");
                    break;

                // LED-I2C
                case 20:
                    MessageBox.Show("Die Information konnte nicht gelesen werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    break;
                case 21:
                    MessageBox.Show("Der Status bzw. die Helligkeit konnte nicht geschrieben werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    this.checkBoxLEDpwmAutoSend.Checked = false;
                    break;
                case 22:
                    MessageBox.Show("Fehler beim Speichern der Liste. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    break;
                case 23:
                    MessageBox.Show("Fehler beim Abarbeiten der Liste. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    this.ButtonLEDStop_Click(null, null);
                    break;
                case 24:
                    MessageBox.Show("Der Mode konnte nicht geschrieben werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    break;
                case 25:
                    MessageBox.Show("Die Subadressen konnten nicht geschrieben werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    break;
                case 26:
                    MessageBox.Show("Die Gruppeneinstellungen (GRPFREQ und GRPPWM) konnten nicht geschrieben werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    this.checkBoxLEDgrpAutoSend.Checked = false;
                    break;
                case 27:
                    MessageBox.Show("Das IREF konnte nicht geschrieben werden. Bitte überprüfen Sie die I²C-Adresse und die Verbindung zwischen LED-I2C und USB-I2C");
                    break;
                default:
                    break;
            }
        }

        #region MainForm Funktionen
        /// <summary>
        /// Refreshes the list of comports.
        /// </summary>
        private void UpdateComList()
        {
            // PortsComboBox
            string[] portNames = SerialPort.GetPortNames();
            this.PortsComboBox.Items.Clear();
            if (0 < portNames.Length)
            {
                this.PortsComboBox.Items.AddRange(portNames);
                this.PortsComboBox.SelectedItem = this.PortsComboBox.Items[0];
            }
            else
            {
                this.ConnectButton.Enabled = false;
            }
        }

        /// <summary>
        /// Refreshes the content of ComboBox while opening.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void PortsComboBox_DropDown(object sender, EventArgs e)
        {
            this.UpdateComList();
        }
        
        /// <summary>
        /// Deactivates control elements.
        /// </summary>
        private void Disable()
        {
            this.ConnectionLabel.Text = "- nicht verbunden -";
            this.ConnectionLabel.BackColor = Color.Red;

            this.buttoniSMTGetFullStatus1.Enabled = false;
            this.buttoniSMTGetFullStatus2.Enabled = false;
            this.buttoniSMTGetOTPParameter.Enabled = false;

            this.buttoniSMTResetPosition.Enabled = false;
            this.buttoniSMTResetToDefault.Enabled = false;
            this.buttoniSMTSetMotorParam.Enabled = false;
            this.buttoniSMTSetPosition.Enabled = false;
            this.buttoniSMTSetStallParam.Enabled = false;

            this.buttoniSMTGotoSP.Enabled = false;
            this.buttoniSMTHardStop.Enabled = false;
            this.buttoniSMTRunInit.Enabled = false;
            this.buttoniSMTRunVelocity.Enabled = false;
            this.buttoniSMTSetOTP.Enabled = false;
            this.buttoniSMTSoftStop.Enabled = false;
            this.buttoniSMTTestBEMF.Enabled = false;
            this.buttoniSMTStart.Enabled = false;

            this.buttonI2CDeliveryreset.Enabled = false;
            this.buttonI2CDelMakro.Enabled = false;
            this.buttonI2CReset.Enabled = false;
            this.buttonI2CShowMakro.Enabled = false;
            this.buttonI2CStartMakro.Enabled = false;
            this.buttonI2CStatusFirmware.Enabled = false;
            this.buttonI2CStopMakro.Enabled = false;
            this.buttonI2CYReset.Enabled = false;
            this.buttonI2Cdirect.Enabled = false;

            this.button3DBSStatus.Enabled = false;
            this.button3DBSValues.Enabled = false;

            this.buttonLEDgetPWM.Enabled = false;
            this.buttonLEDgetStatus.Enabled = false;
            this.buttonLEDListToMakro.Enabled = false;
            this.buttonLEDsetPWM.Enabled = false;
            this.buttonLEDsetStatus.Enabled = false;
            this.buttonLEDStart.Enabled = false;
            this.buttonLEDeflags.Enabled = false;
            this.buttonLEDgetMode1.Enabled = false;
            this.buttonLEDgetMode2.Enabled = false;
            this.buttonLEDsetMode1.Enabled = false;
            this.buttonLEDsetMode2.Enabled = false;
            this.buttonLEDgetGroup.Enabled = false;
            this.buttonLEDsetGroup.Enabled = false;
            this.buttonLEDgetSubadr.Enabled = false;
            this.buttonLEDsetSubadr.Enabled = false;
            this.buttonLEDgetIREF.Enabled = false;
            this.buttonLEDsetIREF.Enabled = false;
            this.checkBoxLEDpwmAutoSend.Checked = false;
            this.checkBoxLEDpwmAutoSend.Enabled = false;
            this.checkBoxLEDgrpAutoSend.Checked = false;
            this.checkBoxLEDgrpAutoSend.Enabled = false;

            this.btI2CLCDInit.Enabled = false;
            this.btI2CLCDlcdClear.Enabled = false;
            this.btI2CLCDlcdControl.Enabled = false;
            this.btI2CLCDLED.Enabled = false;
            this.btI2CLCDTaster.Enabled = false;
            this.chbI2CLCDled1.Enabled = false;
            this.chbI2CLCDled2.Enabled = false;
            this.chbI2CLCDled3.Enabled = false;
            this.chbI2CLCDled4.Enabled = false;

            this.btSaa1064CommitNumbers.Enabled = false;
            this.btSaaTrigger.Enabled = false;
            this.comboBoxSaa1064NumbersPatternChoice.Enabled = false;
            this.comboBoxSaa1064CurrentChoice.Enabled = false;
            this.tbSaa1064CustomNumbers.Enabled = false;
            this.comboBoxSaa1064CurrentChoice.Enabled = false;
            this.comboBoxSaa1064SlaveAddress.Enabled = false;
            this.comboBoxMCP9801SlaveAddress.Enabled = false;
            this.btMCPTrigger.Enabled = false;
            this.rdMCPCelsius.Enabled = false;
            this.rdMCPFahrenheit.Enabled = false;
            this.rdMCPKelvin.Enabled = false;
            this.tbMCPAlarmValueWrite.Enabled = false;
            this.gbMCP9801Einstellungen.Enabled = false;

            this.gbSaa1064Trigger.Enabled = false;
            this.gbMCP9801Trigger.Enabled = false;

            this.btn6DbsLaSetSettings.Enabled = false;
            this.btn6DbsLaReadSettings.Enabled = false;
            this.btn6DbsLaResetSettings.Enabled = false;
            this.btn6dbsLaToggleChartRefreshing.Enabled = false;
            this.btn6dbsLaReadStatus.Enabled = false;
            this.btn6DbsArSetSettings.Enabled = false;
            this.btn6dbsArToggleChartRefreshing.Enabled = false;
            this.btn6dbsArReadStatus.Enabled = false;
            this.btn6DbsArReadSettings.Enabled = false;
            this.btn6DbsArResetSettings.Enabled = false;

            this.connectedWithMCP9801 = false;
        }

        /// <summary>
        /// Activates control elements.
        /// </summary>
        private void Enable()
        {
            this.ConnectionLabel.BackColor = Color.Green;

            this.buttoniSMTGetFullStatus1.Enabled = true;
            this.buttoniSMTGetFullStatus2.Enabled = true;
            this.buttoniSMTGetOTPParameter.Enabled = true;

            this.buttoniSMTResetPosition.Enabled = true;
            this.buttoniSMTResetToDefault.Enabled = true;
            this.buttoniSMTSetMotorParam.Enabled = true;
            this.buttoniSMTSetPosition.Enabled = true;
            this.buttoniSMTSetStallParam.Enabled = true;

            this.buttoniSMTGotoSP.Enabled = true;
            this.buttoniSMTHardStop.Enabled = true;
            this.buttoniSMTRunInit.Enabled = true;
            this.buttoniSMTRunVelocity.Enabled = true;
            this.buttoniSMTSetOTP.Enabled = true;
            this.buttoniSMTSoftStop.Enabled = true;
            this.buttoniSMTTestBEMF.Enabled = true;
            this.buttoniSMTStart.Enabled = true;

            this.buttonI2CDeliveryreset.Enabled = true;
            this.buttonI2CDelMakro.Enabled = true;
            this.buttonI2CReset.Enabled = true;
            this.buttonI2CShowMakro.Enabled = true;
            this.buttonI2CStartMakro.Enabled = true;
            this.buttonI2CStatusFirmware.Enabled = true;
            this.buttonI2CStopMakro.Enabled = true;
            this.buttonI2CYReset.Enabled = true;
            this.buttonI2Cdirect.Enabled = true;

            this.button3DBSStatus.Enabled = true;
            this.button3DBSValues.Enabled = true;

            this.buttonLEDgetPWM.Enabled = true;
            this.buttonLEDgetStatus.Enabled = true;
            this.buttonLEDListToMakro.Enabled = true;
            this.buttonLEDsetPWM.Enabled = true;
            this.buttonLEDsetStatus.Enabled = true;
            this.buttonLEDStart.Enabled = true;
            this.buttonLEDeflags.Enabled = true;
            this.buttonLEDgetMode1.Enabled = true;
            this.buttonLEDgetMode2.Enabled = true;
            this.buttonLEDsetMode1.Enabled = true;
            this.buttonLEDsetMode2.Enabled = true;
            this.buttonLEDgetGroup.Enabled = true;
            this.buttonLEDsetGroup.Enabled = true;
            this.buttonLEDgetSubadr.Enabled = true;
            this.buttonLEDsetSubadr.Enabled = true;
            this.buttonLEDgetIREF.Enabled = true;
            this.buttonLEDsetIREF.Enabled = true;
            this.checkBoxLEDpwmAutoSend.Enabled = true;
            this.checkBoxLEDgrpAutoSend.Enabled = true;

            this.btI2CLCDInit.Enabled = true;
            this.btI2CLCDlcdClear.Enabled = true;
            this.btI2CLCDlcdControl.Enabled = true;
            this.btI2CLCDLED.Enabled = true;
            this.btI2CLCDTaster.Enabled = true;
            this.chbI2CLCDled1.Enabled = true;
            this.chbI2CLCDled2.Enabled = true;
            this.chbI2CLCDled3.Enabled = true;
            this.chbI2CLCDled4.Enabled = true;

            this.btSaa1064CommitNumbers.Enabled = true;
            this.btSaaTrigger.Enabled = true;
            this.comboBoxSaa1064NumbersPatternChoice.Enabled = true;
            this.comboBoxSaa1064CurrentChoice.Enabled = false;
            this.tbSaa1064CustomNumbers.Enabled = true;
            this.comboBoxSaa1064CurrentChoice.Enabled = true;
            this.comboBoxSaa1064SlaveAddress.Enabled = true;
            this.comboBoxMCP9801SlaveAddress.Enabled = true;
            this.btMCPTrigger.Enabled = true;
            this.rdMCPCelsius.Enabled = true;
            this.rdMCPFahrenheit.Enabled = true;
            this.rdMCPKelvin.Enabled = true;
            this.tbMCPAlarmValueWrite.Enabled = true;
            this.gbMCP9801Einstellungen.Enabled = true;

            this.gbSaa1064Trigger.Enabled = true;
            this.gbMCP9801Trigger.Enabled = true;

            this.btn6DbsLaSetSettings.Enabled = true;
            this.btn6DbsLaReadSettings.Enabled = true;
            this.btn6DbsLaResetSettings.Enabled = true;
            this.btn6dbsLaToggleChartRefreshing.Enabled = true;
            this.btn6dbsLaReadStatus.Enabled = true;
            this.btn6DbsArSetSettings.Enabled = true;
            this.btn6dbsArToggleChartRefreshing.Enabled = true;
            this.btn6dbsArReadStatus.Enabled = true;
            this.btn6DbsArReadSettings.Enabled = true;
            this.btn6DbsArResetSettings.Enabled = true;

            this.GetStatusOfI2CLed(false);
            this.ComboBoxSaa1064NumbersPatternChoice_SelectedIndexChanged(null, null);
        }

        /// <summary>
        /// Connects to a ComPort.
        /// </summary>
        private void Connect()
        {
            try
            {
                this.portName = this.PortsComboBox.Text;
                this.serialPort = new SerialPort(
                    this.portName,
                    BAUDRATE,
                    PARITY,
                    DATABITS,
                    STOPBITS);
                this.serialPort.Open();
                Stream stream = this.serialPort.BaseStream;
                stream.ReadTimeout = READTIMEOUT;
                this.usbI2C = new Eq3.misc.USBI2C.UsbI2C(stream);

                string command = "<";
                this.listBoxOutput.Items.Add(command);
                this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                if (!this.usbI2C.SendCommand(command))
                {
                    throw new Exception();
                }

                string status = this.usbI2C.GetStatus();
                this.listBoxOutput.Items.Add("?");
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                if (status == string.Empty)
                {
                    throw new Exception();
                }

                this.ConnectionLabel.Text = "verbunden mit " + this.portName;
                this.ConnectButton.Text = "Trennen";

                this.isConnected = true;

                this.Enable();
                this.InitDevices();
            }
            catch (Exception)
            {
                this.isConnected = false;

                MessageBox.Show("Es konnte keine Verbindung aufgebaut werden. Bitte prüfen Sie ob das Gerät \r\nangeschlossen ist und nicht bereits von einer anderen Software benutzt wird.", "Verbinden");
                if (this.serialPort.IsOpen)
                {
                    this.serialPort.Close();
                }
            }
        }

        /// <summary>
        /// Initializes wrapper for different I2C-devices.
        /// </summary>
        private void InitDevices()
        {
            this.Lcd = new LCD(this.usbI2C);
            this.I2C4DLed = new I2C4DLed(this.usbI2C);
            
            I2C6DBS.UiElements uiElements = new I2C6DBS.UiElements();
            uiElements.ListBoxInput = this.listBoxInput;
            uiElements.ListBoxOutput = this.listBoxOutput;
            this.I2C6Dbs = new I2C6DBS(this.usbI2C);
            this.I2C6Dbs.LinearAcceleration.ImportUiElements(uiElements);
            this.I2C6Dbs.AngularRate.ImportUiElements(uiElements);
        }

        /// <summary>
        /// Closes ComPort-connection.
        /// </summary>
        private void Disconnect()
        {
            if (!this.serialPort.IsOpen)
            {
                return;
            }

            if (this.timerTriggerSaa1064 != null)
            {
                this.BtSaaTrigger_Click(null, null);
            }

            if (this.timerTriggerMcp9801 != null)
            {
                this.BtMCPTrigger_Click(null, null);
            }

            if (this.timerShowTimeOnSaa1064 != null)
            {
                this.DeactivateTimerShowTimeOnSaa1064();
            }

            if (this.timerTriggerChart6dbsLa != null && this.timerTriggerChart6dbsLa.Enabled)
            {
                this.Btn6dbsLaToggleChartRefreshing_Click(null, null);
            }

            if (this.timerTriggerChart6dbsAr != null && this.timerTriggerChart6dbsAr.Enabled)
            {
                this.Btn6dbsArToggleChartRefreshing_Click(null, null);
            }

            try
            {
                this.serialPort.Close();
            }
            catch (IOException)
            {
            }

            this.portName = string.Empty;
            this.ConnectionLabel.Text = "- nicht verbunden -";
            this.ConnectButton.Text = "Verbinden";
            this.Disable();
            this.isConnected = false;
        }

        /// <summary>
        /// Toggles ComPort-connection.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ConnectButton_Click(object sender, EventArgs e)
        {
            if (!this.isConnected)
            {
                this.Connect();
            }
            else
            {
                this.Disconnect();
            }
        }

        /// <summary>
        /// Clears the list of received data (Click-function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonCLearRX_Click(object sender, EventArgs e)
        {
            this.ClearReceived();
        }
        
        /// <summary>
        /// Clears the list of received data.
        /// </summary>
        private void ClearReceived()
        {
            this.listBoxInput.Items.Clear();
        }

        /// <summary>
        /// Clears the list of transmitted data (Click-function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonClearTX_Click(object sender, EventArgs e)
        {
            this.ClearTransmitted();
        }

        /// <summary>
        /// Clears the list of transmitted data.
        /// </summary>
        private void ClearTransmitted()
        {
            this.listBoxOutput.Items.Clear();
        }

        /// <summary>
        /// Opens the info dialog.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonInfo_Click(object sender, EventArgs e)
        {
            FormAbout infoForm = new FormAbout();
            infoForm.ShowDialog();
        }

        /// <summary>
        /// Closes a possible connection.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (this.isConnected)
            {
                this.Disconnect();
            }
        }

        /// <summary>
        /// Sets settings while form loading.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void MainForm_Load(object sender, EventArgs e)
        {
            // Setzen von Tooltips
            ToolTip hint = new ToolTip();
            hint.ShowAlways = true;
            hint.AutoPopDelay = 10000;
            hint.ReshowDelay = 0;
            hint.InitialDelay = 0;
            hint.SetToolTip(this.comboBoxiSMTsmpShaft, "CW -> Uhrzeigersinn, CCW -> gegen den Uhrzeigersinn");
            ToolTip hint2 = new ToolTip();
            hint2.ShowAlways = true;
            hint2.AutoPopDelay = 10000;
            hint2.ReshowDelay = 0;
            hint2.InitialDelay = 0;
            hint2.SetToolTip(this.listBoxiSMTCommandsequenz, "Strg + C -> Kopieren in die Zwischenablage, Strg + V -> Einfügen aus der Zwischenablage, Entf -> Löschen der markierten Einträge");
            
            // ComboBoxen füllen iSMT
            this.comboBoxiSMTsmpIRun.Items.Clear();
            this.comboBoxiSMTsmpIHold.Items.Clear();
            this.comboBoxiSMTsmpVMin.Items.Clear();
            this.comboBoxiSMTsmpVMax.Items.Clear();
            this.comboBoxiSMTsmpAccShape.Items.Clear();
            this.comboBoxiSMTsmpAcc.Items.Clear();
            this.comboBoxiSMTsmpShaft.Items.Clear();
            this.comboBoxiSMTsmpStepMode.Items.Clear();

            this.comboBoxiSMTsspAbsThr.Items.Clear();
            this.comboBoxiSMTsspDelThr.Items.Clear();
            this.comboBoxiSMTsspMinSamples.Items.Clear();
            this.comboBoxiSMTsspStallEn.Items.Clear();

            this.comboBoxiSMTrsVMin.Items.Clear();
            this.comboBoxiSMTrsVMax.Items.Clear();

            for (int i = 0; i < 16; i++)
            {
                if (i < 15)
                {
                    this.comboBoxiSMTsmpIRun.Items.Add(i.ToString("x").ToUpper() + ", " + this.ItoString(i));
                    this.comboBoxiSMTsmpIHold.Items.Add(i.ToString("x").ToUpper() + ", " + this.ItoString(i));
                }
                else
                {
                    this.comboBoxiSMTsmpIRun.Items.Add(i.ToString("x").ToUpper() + ", 800 mA");
                    this.comboBoxiSMTsmpIHold.Items.Add(i.ToString("x").ToUpper() + ", 0mA");
                }

                if (i == 0)
                {
                    this.comboBoxiSMTsmpVMin.Items.Add(i.ToString("x").ToUpper() + ", =VMax");
                    this.comboBoxiSMTsmpVMax.Items.Add(i.ToString("x").ToUpper() + ", slow");
                    this.comboBoxiSMTrsVMin.Items.Add(i.ToString("x").ToUpper() + ", =VMax");
                    this.comboBoxiSMTrsVMax.Items.Add(i.ToString("x").ToUpper() + ", slow");
                }
                else if (i == 15)
                {
                    this.comboBoxiSMTsmpVMin.Items.Add(i.ToString("x").ToUpper() + ", fast");
                    this.comboBoxiSMTsmpVMax.Items.Add(i.ToString("x").ToUpper() + ", fast");
                    this.comboBoxiSMTrsVMin.Items.Add(i.ToString("x").ToUpper() + ", fast");
                    this.comboBoxiSMTrsVMax.Items.Add(i.ToString("x").ToUpper() + ", fast");
                }
                else if (i == 1)
                {
                    this.comboBoxiSMTsmpVMin.Items.Add(i.ToString("x").ToUpper() + ", slow");
                    this.comboBoxiSMTsmpVMax.Items.Add(i.ToString("x").ToUpper());
                    this.comboBoxiSMTrsVMin.Items.Add(i.ToString("x").ToUpper() + ", slow");
                    this.comboBoxiSMTrsVMax.Items.Add(i.ToString("x").ToUpper());
                }
                else
                {
                    this.comboBoxiSMTsmpVMin.Items.Add(i.ToString("x").ToUpper());
                    this.comboBoxiSMTsmpVMax.Items.Add(i.ToString("x").ToUpper());
                    this.comboBoxiSMTrsVMin.Items.Add(i.ToString("x").ToUpper());
                    this.comboBoxiSMTrsVMax.Items.Add(i.ToString("x").ToUpper());
                }

                if (i == 0 || i == 1 || i > 8)
                {
                    this.comboBoxiSMTsmpAcc.Items.Add(i.ToString("x").ToUpper() + ", (*)");
                }
                else
                {
                    this.comboBoxiSMTsmpAcc.Items.Add(i.ToString("x").ToUpper());
                }

                this.comboBoxiSMTsspAbsThr.Items.Add(i.ToString("x").ToUpper());
                this.comboBoxiSMTsspDelThr.Items.Add(i.ToString("x").ToUpper());
            }

            for (int i = 0; i < 8; i++)
            {
                this.comboBoxiSMTsspMinSamples.Items.Add(i.ToString("x").ToUpper());
                this.comboBoxiSMTsspStallEn.Items.Add(i.ToString("x").ToUpper());
            }

            this.comboBoxiSMTsmpAccShape.Items.Add(0);
            this.comboBoxiSMTsmpAccShape.Items.Add(1);
            this.comboBoxiSMTsmpShaft.Items.Add("CW");
            this.comboBoxiSMTsmpShaft.Items.Add("CCW");
            this.comboBoxiSMTsmpStepMode.Items.Add("1/2");
            this.comboBoxiSMTsmpStepMode.Items.Add("1/4");
            this.comboBoxiSMTsmpStepMode.Items.Add("1/8");
            this.comboBoxiSMTsmpStepMode.Items.Add("1/16");

            // ComboBoxen füllen Allgemeiner Teil
            this.comboBoxIICAdress.Items.Clear();
            for (int i = 192; i < 255; i += 2)
            {
                this.comboBoxIICAdress.Items.Add(i.ToString() + ", 0x" + i.ToString("x2").ToUpper());
            }

            this.comboBoxIICAdress.Items.Add("0, 0x00");
            
            // Auswahl der Standardwerte der ComboBoxen
            this.comboBoxiSMTsmpIRun.SelectedIndex = 0;
            this.comboBoxiSMTsmpIHold.SelectedIndex = 0;
            this.comboBoxiSMTsmpVMin.SelectedIndex = 0;
            this.comboBoxiSMTsmpVMax.SelectedIndex = 0;
            this.comboBoxiSMTsmpAccShape.SelectedIndex = 0;
            this.comboBoxiSMTsmpAcc.SelectedIndex = 0;
            this.comboBoxiSMTsmpShaft.SelectedIndex = 0;
            this.comboBoxiSMTsmpStepMode.SelectedIndex = 0;

            this.comboBoxiSMTsspAbsThr.SelectedIndex = 0;
            this.comboBoxiSMTsspDelThr.SelectedIndex = 0;
            this.comboBoxiSMTsspMinSamples.SelectedIndex = 0;
            this.comboBoxiSMTsspStallEn.SelectedIndex = 0;

            this.comboBoxiSMTrsVMin.SelectedIndex = 0;
            this.comboBoxiSMTrsVMax.SelectedIndex = 0;

            this.comboBoxIICAdress.SelectedIndex = 0;

            this.comboBox3DBSbandwidth.SelectedIndex = 0;
            this.comboBox3DBSrange.SelectedIndex = 0;

            this.comboBoxLEDout1.SelectedIndex = 0;
            this.comboBoxLEDout2.SelectedIndex = 0;
            this.comboBoxLEDout3.SelectedIndex = 0;
            this.comboBoxLEDout4.SelectedIndex = 0;
            this.comboBoxLEDout5.SelectedIndex = 0;
            this.comboBoxLEDout6.SelectedIndex = 0;
            this.comboBoxLEDout7.SelectedIndex = 0;
            this.comboBoxLEDout8.SelectedIndex = 0;
            this.comboBoxLEDout9.SelectedIndex = 0;
            this.comboBoxLEDout10.SelectedIndex = 0;
            this.comboBoxLEDout11.SelectedIndex = 0;
            this.comboBoxLEDout12.SelectedIndex = 0;
            this.comboBoxLEDout13.SelectedIndex = 0;
            this.comboBoxLEDout14.SelectedIndex = 0;
            this.comboBoxLEDout15.SelectedIndex = 0;
            this.comboBoxLEDout16.SelectedIndex = 0;

            this.comboBoxLEDcm.SelectedIndex = 1;
            this.comboBoxLEDhc.SelectedIndex = 1;

            this.rdMCPCelsius.Checked = true;
            this.rdSaa1064TriggerIntervall.Checked = true;
            this.rdMCP9801TriggerIntervall.Checked = true;

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
            this.comboBoxSaa1064CurrentChoice.SelectedIndex = 4;
            this.comboBoxSaa1064SlaveAddress.SelectedIndex = 0;
            this.comboBoxMCP9801SlaveAddress.SelectedIndex = 0;
            this.comboBoxMCP9801FaultQueue.SelectedIndex = 0;
            this.comboBoxMCP9801ADCResolution.SelectedIndex = 3;

            this.comboBox6DbsLaSelectDataRate.SelectedIndex = 9;
            this.comboBox6DbsLaSelectFilterMode.SelectedIndex = 0;
            this.comboBox6DbsLaSelectEndianMode.SelectedIndex = 0;
            this.comboBox6DbsLaFullScale.SelectedIndex = 0;
            this.comboBox6DbsLaSelectSpiMode.SelectedIndex = 1;
            this.comboBox6DbsArSelectOdr.SelectedIndex = 0;
            this.comboBox6DbsArSelectFilterMode.SelectedIndex = 0;
            this.comboBox6DbsArSelectEndian.SelectedIndex = 0;
            this.comboBox6DbsArSelectFullScale.SelectedIndex = 0;
            this.comboBox6DbsArSelectBlockDataUpdate.SelectedIndex = 0;
            this.comboBox6DbsArSelectSpiMode.SelectedIndex = 0;
            this.comboBox6DbsLaSelectSlaveAdress.SelectedIndex = 0;
            this.comboBox6DbsArSelectSlaveAdress.SelectedIndex = 0;
            this.comboBox6DbsArSelectDataFilter.SelectedIndex = 0;
            this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex = 0;

            this.InitCharts();
        }
        #endregion

        #region USB-I2C
        /// <summary>
        /// Gets firmware and status string from device.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ButtonStatusFirmware_Click(object sender, EventArgs e)
        {
            string status = this.usbI2C.GetStatus();
            this.listBoxOutput.Items.Add("?");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (status != string.Empty)
            {
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Gets saved macro from device.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonShowMakro_Click(object sender, EventArgs e)
        {
            string result = this.usbI2C.GetMakro();
            this.listBoxOutput.Items.Add("u");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (result != string.Empty)
            {
                this.listBoxInput.Items.AddRange(result.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Removes saved macro.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ButtonDelMakro_Click(object sender, EventArgs e)
        {
            this.StopMacro();
            System.Threading.Thread.Sleep(500);

            string result = this.usbI2C.DeleteMakro();
            this.listBoxOutput.Items.Add("u v00{} u");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (result != string.Empty)
            {
                this.listBoxInput.Items.AddRange(result.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Starts saved macro.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonStartMakro_Click(object sender, EventArgs e)
        {
            string command = ">00";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Stopps a executed macro.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ButtonStopMakro_Click(object sender, EventArgs e)
        {
            this.StopMacro();
        }

        /// <summary>
        /// Stopps a executed macro.
        /// </summary>
        private void StopMacro()
        {
            const string Command = "<";

            this.listBoxOutput.Items.Add(Command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(Command))
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Resets the device.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonDeliveryreset_Click(object sender, EventArgs e)
        {
            string status = this.usbI2C.FullReset();
            this.listBoxOutput.Items.Add("zAA");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (status != string.Empty)
            {
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Resets y-parameter of the device.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ButtonYReset_Click(object sender, EventArgs e)
        {
            string status = this.usbI2C.ResetY();
            this.listBoxOutput.Items.Add("y00 y10 y20 y30 y40 y50 y60 y70 ?");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (status != string.Empty)
            {
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Does a soft-reset.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonReset_Click(object sender, EventArgs e)
        {
            string status = this.usbI2C.Reset();
            this.listBoxOutput.Items.Add("z4b");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (status != string.Empty)
            {
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Transmits a command directly to the device.
        /// </summary>
        /// <param name="sender">Standard parameter object sender.</param>
        /// <param name="e">Standard parameter EventArgs e.</param>
        private void ButtonI2Cdirect_Click(object sender, EventArgs e)
        {
            this.I2CDirectCommand();
        }

        /// <summary>
        /// Sends the command by entering enter or return.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxI2Cdirect_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return)
            {
                this.I2CDirectCommand();
            }
        }

        /// <summary>
        /// Transmits a I2C-command.
        /// </summary>
        private void I2CDirectCommand()
        {
            string command = this.textBoxI2Cdirect.Text;
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string status = this.usbI2C.SendTryReceiveCommand(command);
            if (status == null)
            {
                this.ShowError(0);
            }
            else
            {
                this.listBoxInput.Items.AddRange(status.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;
            }
        }
        #endregion

        #region iSMT
        /// <summary>
        /// Gets full status 1.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonGetFullStatus1_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 81 s" + ((int)address + 1).ToString("X2") + " 08 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(1);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(1);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.textBoxiSMTgfs1Adress.Text = resultBytes[0].ToString("X2");

                    if (((byte)resultBytes[1] >> 4) < 0x0F)
                    {
                        this.textBoxiSMTgfs1IRun.Text = ((byte)resultBytes[1] >> 4).ToString("X") + ", " + this.ItoString((int)((byte)resultBytes[1] >> 4));
                    }
                    else
                    {
                        this.textBoxiSMTgfs1IRun.Text = ((byte)resultBytes[1] >> 4).ToString("X") + ", 800mA";
                    }

                    if (((byte)resultBytes[1] & 0xf) < 0x0F)
                    {
                        this.textBoxiSMTgfs1IHold.Text = ((byte)resultBytes[1] & 0xf).ToString("X") + ", " + this.ItoString((int)((byte)resultBytes[1] & 0xf));
                    }
                    else
                    {
                        this.textBoxiSMTgfs1IHold.Text = ((byte)resultBytes[1] & 0xf).ToString("X") + ", 0mA";
                    }

                    if (((byte)resultBytes[2] >> 4) == 0)
                    {
                        this.textBoxiSMTgfs1VMax.Text = ((byte)resultBytes[2] >> 4).ToString("X") + ", slow";
                    }
                    else if (((byte)resultBytes[2] >> 4) == 15)
                    {
                        this.textBoxiSMTgfs1VMax.Text = ((byte)resultBytes[2] >> 4).ToString("X") + ", fast";
                    }
                    else
                    {
                        this.textBoxiSMTgfs1VMax.Text = ((byte)resultBytes[2] >> 4).ToString("X");
                    }

                    if (((byte)resultBytes[2] >> 4) == 0)
                    {
                        this.textBoxiSMTgfs1VMin.Text = ((byte)resultBytes[2] & 0xf).ToString("X") + ", =VMax";
                    }
                    else if (((byte)resultBytes[2] >> 4) == 1)
                    {
                        this.textBoxiSMTgfs1VMin.Text = ((byte)resultBytes[2] & 0xf).ToString("X") + ", slow";
                    }
                    else if (((byte)resultBytes[2] >> 4) == 15)
                    {
                        this.textBoxiSMTgfs1VMin.Text = ((byte)resultBytes[2] & 0xf).ToString("X") + ", fast";
                    }
                    else
                    {
                        this.textBoxiSMTgfs1VMin.Text = ((byte)resultBytes[2] & 0xf).ToString("X");
                    }

                    this.textBoxiSMTgfs1AccShape.Text = ((byte)resultBytes[3] >> 7).ToString("X");
                    this.textBoxiSMTgfs1StepMode.Text = ((byte)(resultBytes[3] >> 5) & 0x3).ToString("X");
                    this.textBoxiSMTgfs1Shaft.Text = ((byte)(resultBytes[3] >> 4) & 0x1).ToString("X");
                    this.textBoxiSMTgfs1Acceleration.Text = ((byte)resultBytes[3] & 0xf).ToString("X");
                    this.checkBoxiSMTgfs1VddReset.Checked = ((byte)resultBytes[4] & 0x80) != 0;
                    this.checkBoxiSMTgfs1StepLoss.Checked = ((byte)resultBytes[4] & 0x40) != 0;
                    this.checkBoxiSMTgfs1ElDef.Checked = ((byte)resultBytes[4] & 0x20) != 0;
                    this.checkBoxiSMTgfs1UV2.Checked = ((byte)resultBytes[4] & 0x10) != 0;
                    this.checkBoxiSMTgfs1TSD.Checked = ((byte)resultBytes[4] & 0x08) != 0;
                    this.checkBoxiSMTgfs1TW.Checked = ((byte)resultBytes[4] & 0x04) != 0;
                    this.textBoxiSMTgfs1TInfo.Text = ((byte)resultBytes[4] & 0x3).ToString("X");
                    this.textBoxiSMTgfs1Motion.Text = ((byte)(resultBytes[5] >> 5) & 0x7).ToString("X");
                    this.checkBoxiSMTgfs1ESW.Checked = ((byte)resultBytes[5] & 0x10) != 0;
                    this.checkBoxiSMTgfs1OVC1.Checked = ((byte)resultBytes[5] & 0x08) != 0;
                    this.checkBoxiSMTgfs1OVC2.Checked = ((byte)resultBytes[5] & 0x04) != 0;
                    this.checkBoxiSMTgfs1CPFail.Checked = ((byte)resultBytes[5] & 0x01) != 0;

                    this.textBoxiSMTgfs1AbsThr.Text = ((byte)resultBytes[7] >> 4).ToString("X");
                    this.textBoxiSMTgfs1DelThr.Text = ((byte)resultBytes[7] & 0xf).ToString("X");
                }
                catch
                {
                    this.ShowError(1);
                }
            }
        }

        /// <summary>
        /// Gets full status 2.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonGetFullStatus2_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " FC s" + ((int)address + 1).ToString("X2") + " 08 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(1);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(1);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.textBoxiSMTgfs2Adress.Text = resultBytes[0].ToString("X2");
                    this.textBoxiSMTgfs2APX.Text = "0x" + resultBytes[1].ToString("X2") + resultBytes[2].ToString("X2");
                    this.textBoxiSMTgfs2TPX.Text = "0x" + resultBytes[3].ToString("X2") + resultBytes[4].ToString("X2");
                    this.textBoxiSMTgfs2SPX.Text = "0x" + ((byte)resultBytes[6] & 0x7).ToString("X") + resultBytes[5].ToString("X2");
                    this.textBoxiSMTgfs2APD.Text = (((int)resultBytes[1] * 256) + resultBytes[2]).ToString();
                    this.textBoxiSMTgfs2TPD.Text = (((int)resultBytes[3] * 256) + resultBytes[4]).ToString();
                    this.textBoxiSMTgfs2SPD.Text = (((int)((byte)resultBytes[6] & 0x7) * 256) + resultBytes[5]).ToString();

                    this.checkBoxiSMTgfs2DC100.Checked = ((byte)resultBytes[6] & 0x08) != 0;
                    this.textBoxiSMTgfs2FS2StallEn.Text = ((byte)resultBytes[6] >> 5).ToString("X");
                    this.checkBoxiSMTgfs2AbsStall.Checked = ((byte)resultBytes[7] & 0x80) != 0;
                    this.checkBoxiSMTgfs2DelStall.Checked = ((byte)resultBytes[7] & 0x40) != 0;
                    this.checkBoxiSMTgfs2DelStallHi.Checked = ((byte)resultBytes[7] & 0x20) != 0;
                    this.textBoxiSMTgfs2MinSample.Text = ((byte)(resultBytes[7] >> 2) & 0x7).ToString("X");
                    this.checkBoxiSMTgfs2EnDC.Checked = ((byte)resultBytes[7] & 0x02) != 0;
                    this.checkBoxiSMTgfs2PWMJit.Checked = ((byte)resultBytes[7] & 0x01) != 0;
                }
                catch
                {
                    this.ShowError(1);
                }
            }
        }

        /// <summary>
        /// Gets otp parameter.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonGetOTPParameter_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 82 s" + ((int)address + 1).ToString("X2") + " 08 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(1);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(1);
                    return;
                }

                try
                {
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.textBoxiSMTgOTPP0.Text = resultBytes[0].ToString("X2");
                    this.textBoxiSMTgOTPP1.Text = resultBytes[1].ToString("X2");
                    this.textBoxiSMTgOTPP2.Text = resultBytes[2].ToString("X2");
                    this.textBoxiSMTgOTPP3.Text = resultBytes[3].ToString("X2");
                    this.textBoxiSMTgOTPP4.Text = resultBytes[4].ToString("X2");
                    this.textBoxiSMTgOTPP5.Text = resultBytes[5].ToString("X2");
                    this.textBoxiSMTgOTPP6.Text = resultBytes[6].ToString("X2");
                    this.textBoxiSMTgOTPP7.Text = resultBytes[7].ToString("X2");
                }
                catch
                {
                    this.ShowError(1);
                }
            }
        }

        /// <summary>
        /// Sets motor parameter.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSetMotorParam_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            int temp = 0;
            string command = "s" + address.ToString("X2");
            command += " 89 FF FF";
            command += " " + this.comboBoxiSMTsmpIRun.SelectedIndex.ToString("x") + this.comboBoxiSMTsmpIHold.SelectedIndex.ToString("x");
            command += " " + this.comboBoxiSMTsmpVMax.SelectedIndex.ToString("x") + this.comboBoxiSMTsmpVMin.SelectedIndex.ToString("x");
            temp = (int)this.numericUpDowniSMTsmpsp.Value;
            temp = (temp - (temp % 256)) / 128;
            temp += this.comboBoxiSMTsmpShaft.SelectedIndex;
            command += " " + temp.ToString("x") + this.comboBoxiSMTsmpAcc.SelectedIndex.ToString("x");
            command += " " + ((int)(this.numericUpDowniSMTsmpsp.Value % 256)).ToString("x2");
            temp = 128 + 32 + 2;
            temp += (this.comboBoxiSMTsmpAccShape.SelectedIndex * 16) + (this.comboBoxiSMTsmpStepMode.SelectedIndex * 4);
            if (this.checkBoxiSMTsmpPWMFreq.Checked)
            {
                temp += 64;
            }

            if (this.checkBoxiSMTsmpPWMJEn.Checked)
            {
                temp += 1;
            }

            command += " " + temp.ToString("x2");
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;            
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(2);
            }
        }

        /// <summary>
        /// Sets stall parameter.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSetStallParam_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            int temp = 0;
            string command = "s" + address.ToString("X2");
            command += " 96 FF FF";
            command += " " + this.comboBoxiSMTsmpIRun.SelectedIndex.ToString("x") + this.comboBoxiSMTsmpIHold.SelectedIndex.ToString("x");
            command += " " + this.comboBoxiSMTsmpVMax.SelectedIndex.ToString("x") + this.comboBoxiSMTsmpVMin.SelectedIndex.ToString("x");
            temp = this.comboBoxiSMTsspMinSamples.SelectedIndex * 2;
            temp += this.comboBoxiSMTsmpShaft.SelectedIndex;
            command += " " + temp.ToString("x") + this.comboBoxiSMTsmpAcc.SelectedIndex.ToString("x");
            command += " " + this.comboBoxiSMTsspAbsThr.SelectedIndex.ToString("x") + this.comboBoxiSMTsspDelThr.SelectedIndex.ToString("x");
            temp = (this.comboBoxiSMTsspStallEn.SelectedIndex * 32) + (this.comboBoxiSMTsmpAccShape.SelectedIndex * 16);
            temp += this.comboBoxiSMTsmpStepMode.SelectedIndex * 4;
            if (this.checkBoxiSMTsspEnDC.Checked)
            {
                temp += 2;
            }

            if (this.checkBoxiSMTsspPWMJEn.Checked)
            {
                temp += 1;
            }

            command += " " + temp.ToString("x2");
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(3);
            }
        }

        /// <summary>
        /// Resets iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonResetToDefault_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 87 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Sets position of iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSetPosition_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 8B FF FF";
            command += " " + ((short)this.numericUpDowniSMTtp.Value).ToString("x4");
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Resets current position.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonResetPosition_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 86 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Send a soft stop to iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSoftStop_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 8F p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Send a hard stop to iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonHardStop_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 85 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Sets current position as secure position.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonGotoSP_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 84 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Execute RunInit-command for iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonRunInit_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 88 FF FF";
            command += " " + this.comboBoxiSMTrsVMax.SelectedIndex.ToString("x") + this.comboBoxiSMTrsVMin.SelectedIndex.ToString("x");
            command += " " + ((short)this.numericUpDowniSMTrsTP1.Value).ToString("x4");
            command += " " + ((short)this.numericUpDowniSMTrsTP2.Value).ToString("x4");
            
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Execute RunVelocity-command for iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonRunVelocity_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 97 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Execute TestBEMF-command for iSMT.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonTestBEMF_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 9F p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(4);
            }
        }

        /// <summary>
        /// Sets a otp parameter.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSetOTP_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 90 FF FF";
            command += " F" + ((int)(8 + this.numericUpDowniSMTotpa.Value)).ToString("x");
            command += " " + ((int)this.numericUpDowniSMTpBit.Value).ToString("x2");
            command += " p";

            if (MessageBox.Show("Wollen Sie wirklich die OTP Adresse " + this.numericUpDowniSMTotpa.Value.ToString() + " auf den Wert " + this.numericUpDowniSMTpBit.Value.ToString() + " setzen?", "OTP setzen?", MessageBoxButtons.YesNo) == DialogResult.Yes)
            {
                this.listBoxOutput.Items.Add(command);
                this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                if (!this.usbI2C.SendCommand(command))
                {
                    this.ShowError(5);
                }
            }
        }

        /// <summary>
        /// Converts i-values to currents.
        /// </summary>
        /// <param name="i">Source as int.</param>
        /// <returns>Current as string.</returns>
        private string ItoString(int i)
        {
            switch (i)
            {
                case 0:
                    return "59 mA";
                case 1:
                    return "71 mA";
                case 2:
                    return "84 mA";
                case 3:
                    return "100 mA";
                case 4:
                    return "119 mA";
                case 5:
                    return "141 mA";
                case 6:
                    return "168 mA";
                case 7:
                    return "200 mA";
                case 8:
                    return "238 mA";
                case 9:
                    return "283 mA";
                case 10:
                    return "336 mA";
                case 11:
                    return "400 mA";
                case 12:
                    return "476 mA";
                case 13:
                    return "566 mA";
                case 14:
                    return "673 mA";
                default:
                    return string.Empty;
            }
        }

        /// <summary>
        /// Traverse list of commands in program tab.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TimerSequenz_Tick(object sender, EventArgs e)
        {
            if (this.listBoxiSMTCommandsequenz.Items.Count > this.listPosition)
            {
                if (this.waitcounter > 0)
                {
                    this.waitcounter--;
                }
                else
                {
                    while (this.waitcounter == 0)
                    {
                        string command = this.listBoxiSMTCommandsequenz.Items[this.listPosition].ToString();
                        command = command.Trim();
                        
                        this.listBoxiSMTCommandsequenz.SelectedIndex = this.listPosition;

                        if (command.StartsWith("wait"))
                        {
                            this.waitcounter = int.Parse(command.Split(' ')[1]);
                        }
                        else
                        {
                            this.listBoxOutput.Items.Add(command);
                            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                            if (!this.usbI2C.SendCommand(command))
                            {
                                this.ShowError(6);
                            }
                        }

                        this.listPosition++;

                        if (this.listBoxiSMTCommandsequenz.Items.Count <= this.listPosition)
                        {
                            this.timerSequenz.Enabled = false;
                            this.buttonStop.Enabled = false;
                            this.listPosition = 0;
                            break;
                        }
                    }
                }
            }
            else
            {
                this.timerSequenz.Enabled = false;
                this.buttonStop.Enabled = false;
                this.listPosition = 0;
            }
        }

        /// <summary>
        /// Saves command list as macro in I2C.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtoniSMTListToMakro_Click(object sender, EventArgs e)
        {
            this.listPosition = 0;
            string result = this.usbI2C.DeleteMakro();
            this.listBoxOutput.Items.Add("u v00{} u");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (result != string.Empty)
            {
                this.listBoxInput.Items.AddRange(result.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                string command = "V 00{";
                string temp = string.Empty;
                int wait;

                while (command.Length < (255 + 5))
                {
                    temp = this.listBoxiSMTCommandsequenz.Items[this.listPosition].ToString().Trim();
                    
                    if (temp.StartsWith("wait"))
                    {
                        wait = int.Parse(temp.Split(' ')[1]);
                        wait = wait * 100;
                        temp = "L" + wait.ToString("X4");
                    }

                    if ((command.Length + temp.Length + 1) < (255 + 5))
                    {
                        command += temp + " ";
                    }
                    else
                    {
                        this.listPosition = 0;
                        break;
                    }

                    this.listPosition++;

                    if (this.listBoxiSMTCommandsequenz.Items.Count <= this.listPosition)
                    {
                        this.listPosition = 0;
                        break;
                    }
                }

                command += "}";
                this.listBoxOutput.Items.Add(command);
                this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                if (!this.usbI2C.SendCommand(command))
                {
                    this.ShowError(7);
                }
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Starts eradication of list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonStart_Click(object sender, EventArgs e)
        {
            this.timerSequenz.Interval = 100;
            this.timerSequenz.Enabled = true;
            this.buttonStop.Enabled = true;
        }

        /// <summary>
        /// Stopps eradication of list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonStop_Click(object sender, EventArgs e)
        {
            this.timerSequenz.Enabled = false;
            this.buttonStop.Enabled = false;
            this.listPosition = 0;
        }

        /// <summary>
        /// Add a pause to command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonPause_Click(object sender, EventArgs e)
        {
            string command = "wait " + this.numericUpDowniSMTInterval.Value.ToString() + " * 100 ms";
            this.listBoxiSMTCommandsequenz.Items.Add(command);
        }  

        /// <summary>
        /// Adds a SetPosition-command to the list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSetPositionList_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 8B FF FF";
            command += " " + ((short)this.numericUpDowniSMTprogramTP.Value).ToString("x4");
            command += " p";

            this.listBoxiSMTCommandsequenz.Items.Add(command);            
        }

        /// <summary>
        /// Adds a RunVelocity-command to the list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonRunVelocityList_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 97 p";

            this.listBoxiSMTCommandsequenz.Items.Add(command); 
        }

        /// <summary>
        /// Adds a SoftStop-command to the list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonSoftStopList_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 8F p";

            this.listBoxiSMTCommandsequenz.Items.Add(command); 
        }

        /// <summary>
        /// Clears the list of executing commands.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonClearList_Click(object sender, EventArgs e)
        {
            this.listBoxiSMTCommandsequenz.Items.Clear();
        }        
        #endregion

        #region 3D-BS
        /// <summary>
        /// Reads settings register of 3D-BS.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Button3DBSStatus_Click(object sender, EventArgs e)
        {
            string command = "s70";
            command += " 00 s71 16 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(10);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // get status
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.textBox3DBSChipID.Text = ((byte)resultBytes[0] & 0x07).ToString("X").ToUpper();
                    this.textBox3DBSalVersion.Text = ((byte)resultBytes[1] >> 4).ToString("X").ToUpper();
                    this.textBox3DBSmlVersion.Text = ((byte)resultBytes[1] & 0x0F).ToString("X").ToUpper();

                    byte[] values = new byte[6];
                    for (int i = 0; i < 6; i++)
                    {
                        values[i] = resultBytes[i + 2];
                    }

                    this.Set3DBSValues(values);

                    this.checkBox3DBSstResult.Checked = ((byte)resultBytes[9] & 0x80) != 0;
                    this.checkBox3DBSalertPhase.Checked = ((byte)resultBytes[9] & 0x10) != 0;
                    this.checkBox3DBSlgLatched.Checked = ((byte)resultBytes[9] & 0x08) != 0;
                    this.checkBox3DBShgLatched.Checked = ((byte)resultBytes[9] & 0x04) != 0;
                    this.checkBox3DBSstatusLG.Checked = ((byte)resultBytes[9] & 0x02) != 0;
                    this.checkBox3DBSstatusHG.Checked = ((byte)resultBytes[9] & 0x01) != 0;
                    this.checkBox3DBSresetINT.Checked = ((byte)resultBytes[10] & 0x40) != 0;
                    this.checkBox3DBSselfTest1.Checked = ((byte)resultBytes[10] & 0x08) != 0;
                    this.checkBox3DBSselfTest0.Checked = ((byte)resultBytes[10] & 0x04) != 0;
                    this.checkBox3DBSsoftReset.Checked = ((byte)resultBytes[10] & 0x02) != 0;
                    this.checkBox3DBSsleep.Checked = ((byte)resultBytes[10] & 0x01) != 0;
                    
                    this.checkBox3DBSalert.Checked = ((byte)resultBytes[11] & 0x80) != 0;
                    this.checkBox3DBSanyMotion.Checked = ((byte)resultBytes[11] & 0x40) != 0;
                    this.textBox3DBShgCounter.Text = this.ValueTranslator3DBS(((byte)(resultBytes[11] >> 4 & 0x03)), 114);
                    this.textBox3DBSlgCounter.Text = this.ValueTranslator3DBS(((byte)(resultBytes[11] >> 4 & 0x03)), 112);
                    this.checkBox3DBSenableHG.Checked = ((byte)resultBytes[11] & 0x02) != 0;
                    this.checkBox3DBSenableLG.Checked = ((byte)resultBytes[11] & 0x01) != 0;
                    this.textBox3DBSlgThres.Text = resultBytes[12].ToString("X").ToUpper();
                    this.textBox3DBSlgDur.Text = resultBytes[13].ToString("X").ToUpper();
                    this.textBox3DBShgThres.Text = resultBytes[14].ToString("X").ToUpper();
                    this.textBox3DBShgDur.Text = resultBytes[15].ToString("X").ToUpper();
                    this.textBox3DBSanyMotionThres.Text = resultBytes[16].ToString("X").ToUpper();
                    this.textBox3DBSanyMotionDur.Text = this.ValueTranslator3DBS(((byte)(resultBytes[17] >> 6 & 0x02)), 176);
                    this.textBox3DBShgHyst.Text = ((byte)resultBytes[17] >> 3 & 0x07).ToString("X").ToUpper();
                    this.textBox3DBSlgHyst.Text = ((byte)resultBytes[17] & 0x07).ToString("X").ToUpper();
                    this.textBox3DBSrange.Text = this.ValueTranslator3DBS(((byte)(resultBytes[20] >> 3 & 0x03)), 203);
                    this.textBox3DBSbandwidth.Text = this.ValueTranslator3DBS(((byte)(resultBytes[20] & 0x07)), 200);
                    this.checkBox3DBSspi4.Checked = ((byte)resultBytes[21] & 0x80) != 0;
                    this.checkBox3DBSadvINT.Checked = ((byte)resultBytes[21] & 0x40) != 0;
                    this.checkBox3DBSnewINT.Checked = ((byte)resultBytes[21] & 0x20) != 0;
                    this.checkBox3DBSlatchINT.Checked = ((byte)resultBytes[21] & 0x10) != 0;
                    this.checkBox3DBSshadowDis.Checked = ((byte)resultBytes[21] & 0x08) != 0;
                    this.textBox3DBSwakeUpPause.Text = this.ValueTranslator3DBS(((byte)(resultBytes[21] >> 1 & 0x03)), 211);
                    this.checkBox3DBSwakeUp.Checked = ((byte)resultBytes[21] & 0x01) != 0;
                }
                catch
                {
                    this.ShowError(10);
                }
            }
        }

        /// <summary>
        /// Gets current acceleration values.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Button3DBSValues_Click(object sender, EventArgs e)
        {
            string command = "s70";
            command += " 02 s71 06 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(10);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // get Status
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.Set3DBSValues(resultBytes);
                }
                catch
                {
                    this.ShowError(10);
                }
            }
        }

        /// <summary>
        /// Commits acceleration valuesto the form.
        /// </summary>
        /// <param name="resultBytes">ResultBytes als byte[].</param>
        private void Set3DBSValues(byte[] resultBytes)
        {
            int tempvalue = 0;

            tempvalue = (int)(resultBytes[0] >> 6);
            tempvalue += resultBytes[1] * 4;
            this.textBox3DBSaccX.Text = tempvalue.ToString("X").ToUpper();
            this.checkBox3DBSnewX.Checked = ((byte)resultBytes[0] & 0x01) != 0;
            tempvalue = (int)(resultBytes[2] >> 6);
            tempvalue += resultBytes[3] * 4;
            this.textBox3DBSaccY.Text = tempvalue.ToString("X").ToUpper();
            this.checkBox3DBSnewY.Checked = ((byte)resultBytes[2] & 0x01) != 0;
            tempvalue = (int)(resultBytes[4] >> 6);
            tempvalue += resultBytes[5] * 4;
            this.textBox3DBSaccZ.Text = tempvalue.ToString("X").ToUpper();
            this.checkBox3DBSnewZ.Checked = ((byte)resultBytes[4] & 0x01) != 0;                
        }

        /// <summary>
        /// Converts byte to string according to the type.
        /// </summary>
        /// <param name="value">Value as byte.</param>
        /// <param name="type">Type as int.</param>
        /// <returns>Converted string.</returns>
        private string ValueTranslator3DBS(byte value, int type)
        {
            switch (type)
            {
                case 211:
                    // byte 0x15 (21) bit 2-1 wake_up_pause
                    if (value == 0x00)
                    {
                        return "20 ms";
                    }
                    else if (value == 0x01)
                    {
                        return "80 ms";
                    }
                    else if (value == 0x02)
                    {
                        return "320 ms";
                    }
                    else
                    {
                        return "2560 ms";
                    }

                case 200:
                    // byte 0x14 (20) bit 2-0 bandwidth
                    if (value == 0x00)
                    {
                        return "25 Hz";
                    }
                    else if (value == 0x01)
                    {
                        return "50 Hz";
                    }
                    else if (value == 0x02)
                    {
                        return "100 Hz";
                    }
                    else if (value == 0x03)
                    {
                        return "190 Hz";
                    }
                    else if (value == 0x04)
                    {
                        return "375 Hz";
                    }
                    else if (value == 0x05)
                    {
                        return "750 Hz";
                    }
                    else if (value == 0x06)
                    {
                        return "1500 Hz";
                    }
                    else
                    {
                        return "Nac"; // not authorised code
                    }

                case 203:
                    // byte 0x14 (20) bit 4-3 range
                    if (value == 0x00)
                    {
                        return "+/- 2g";
                    }
                    else if (value == 0x01)
                    {
                        return "+/- 4g";
                    }
                    else if (value == 0x02)
                    {
                        return "+/- 8g";
                    }
                    else
                    {
                        return "Nac"; // not authorised code
                    }

                case 176:
                    // byte 0x11 (17) bit 5-4 any_motion_dur
                    if (value == 0x00)
                    {
                        return "1";
                    }
                    else if (value == 0x01)
                    {
                        return "3";
                    }
                    else if (value == 0x02)
                    {
                        return "5";
                    }
                    else
                    {
                        return "6";
                    }

                case 114:
                case 112:
                    // byte 0x0B (11) bit 5-4 counter_HG bzw 3-2 counter_LG
                    if (value == 0x00)
                    {
                        return "reset";
                    }
                    else if (value == 0x01)
                    {
                        return "1 LSB/ms";
                    }
                    else if (value == 0x02)
                    {
                        return "2 LSB/ms";
                    }
                    else
                    {
                        return "3 LSB/ms";
                    }
                
                default:
                    return value.ToString("X").ToUpper();
            }
        }

        /// <summary>
        /// Sets the range of the sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Button3DBSsetRange_Click(object sender, EventArgs e)
        {
            string command = "s70";
            command += " 14 s71 01 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(10);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    int newbyte = resultBytes[0] & 0xE0;
                    newbyte += (this.comboBox3DBSrange.SelectedIndex * 8) + this.comboBox3DBSbandwidth.SelectedIndex;

                    command = "s70 14 ";
                    command += newbyte.ToString("X2").ToUpper();
                    command += " p";

                    this.listBoxOutput.Items.Add(command);
                    this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                    if (!this.usbI2C.SendCommand(command))
                    {
                        this.ShowError(11);
                    }

                    this.Button3DBSStatus_Click(sender, e);
                }
                catch
                {
                    this.ShowError(10);
                }
            }
        }
        #endregion

        #region LED-I2C
        #region PWM
        /// <summary>
        /// Reads current PWM-values from register.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetPWM_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 82 s" + ((int)address + 1).ToString("X2") + " 10 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    for (int i = 0; i < 16; i++)
                    {
                        this.SetLEDpwmValue(i + 1, (int)resultBytes[i]);
                    }
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets PWM-values (Click-function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetPWM_Click(object sender, EventArgs e)
        {
            this.SendLEDpwm();
        }

        /// <summary>
        /// Sets PWM-values.
        /// </summary>
        private void SendLEDpwm()
        {
            string command = this.LEDpwmCommand();
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(21);
            }
        }

        /// <summary>
        /// Add a command for setting the PWM-values to the command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDpwmList_Click(object sender, EventArgs e)
        {
            string command = this.LEDpwmCommand();
            this.listBoxLEDsequence.Items.Add(command); 
        }

        /// <summary>
        /// Converts the binary input of the text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDpwmB_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-1 ]{8}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 8 Binärzahlen (0 oder 1) ein.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 2);
                int ledID = int.Parse(textBoxSender.Name.Split('B')[textBoxSender.Name.Split('B').Length - 1]);
                this.SetLEDpwmValue(ledID, newvalue);
            }
        }

        /// <summary>
        /// Validates hex input in text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDpwmX_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-9ABCDEFabcdef]{1,2}$)|(^0x[0-9ABCDEFabcdef]{1,2}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 2 Hexadezimalzahlen (0-F) mit oder ohne führenden 0x.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 16);
                int ledID = int.Parse(textBoxSender.Name.Split('X')[textBoxSender.Name.Split('X').Length - 1]);
                this.SetLEDpwmValue(ledID, newvalue);
            }
        }

        /// <summary>
        /// Handles changes of the track bar.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TrackBarLEDpwm_ValueChanged(object sender, EventArgs e)
        {
            TrackBar trackBarSender = (TrackBar)sender;
            int newvalue = trackBarSender.Value;
            int ledID = int.Parse(trackBarSender.Name.Split('m')[trackBarSender.Name.Split('m').Length - 1]);
            this.SetLEDpwmValue(ledID, newvalue);
        }

        /// <summary>
        /// Sets the display forms to the same values.
        /// </summary>
        /// <param name="ledID">LED to change as int.</param>
        /// <param name="value">Target value as int.</param>
        private void SetLEDpwmValue(int ledID, int value)
        {
            string hexstring = "0x" + value.ToString("X2");
            string binaerstring = Convert.ToString(value, 2);
            for (int i = binaerstring.Length; i < 8; i++)
            {
                binaerstring = "0" + binaerstring;
            }

            switch (ledID)
            {
                case 1:
                    this.textBoxLEDpwmB1.Text = binaerstring;
                    this.textBoxLEDpwmX1.Text = hexstring;
                    this.trackBarLEDpwm1.Value = value;
                    break;
                case 2:
                    this.textBoxLEDpwmB2.Text = binaerstring;
                    this.textBoxLEDpwmX2.Text = hexstring;
                    this.trackBarLEDpwm2.Value = value;
                    break;
                case 3:
                    this.textBoxLEDpwmB3.Text = binaerstring;
                    this.textBoxLEDpwmX3.Text = hexstring;
                    this.trackBarLEDpwm3.Value = value;
                    break;
                case 4:
                    this.textBoxLEDpwmB4.Text = binaerstring;
                    this.textBoxLEDpwmX4.Text = hexstring;
                    this.trackBarLEDpwm4.Value = value;
                    break;
                case 5:
                    this.textBoxLEDpwmB5.Text = binaerstring;
                    this.textBoxLEDpwmX5.Text = hexstring;
                    this.trackBarLEDpwm5.Value = value;
                    break;
                case 6:
                    this.textBoxLEDpwmB6.Text = binaerstring;
                    this.textBoxLEDpwmX6.Text = hexstring;
                    this.trackBarLEDpwm6.Value = value;
                    break;
                case 7:
                    this.textBoxLEDpwmB7.Text = binaerstring;
                    this.textBoxLEDpwmX7.Text = hexstring;
                    this.trackBarLEDpwm7.Value = value;
                    break;
                case 8:
                    this.textBoxLEDpwmB8.Text = binaerstring;
                    this.textBoxLEDpwmX8.Text = hexstring;
                    this.trackBarLEDpwm8.Value = value;
                    break;
                case 9:
                    this.textBoxLEDpwmB9.Text = binaerstring;
                    this.textBoxLEDpwmX9.Text = hexstring;
                    this.trackBarLEDpwm9.Value = value;
                    break;
                case 10:
                    this.textBoxLEDpwmB10.Text = binaerstring;
                    this.textBoxLEDpwmX10.Text = hexstring;
                    this.trackBarLEDpwm10.Value = value;
                    break;
                case 11:
                    this.textBoxLEDpwmB11.Text = binaerstring;
                    this.textBoxLEDpwmX11.Text = hexstring;
                    this.trackBarLEDpwm11.Value = value;
                    break;
                case 12:
                    this.textBoxLEDpwmB12.Text = binaerstring;
                    this.textBoxLEDpwmX12.Text = hexstring;
                    this.trackBarLEDpwm12.Value = value;
                    break;
                case 13:
                    this.textBoxLEDpwmB13.Text = binaerstring;
                    this.textBoxLEDpwmX13.Text = hexstring;
                    this.trackBarLEDpwm13.Value = value;
                    break;
                case 14:
                    this.textBoxLEDpwmB14.Text = binaerstring;
                    this.textBoxLEDpwmX14.Text = hexstring;
                    this.trackBarLEDpwm14.Value = value;
                    break;
                case 15:
                    this.textBoxLEDpwmB15.Text = binaerstring;
                    this.textBoxLEDpwmX15.Text = hexstring;
                    this.trackBarLEDpwm15.Value = value;
                    break;
                case 16:
                    this.textBoxLEDpwmB16.Text = binaerstring;
                    this.textBoxLEDpwmX16.Text = hexstring;
                    this.trackBarLEDpwm16.Value = value;
                    break;

                default:
                    break;
            }

            if (this.checkBoxLEDpwmAutoSend.Checked)
            {
                this.SendLEDpwm();
            }
        }

        /// <summary>
        /// Builds the command string.
        /// </summary>
        /// <returns>Command to set the PWM-register.</returns>
        private string LEDpwmCommand()
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 82";
            command += " " + this.trackBarLEDpwm1.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm2.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm3.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm4.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm5.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm6.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm7.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm8.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm9.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm10.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm11.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm12.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm13.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm14.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm15.Value.ToString("X2");
            command += " " + this.trackBarLEDpwm16.Value.ToString("X2");
            command += " p";

            return command;
        }
        #endregion

        #region LEDoutput
        /// <summary>
        /// DoWork-method of BackgroundWorker for showing the sequence on I2C-LED.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BackgroundWorkerLEDsequence_DoWork(object sender, DoWorkEventArgs e)
        {
            for (this.listPosition = 0; this.listPosition < this.listBoxLEDsequence.Items.Count; this.listPosition++)
            {
                if (this.backgroundWorkerLEDsequence.CancellationPending)
                {
                    return;
                }

                string command = this.listBoxLEDsequence.Items[this.listPosition].ToString().Trim();
                if (command.StartsWith("wait"))
                {
                    System.Threading.Thread.Sleep(int.Parse(command.Split(' ')[1]));
                }
                else
                {
                    this.listBoxOutput.BeginInvoke(new DelegateAddCommandToListBoxOutput(this.AddCommandToListBoxOutput), command);
                    if (!this.usbI2C.SendCommand(command))
                    {
                        this.ShowError(23);
                    }
                }

                this.listBoxLEDsequence.BeginInvoke(new DelegateSetSelectedIndexOfListBoxLEDsequence(this.SetSelectedIndexOfListBoxLEDsequence));
            }
        }

        /// <summary>
        /// Adds a command to the list of send commands.
        /// </summary>
        /// <param name="command">Command to add.</param>
        private void AddCommandToListBoxOutput(string command)
        {
            this.listBoxOutput.Items.Add(command);
        }

        /// <summary>
        /// Updates the selected index of ListBoxLEDsequence.
        /// </summary>
        private void SetSelectedIndexOfListBoxLEDsequence()
        {
            if (!(this.listPosition + 1 >= this.listBoxLEDsequence.Items.Count))
            {
                this.listBoxLEDsequence.SelectedIndex = this.listPosition;
            }
        }

        /// <summary>
        /// Starts work off of command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDStart_Click(object sender, EventArgs e)
        {
            if (this.backgroundWorkerLEDsequence.IsBusy)
            {
                return;
            }

            this.backgroundWorkerLEDsequence.RunWorkerAsync();
        }

        /// <summary>
        /// Stopps work off of command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDStop_Click(object sender, EventArgs e)
        {
            if (this.backgroundWorkerLEDsequence != null && this.backgroundWorkerLEDsequence.IsBusy)
            {
                this.backgroundWorkerLEDsequence.CancelAsync();
                this.listPosition = 0;
            }
        }

        /// <summary>
        /// Clears the command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDClear_Click(object sender, EventArgs e)
        {
            this.listBoxLEDsequence.Items.Clear();
        }

        /// <summary>
        /// Saves the command list as macro for the device.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDListToMakro_Click(object sender, EventArgs e)
        {
            if (this.listBoxLEDsequence.Items.Count <= 0)
            {
                return;
            }

            // stop macro before writing a new one
            this.StopMacro();
            System.Threading.Thread.Sleep(500);

            this.listPosition = 0;
            int addedCommands = 0;
            string result = this.usbI2C.DeleteMakro();
            this.listBoxOutput.Items.Add("u v00{} u");
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (result != string.Empty)
            {
                this.listBoxInput.Items.AddRange(result.Split('\r'));
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                string command = "V 00{";
                string temp = string.Empty;
                int wait;
                int maxlenght = 255 + 5;
                if (this.checkBoxLEDoutRepeat.Checked)
                {
                    maxlenght = maxlenght - 4;
                }

                while (command.Length < maxlenght)
                {
                    temp = this.listBoxLEDsequence.Items[this.listPosition].ToString().Trim();

                    if (temp.StartsWith("wait"))
                    {
                        wait = int.Parse(temp.Split(' ')[1]);
                        wait = wait * 100;
                        temp = "L" + wait.ToString("X4");
                    }

                    if ((command.Length + temp.Length + 1) < maxlenght)
                    {
                        command += temp + " ";
                    }
                    else
                    {
                        this.listPosition = 0;

                        MessageBox.Show(
                            string.Format("Der Makrospeicher reicht nicht aus, um alle Befehle abzuspeichern.\nEs wurden nur die ersten {0} Befehle übernommen.", addedCommands),
                            "Warnung",
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Warning);

                        break;
                    }

                    this.listPosition++;

                    if (this.listBoxLEDsequence.Items.Count <= this.listPosition)
                    {
                        this.listPosition = 0;
                        break;
                    }

                    addedCommands++;
                }

                if (this.checkBoxLEDoutRepeat.Checked)
                {
                    command += " >00";
                }

                command += "}";
                this.listBoxOutput.Items.Add(command);
                this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
                if (!this.usbI2C.SendCommand(command))
                {
                    this.ShowError(22);
                }
            }
            else
            {
                this.ShowError(0);
            }
        }

        /// <summary>
        /// Adds a pause to the command list.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDPause_Click(object sender, EventArgs e)
        {
            string command = "wait " + this.numericUpDownLEDinterval.Value.ToString() + " * 100 ms";
            this.listBoxLEDsequence.Items.Add(command);
        }

        /// <summary>
        /// Reads out the register of led outputs.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetStatus_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 94 s" + ((int)address + 1).ToString("X2") + " 04 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(20);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.comboBoxLEDout1.SelectedIndex = resultBytes[0] & 0x3;
                    this.comboBoxLEDout2.SelectedIndex = resultBytes[0] >> 2 & 0x3;
                    this.comboBoxLEDout3.SelectedIndex = resultBytes[0] >> 4 & 0x3;
                    this.comboBoxLEDout4.SelectedIndex = resultBytes[0] >> 6 & 0x3;
                    this.comboBoxLEDout5.SelectedIndex = resultBytes[1] & 0x3;
                    this.comboBoxLEDout6.SelectedIndex = resultBytes[1] >> 2 & 0x3;
                    this.comboBoxLEDout7.SelectedIndex = resultBytes[1] >> 4 & 0x3;
                    this.comboBoxLEDout8.SelectedIndex = resultBytes[1] >> 6 & 0x3;
                    this.comboBoxLEDout9.SelectedIndex = resultBytes[2] & 0x3;
                    this.comboBoxLEDout10.SelectedIndex = resultBytes[2] >> 2 & 0x3;
                    this.comboBoxLEDout11.SelectedIndex = resultBytes[2] >> 4 & 0x3;
                    this.comboBoxLEDout12.SelectedIndex = resultBytes[2] >> 6 & 0x3;
                    this.comboBoxLEDout13.SelectedIndex = resultBytes[3] & 0x3;
                    this.comboBoxLEDout14.SelectedIndex = resultBytes[3] >> 2 & 0x3;
                    this.comboBoxLEDout15.SelectedIndex = resultBytes[3] >> 4 & 0x3;
                    this.comboBoxLEDout16.SelectedIndex = resultBytes[3] >> 6 & 0x3;
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets the led outputs.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetStatus_Click(object sender, EventArgs e)
        {
            string command = this.LEDsetCommand();
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(21);
            }
        }

        /// <summary>
        /// Adds the command to set the led outputs.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDoutputStatusList_Click(object sender, EventArgs e)
        {
            string command = this.LEDsetCommand();
            this.listBoxLEDsequence.Items.Add(command); 
        }

        /// <summary>
        /// Builds the command to set the led outputs.
        /// </summary>
        /// <returns>Built command.</returns>
        private string LEDsetCommand()
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 94";
            int temp;
            temp = this.comboBoxLEDout1.SelectedIndex + (this.comboBoxLEDout2.SelectedIndex * 4);
            temp += (this.comboBoxLEDout3.SelectedIndex * 16) + (this.comboBoxLEDout4.SelectedIndex * 64);
            command += " " + temp.ToString("X2");
            temp = this.comboBoxLEDout5.SelectedIndex + (this.comboBoxLEDout6.SelectedIndex * 4);
            temp += (this.comboBoxLEDout7.SelectedIndex * 16) + (this.comboBoxLEDout8.SelectedIndex * 64);
            command += " " + temp.ToString("X2");
            temp = this.comboBoxLEDout9.SelectedIndex + (this.comboBoxLEDout10.SelectedIndex * 4);
            temp += (this.comboBoxLEDout11.SelectedIndex * 16) + (this.comboBoxLEDout12.SelectedIndex * 64);
            command += " " + temp.ToString("X2");
            temp = this.comboBoxLEDout13.SelectedIndex + (this.comboBoxLEDout14.SelectedIndex * 4);
            temp += (this.comboBoxLEDout15.SelectedIndex * 16) + (this.comboBoxLEDout16.SelectedIndex * 64);
            command += " " + temp.ToString("X2");

            command += " p";

            return command;
        }

        /// <summary>
        /// Equalize all outputs.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ComboBoxLEDoutAll_SelectedIndexChanged(object sender, EventArgs e)
        {
            int index = this.comboBoxLEDoutAll.SelectedIndex;
            this.comboBoxLEDout1.SelectedIndex = index;
            this.comboBoxLEDout2.SelectedIndex = index;
            this.comboBoxLEDout3.SelectedIndex = index;
            this.comboBoxLEDout4.SelectedIndex = index;
            this.comboBoxLEDout5.SelectedIndex = index;
            this.comboBoxLEDout6.SelectedIndex = index;
            this.comboBoxLEDout7.SelectedIndex = index;
            this.comboBoxLEDout8.SelectedIndex = index;
            this.comboBoxLEDout9.SelectedIndex = index;
            this.comboBoxLEDout10.SelectedIndex = index;
            this.comboBoxLEDout11.SelectedIndex = index;
            this.comboBoxLEDout12.SelectedIndex = index;
            this.comboBoxLEDout13.SelectedIndex = index;
            this.comboBoxLEDout14.SelectedIndex = index;
            this.comboBoxLEDout15.SelectedIndex = index;
            this.comboBoxLEDout16.SelectedIndex = index;
        }
        #endregion

        #region LED Mode
        /// <summary>
        /// Reads out mode status of I2C-LED (click function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetMode1_Click(object sender, EventArgs e)
        {
            this.GetStatusOfI2CLed(true);
        }

        /// <summary>
        /// Reads out mode status of I2C-LED.
        /// </summary>
        /// <param name="showPossibleError">Show or ignore possible errors.</param>
        private void GetStatusOfI2CLed(bool showPossibleError)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 80 s" + ((int)address + 1).ToString("X2") + " 01 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                if (showPossibleError)
                {
                    this.ShowError(20);
                }
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    if (showPossibleError)
                    {
                        this.ShowError(20);
                    }

                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.checkBoxLEDmode10.Checked = ((byte)resultBytes[0] & 0x01) != 0;
                    this.checkBoxLEDmode11.Checked = ((byte)resultBytes[0] & 0x02) != 0;
                    this.checkBoxLEDmode12.Checked = ((byte)resultBytes[0] & 0x04) != 0;
                    this.checkBoxLEDmode13.Checked = ((byte)resultBytes[0] & 0x08) != 0;
                    this.checkBoxLEDmode14.Checked = ((byte)resultBytes[0] & 0x10) != 0;
                    this.checkBoxLEDmode15.Checked = ((byte)resultBytes[0] & 0x20) != 0;
                    this.checkBoxLEDmode16.Checked = ((byte)resultBytes[0] & 0x40) != 0;
                    this.checkBoxLEDmode17.Checked = ((byte)resultBytes[0] & 0x80) != 0;
                }
                catch
                {
                    if (showPossibleError)
                    {
                        this.ShowError(20);
                    }
                }
            }
        }

        /// <summary>
        /// Sets mode 1.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetMode1_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 80";
            byte newvalue = 0x00;

            if (this.checkBoxLEDmode10.Checked)
            {
                newvalue += 0x01;
            }

            if (this.checkBoxLEDmode11.Checked)
            {
                newvalue += 0x02;
            }

            if (this.checkBoxLEDmode12.Checked)
            {
                newvalue += 0x04;
            }

            if (this.checkBoxLEDmode13.Checked)
            {
                newvalue += 0x08;
            }

            if (this.checkBoxLEDmode14.Checked)
            {
                newvalue += 0x10;
            }

            if (this.checkBoxLEDmode15.Checked)
            {
                newvalue += 0x20;
            }

            if (this.checkBoxLEDmode16.Checked)
            {
                newvalue += 0x40;
            }

            if (this.checkBoxLEDmode17.Checked)
            {
                newvalue += 0x80;
            }

            command += " " + newvalue.ToString("X2");
            command += " p";
            
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(24);
            }
        }

        /// <summary>
        /// Reads out mode 2 from the register.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetMode2_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 81 s" + ((int)address + 1).ToString("X2") + " 01 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(20);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.checkBoxLEDmode23.Checked = ((byte)resultBytes[0] & 0x08) != 0;
                    this.checkBoxLEDmode25.Checked = ((byte)resultBytes[0] & 0x20) != 0;
                    this.checkBoxLEDmode27.Checked = ((byte)resultBytes[0] & 0x80) != 0;
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets mode 2.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetMode2_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 81";
            byte newvalue = 0x00;

            if (this.checkBoxLEDmode23.Checked)
            {
                newvalue += 0x08;
            }

            if (this.checkBoxLEDmode25.Checked)
            {
                newvalue += 0x20;
            }

            if (this.checkBoxLEDmode27.Checked)
            {
                newvalue += 0x80;
            }

            command += " " + newvalue.ToString("X2");
            command += " p"; 
            
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(24);
            }
        }

        /// <summary>
        /// Refreshes mode tab automatically (click function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TabPageLEDmode_Enter(object sender, TabControlCancelEventArgs e)
        {
            if (this.isConnected && e.TabPage.Text == "Mode")
            {
                this.GetStatusOfI2CLed(false);
            }
        }
        #endregion

        #region LED Group
        /// <summary>
        /// Reads out the current GRP-parameter from the register.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetGroup_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 92 s" + ((int)address + 1).ToString("X2") + " 02 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.SetLEDgrpValue("grppwm", (int)resultBytes[0]);
                    this.SetLEDgrpValue("grpfre", (int)resultBytes[1]);
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets GRP-parameter (Click-function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetGroup_Click(object sender, EventArgs e)
        {
            this.SendLEDgrp();
        }

        /// <summary>
        /// Sets GRP-parameter.
        /// </summary>
        private void SendLEDgrp()
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 92";
            command += " " + this.trackBarLEDgrppwm.Value.ToString("X2");
            command += " " + this.trackBarLEDgrpfreq.Value.ToString("X2");
            command += " p";
            
            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(26);
            }
        }

        /// <summary>
        /// Validates binary input of text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDgrpB_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-1 ]{8}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 8 Binärzahlen (0 oder 1) ein.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 2);
                string grp = textBoxSender.Name.Substring(10, 6);
                this.SetLEDgrpValue(grp, newvalue);
            }
        }

        /// <summary>
        /// Validates hex input of text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDgrpX_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-9ABCDEFabcdef]{1,2}$)|(^0x[0-9ABCDEFabcdef]{1,2}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 2 Hexadezimalzahlen (0-F) mit oder ohne führenden 0x.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 16);
                string grp = textBoxSender.Name.Substring(10, 6);
                this.SetLEDgrpValue(grp, newvalue);
            }
        }

        /// <summary>
        /// Tracks changes track bar.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TrackBarLEDgrp_ValueChanged(object sender, EventArgs e)
        {
            TrackBar trackBarSender = (TrackBar)sender;
            int newvalue = trackBarSender.Value;
            string grp = trackBarSender.Name.Substring(11, 6);
            this.SetLEDgrpValue(grp, newvalue);
        }

        /// <summary>
        /// Sets changes to the gui.
        /// </summary>
        /// <param name="grp">Parameter grp as string.</param>
        /// <param name="value">Parameter value as int.</param>
        private void SetLEDgrpValue(string grp, int value)
        {
            string hexstring = "0x" + value.ToString("X2");
            string binaerstring = Convert.ToString(value, 2);
            for (int i = binaerstring.Length; i < 8; i++)
            {
                binaerstring = "0" + binaerstring;
            }

            switch (grp)
            {
                case "grppwm":
                    this.textBoxLEDgrppwmB.Text = binaerstring;
                    this.textBoxLEDgrppwmX.Text = hexstring;
                    this.trackBarLEDgrppwm.Value = value;
                    break;
                case "grpfre":
                    this.textBoxLEDgrpfreqB.Text = binaerstring;
                    this.textBoxLEDgrpfreqX.Text = hexstring;
                    this.trackBarLEDgrpfreq.Value = value;
                    break;

                default:
                    break;
            }

            if (this.checkBoxLEDgrpAutoSend.Checked)
            {
                this.SendLEDgrp();
            }
        }

        /// <summary>
        /// Reads out eflags.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDeflags_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 9D s" + ((int)address + 1).ToString("X2") + " 02 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(20);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.checkBoxLEDeflag1.Checked = ((byte)resultBytes[0] & 0x01) != 0;
                    this.checkBoxLEDeflag2.Checked = ((byte)resultBytes[0] & 0x02) != 0;
                    this.checkBoxLEDeflag3.Checked = ((byte)resultBytes[0] & 0x04) != 0;
                    this.checkBoxLEDeflag4.Checked = ((byte)resultBytes[0] & 0x08) != 0;
                    this.checkBoxLEDeflag5.Checked = ((byte)resultBytes[0] & 0x10) != 0;
                    this.checkBoxLEDeflag6.Checked = ((byte)resultBytes[0] & 0x20) != 0;
                    this.checkBoxLEDeflag7.Checked = ((byte)resultBytes[0] & 0x40) != 0;
                    this.checkBoxLEDeflag8.Checked = ((byte)resultBytes[0] & 0x80) != 0;
                    this.checkBoxLEDeflag9.Checked = ((byte)resultBytes[1] & 0x01) != 0;
                    this.checkBoxLEDeflag10.Checked = ((byte)resultBytes[1] & 0x02) != 0;
                    this.checkBoxLEDeflag11.Checked = ((byte)resultBytes[1] & 0x04) != 0;
                    this.checkBoxLEDeflag12.Checked = ((byte)resultBytes[1] & 0x08) != 0;
                    this.checkBoxLEDeflag13.Checked = ((byte)resultBytes[1] & 0x10) != 0;
                    this.checkBoxLEDeflag14.Checked = ((byte)resultBytes[1] & 0x20) != 0;
                    this.checkBoxLEDeflag15.Checked = ((byte)resultBytes[1] & 0x40) != 0;
                    this.checkBoxLEDeflag16.Checked = ((byte)resultBytes[1] & 0x80) != 0;
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }
        #endregion

        #region LED misc
        /// <summary>
        /// Reads out register with subaddress settings.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetSubadr_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 98 s" + ((int)address + 1).ToString("X2") + " 04 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.SetLEDmiscValue("subadr1", (int)resultBytes[0]);
                    this.SetLEDmiscValue("subadr2", (int)resultBytes[1]);
                    this.SetLEDmiscValue("subadr3", (int)resultBytes[2]);
                    this.SetLEDmiscValue("allcall", (int)resultBytes[3]);
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets subaddress settings.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetSubadr_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 98";
            command += " " + this.trackBarLEDsubadr1.Value.ToString("X2");
            command += " " + this.trackBarLEDsubadr2.Value.ToString("X2");
            command += " " + this.trackBarLEDsubadr3.Value.ToString("X2");
            command += " " + this.trackBarLEDallcall.Value.ToString("X2");
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(25);
            }
        }

        /// <summary>
        /// Validates binary input of a text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDmiscB_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-1 ]{8}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 8 Binärzahlen (0 oder 1) ein.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 2);
                string misc = textBoxSender.Name.Substring(10, 8);
                misc = misc.Replace("B", string.Empty);
                this.SetLEDmiscValue(misc, newvalue);
            }
        }

        /// <summary>
        /// Validates hex input of a text box.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TextBoxLEDmiscX_Validating(object sender, CancelEventArgs e)
        {
            TextBox textBoxSender = (TextBox)sender;
            Regex r = new Regex("(^[0-9ABCDEFabcdef]{1,2}$)|(^0x[0-9ABCDEFabcdef]{1,2}$)");
            if (!r.IsMatch(textBoxSender.Text))
            {
                MessageBox.Show("Bitte geben Sie 2 Hexadezimalzahlen (0-F) mit oder ohne führenden 0x.");
                textBoxSender.Focus();
            }
            else
            {
                int newvalue = Convert.ToInt32(textBoxSender.Text, 16);
                string misc = textBoxSender.Name.Substring(10, 8);
                misc = misc.Replace("X", string.Empty);
                this.SetLEDmiscValue(misc, newvalue);
            }
        }

        /// <summary>
        /// Tracks changes of a track bar.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TrackBarLEDmisc_ValueChanged(object sender, EventArgs e)
        {
            TrackBar trackBarSender = (TrackBar)sender;
            int newvalue = trackBarSender.Value;
            string misc = trackBarSender.Name.Substring(11, 7);
            this.SetLEDmiscValue(misc, newvalue);
        }

        /// <summary>
        /// Commits changed values to the specific forms.
        /// </summary>
        /// <param name="misc">Parameter misc as string.</param>
        /// <param name="value">Parameter value as int.</param>
        private void SetLEDmiscValue(string misc, int value)
        {
            if (misc == "current")
            {
                value = value % 64;
            }
            else
            {
                value = value - (value % 2);
            }

            string hexstring = "0x" + value.ToString("X2");
            string binaerstring = Convert.ToString(value, 2);
            for (int i = binaerstring.Length; i < 8; i++)
            {
                binaerstring = "0" + binaerstring;
            }

            switch (misc)
            {
                case "subadr1":
                    this.textBoxLEDsubadrB1.Text = binaerstring;
                    this.textBoxLEDsubadrX1.Text = hexstring;
                    this.trackBarLEDsubadr1.Value = value;
                    break;
                case "subadr2":
                    this.textBoxLEDsubadrB2.Text = binaerstring;
                    this.textBoxLEDsubadrX2.Text = hexstring;
                    this.trackBarLEDsubadr2.Value = value;
                    break;
                case "subadr3":
                    this.textBoxLEDsubadrB3.Text = binaerstring;
                    this.textBoxLEDsubadrX3.Text = hexstring;
                    this.trackBarLEDsubadr3.Value = value;
                    break;
                case "allcall":
                    this.textBoxLEDallcallB.Text = binaerstring;
                    this.textBoxLEDallcallX.Text = hexstring;
                    this.trackBarLEDallcall.Value = value;
                    break;
                case "current":
                    this.textBoxLEDcurrentmultiplierB.Text = binaerstring;
                    this.textBoxLEDcurrentmultiplierX.Text = hexstring;
                    this.trackBarLEDcurrentmultiplier.Value = value;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// Reads out IREF-parameter from register.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDgetIREF_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 9C s" + ((int)address + 1).ToString("X2") + " 01 p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            string result = this.usbI2C.SendReceiveCommand(command);
            if (result == string.Empty)
            {
                this.ShowError(20);
            }
            else
            {
                this.listBoxInput.Items.Add(result);
                this.listBoxInput.TopIndex = this.listBoxInput.Items.Count - 1;

                if (result.StartsWith("Err") || result.StartsWith("R"))
                {
                    this.ShowError(10);
                    return;
                }

                try
                {
                    // Status auswerten
                    result = result.Replace(" ", string.Empty);
                    byte[] resultBytes = new byte[result.Length / 2];
                    for (int i = 0; i < result.Length; i += 2)
                    {
                        resultBytes[i / 2] = Convert.ToByte(result.Substring(i, 2), 16);
                    }

                    this.SetLEDmiscValue("current", (int)resultBytes[0]);
                    this.comboBoxLEDcm.SelectedIndex = resultBytes[0] >> 6 & 0x1;
                    this.comboBoxLEDhc.SelectedIndex = resultBytes[0] >> 7 & 0x1;
                }
                catch
                {
                    this.ShowError(20);
                }
            }
        }

        /// <summary>
        /// Sets IREF-parameter.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ButtonLEDsetIREF_Click(object sender, EventArgs e)
        {
            int address = int.Parse(this.comboBoxIICAdress.SelectedItem.ToString().Split(',')[0]);
            string command = "s" + address.ToString("X2");
            command += " 9C";
            int temp = this.trackBarLEDcurrentmultiplier.Value;
            temp += 64 * this.comboBoxLEDhc.SelectedIndex;
            temp += 128 * this.comboBoxLEDcm.SelectedIndex;
            command += " " + temp.ToString("X2");
            command += " p";

            this.listBoxOutput.Items.Add(command);
            this.listBoxOutput.TopIndex = this.listBoxOutput.Items.Count - 1;
            if (!this.usbI2C.SendCommand(command))
            {
                this.ShowError(27);
            }
        }
        #endregion
        #endregion

        #region I2C-LCD
        /// <summary>
        /// Initializes the I2C-LCD.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtI2CLCDInit_Click(object sender, EventArgs e)
        {
            this.SetDataLcd();

            this.Lcd.InitDevice();
        }

        /// <summary>
        /// Sets the UI-Elements of the I2C-LCD.
        /// </summary>
        private void SetDataLcd()
        {
            LCD.UiElements uiElements = new LCD.UiElements();
            uiElements.ListBoxOutput = this.listBoxOutput;
            uiElements.ListBoxInput = this.listBoxInput;
            this.Lcd.ImportUiElements(uiElements);
        }

        /// <summary>
        /// Set wanted LEDs.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtI2CLCDLED_Click(object sender, EventArgs e)
        {
            this.SetDataLcd();

            LCD.LedKeyState ledState = new LCD.LedKeyState();
            ledState.Object1 = !this.chbI2CLCDled1.Checked;
            ledState.Object2 = !this.chbI2CLCDled2.Checked;
            ledState.Object3 = !this.chbI2CLCDled3.Checked;
            ledState.Object4 = !this.chbI2CLCDled4.Checked;
            this.Lcd.SetLeds(ledState);
        }

        /// <summary>
        /// Get the state of keys.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtI2CLCDTaster_Click(object sender, EventArgs e)
        {
            this.SetDataLcd();

            LCD.LedKeyState keyState = this.Lcd.GetKeyState();
            this.chbI2CLCDta1.Checked = keyState.Object1;
            this.chbI2CLCDta2.Checked = keyState.Object2;
            this.chbI2CLCDta3.Checked = keyState.Object3;
            this.chbI2CLCDta4.Checked = keyState.Object4;
        }

        /// <summary>
        /// Trigger the display of the I2C-LCD.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtI2CLCDTriggerLcd_Click(object sender, EventArgs e)
        {
            this.SetDataLcd();

            string lcdState = this.GetLcdState();
            this.Lcd.TriggerLcd(lcdState);
        }

        /// <summary>
        /// Get wanted state of the display of the I2C-LCD.
        /// </summary>
        /// <returns>Command to transmit.</returns>
        private string GetLcdState()
        {
            string lcdState = string.Empty;

            // Byte 1
            byte lcdState1 = 0x00;
            if (this.ptI2CLCDvolt.Image != null)
            {
                lcdState1 += 0x80;
            }

            if (this.ptI2CLCDohm.Image != null)
            {
                lcdState1 += 0x40;
            }

            if (this.ptI2CLCDstunden.Image != null)
            {
                lcdState1 += 0x20;
            }

            if (this.ptI2CLCDampere.Image != null)
            {
                lcdState1 += 0x10;
            }

            if (this.ptI2CLCDT4.Image != null)
            {
                lcdState1 += 0x08;
            }

            if (this.ptI2CLCDT3.Image != null)
            {
                lcdState1 += 0x04;
            }

            if (this.ptI2CLCDT2.Image != null)
            {
                lcdState1 += 0x02;
            }

            if (this.ptI2CLCDT1.Image != null)
            {
                lcdState1 += 0x01;
            }

            // Byte 2
            byte lcdState2 = 0x00;
            if (this.ptI2CLCDR1.Image != null)
            {
                lcdState2 += 0x80;
            }

            if (this.ptI2CLCDR2.Image != null)
            {
                lcdState2 += 0x40;
            }

            if (this.ptI2CLCDnr1E.Image != null)
            {
                lcdState2 += 0x20;
            }

            if (this.ptI2CLCDnr1F.Image != null)
            {
                lcdState2 += 0x10;
            }

            if (this.ptI2CLCDcelsius.Image != null)
            {
                lcdState2 += 0x08;
            }

            if (this.ptI2CLCDprozent.Image != null)
            {
                lcdState2 += 0x04;
            }

            if (this.ptI2CLCDmikro.Image != null)
            {
                lcdState2 += 0x02;
            }

            if (this.ptI2CLCDmini.Image != null)
            {
                lcdState2 += 0x01;
            }

            // Byte 3
            byte lcdState3 = 0x00;
            if (this.ptI2CLCDnr1D.Image != null)
            {
                lcdState3 += 0x80;
            }

            if (this.ptI2CLCDnr1J.Image != null)
            {
                lcdState3 += 0x40;
            }

            if (this.ptI2CLCDnr1I.Image != null)
            {
                lcdState3 += 0x20;
            }

            if (this.ptI2CLCDnr1L.Image != null)
            {
                lcdState3 += 0x10;
            }

            if (this.ptI2CLCDnr1M.Image != null)
            {
                lcdState3 += 0x08;
            }

            if (this.ptI2CLCDnr1G.Image != null)
            {
                lcdState3 += 0x04;
            }

            if (this.ptI2CLCDnr1K.Image != null)
            {
                lcdState3 += 0x02;
            }

            if (this.ptI2CLCDnr1A.Image != null)
            {
                lcdState3 += 0x01;
            }

            // Byte 4
            byte lcdState4 = 0x00;
            if (this.ptI2CLCDnr4C.Image != null)
            {
                lcdState4 += 0x80;
            }

            if (this.ptI2CLCDnr4N.Image != null)
            {
                lcdState4 += 0x40;
            }

            if (this.ptI2CLCDnr4H.Image != null)
            {
                lcdState4 += 0x20;
            }

            if (this.ptI2CLCDnr4B.Image != null)
            {
                lcdState4 += 0x10;
            }

            if (this.ptI2CLCDnr1C.Image != null)
            {
                lcdState4 += 0x08;
            }

            if (this.ptI2CLCDnr1N.Image != null)
            {
                lcdState4 += 0x04;
            }

            if (this.ptI2CLCDnr1H.Image != null)
            {
                lcdState4 += 0x02;
            }

            if (this.ptI2CLCDnr1B.Image != null)
            {
                lcdState4 += 0x01;
            }

            // Byte 5
            byte lcdState5 = 0x00;
            if (this.ptI2CLCDnr4M.Image != null)
            {
                lcdState5 += 0x80;
            }

            if (this.ptI2CLCDnr4G.Image != null)
            {
                lcdState5 += 0x40;
            }

            if (this.ptI2CLCDnr4K.Image != null)
            {
                lcdState5 += 0x20;
            }

            if (this.ptI2CLCDnr4A.Image != null)
            {
                lcdState5 += 0x10;
            }

            if (this.ptI2CLCDnr4D.Image != null)
            {
                lcdState5 += 0x08;
            }

            if (this.ptI2CLCDnr4J.Image != null)
            {
                lcdState5 += 0x04;
            }

            if (this.ptI2CLCDnr4I.Image != null)
            {
                lcdState5 += 0x02;
            }

            if (this.ptI2CLCDnr4L.Image != null)
            {
                lcdState5 += 0x01;
            }

            // Byte 6
            byte lcdState6 = 0x00;
            if (this.ptI2CLCDnr3C.Image != null)
            {
                lcdState6 += 0x80;
            }

            if (this.ptI2CLCDnr3N.Image != null)
            {
                lcdState6 += 0x40;
            }

            if (this.ptI2CLCDnr3H.Image != null)
            {
                lcdState6 += 0x20;
            }

            if (this.ptI2CLCDnr3B.Image != null)
            {
                lcdState6 += 0x10;
            }

            if (this.ptI2CLCDP3.Image != null)
            {
                lcdState6 += 0x04;
            }

            if (this.ptI2CLCDnr4E.Image != null)
            {
                lcdState6 += 0x02;
            }

            if (this.ptI2CLCDnr4F.Image != null)
            {
                lcdState6 += 0x01;
            }

            // Byte 7
            byte lcdState7 = 0x00;
            if (this.ptI2CLCDnr3M.Image != null)
            {
                lcdState7 += 0x80;
            }

            if (this.ptI2CLCDnr3G.Image != null)
            {
                lcdState7 += 0x40;
            }

            if (this.ptI2CLCDnr3K.Image != null)
            {
                lcdState7 += 0x20;
            }

            if (this.ptI2CLCDnr3A.Image != null)
            {
                lcdState7 += 0x10;
            }

            if (this.ptI2CLCDnr3D.Image != null)
            {
                lcdState7 += 0x08;
            }

            if (this.ptI2CLCDnr3J.Image != null)
            {
                lcdState7 += 0x04;
            }

            if (this.ptI2CLCDnr3I.Image != null)
            {
                lcdState7 += 0x02;
            }

            if (this.ptI2CLCDnr3L.Image != null)
            {
                lcdState7 += 0x01;
            }

            // Byte 8
            byte lcdState8 = 0x00;
            if (this.ptI2CLCDnr2C.Image != null)
            {
                lcdState8 += 0x80;
            }

            if (this.ptI2CLCDnr2N.Image != null)
            {
                lcdState8 += 0x40;
            }

            if (this.ptI2CLCDnr2H.Image != null)
            {
                lcdState8 += 0x20;
            }

            if (this.ptI2CLCDnr2B.Image != null)
            {
                lcdState8 += 0x10;
            }

            if (this.ptI2CLCDP2.Image != null)
            {
                lcdState8 += 0x08;
            }

            if (this.ptI2CLCDC1.Image != null)
            {
                lcdState8 += 0x04;
            }

            if (this.ptI2CLCDnr3E.Image != null)
            {
                lcdState8 += 0x02;
            }

            if (this.ptI2CLCDnr3F.Image != null)
            {
                lcdState8 += 0x01;
            }

            // Byte 9
            byte lcdState9 = 0x00;
            if (this.ptI2CLCDnr2M.Image != null)
            {
                lcdState9 += 0x80;
            }

            if (this.ptI2CLCDnr2G.Image != null)
            {
                lcdState9 += 0x40;
            }

            if (this.ptI2CLCDnr2K.Image != null)
            {
                lcdState9 += 0x20;
            }

            if (this.ptI2CLCDnr2A.Image != null)
            {
                lcdState9 += 0x10;
            }

            if (this.ptI2CLCDnr2D.Image != null)
            {
                lcdState9 += 0x08;
            }

            if (this.ptI2CLCDnr2J.Image != null)
            {
                lcdState9 += 0x04;
            }

            if (this.ptI2CLCDnr2I.Image != null)
            {
                lcdState9 += 0x02;
            }

            if (this.ptI2CLCDnr2L.Image != null)
            {
                lcdState9 += 0x01;
            }

            // Byte 10
            byte lcdState10 = 0x00;

            if (this.ptI2CLCDP1.Image != null)
            {
                lcdState10 += 0x04;
            }

            if (this.ptI2CLCDnr2E.Image != null)
            {
                lcdState10 += 0x02;
            }

            if (this.ptI2CLCDnr2F.Image != null)
            {
                lcdState10 += 0x01;
            }

            lcdState += lcdState1.ToString("X2") + " ";
            lcdState += lcdState2.ToString("X2") + " ";
            lcdState += lcdState3.ToString("X2") + " ";
            lcdState += lcdState4.ToString("X2") + " ";
            lcdState += lcdState5.ToString("X2") + " ";
            lcdState += lcdState6.ToString("X2") + " ";
            lcdState += lcdState7.ToString("X2") + " ";
            lcdState += lcdState8.ToString("X2") + " ";
            lcdState += lcdState9.ToString("X2") + " ";
            lcdState += lcdState10.ToString("X2") + " ";

            return lcdState;
        }

        /// <summary>
        /// Clears the display of the I2C-LCD.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtI2CLCDlcdClear_Click(object sender, EventArgs e)
        {
            this.SetDataLcd();

            this.Lcd.ClearDisplay();

            foreach (PictureBox pictureBox in this.panel3.Controls)
            {
                pictureBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void HorizontalToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalRot;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void HorizontalToggleWhite_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void HorizontalSmallToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalKleinRot;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void VerticalToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalRot;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void VerticalToggleWhite_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void VerticalSmallToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalKleinRot;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void DiagonalOnTopLeftToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerDiagonalObenLinksRot;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void DiagonalOnTopRightToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.NummerDiagonalObenRechtsRot;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void DotToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.PunktRot;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void DotToggleWhite_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.PunktWeiss;
            }
            else
            {
                senderBox.Image = null;
            }

            this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex = 0;
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TraingleToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.DreieckRot;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BatteryToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.Batterie;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void CelsiusToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.CelsiusSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ProzentToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.ProzentSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void MikroToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.MikroSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void MiniToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.MiniSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void AmpereToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.AmpereSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void StundenToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.StundenSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void OhmToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.OhmSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void VoltToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.VoltSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        /// <summary>
        /// Toggles visibility of UI-element.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void DoppelpunktToggle_Click(object sender, EventArgs e)
        {
            PictureBox senderBox = (PictureBox)sender;
            if (senderBox.Image == null)
            {
                senderBox.Image = Eq3.misc.USBI2C.Properties.Resources.DoppelpunktSchwarz;
            }
            else
            {
                senderBox.Image = null;
            }
        }

        #endregion

        #region 4DLED
        #region 4DLED-SAA1064
        /// <summary>
        /// Click-function of button for triggering of 4DLED-SAA1064.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtSaaTrigger_Click(object sender, EventArgs e)
        {
            if (this.rdSaa1064TriggerSingle.Checked)
            {
                this.TriggerSaa1064(null, null);
            }
            else
            {
                if (this.timerTriggerSaa1064 == null || !this.timerTriggerSaa1064.Enabled)
                {
                    int intervall = this.GetIntervallForTimerTrigger(true);
                    if (intervall == -1)
                    {
                        return;
                    }

                    this.timerTriggerSaa1064 = new Timer();
                    this.timerTriggerSaa1064.Interval = intervall;
                    this.timerTriggerSaa1064.Tick += new EventHandler(this.TriggerSaa1064);
                    this.timerTriggerSaa1064.Start();

                    this.btSaaTrigger.Text = "Stopp";
                    this.tbSaa1064TriggerIntervall.Enabled = false;
                    this.rdSaa1064TriggerIntervall.Enabled = false;
                    this.rdSaa1064TriggerSingle.Enabled = false;
                }
                else
                {
                    this.timerTriggerSaa1064.Stop();

                    this.btSaaTrigger.Text = "Senden";
                    this.tbSaa1064TriggerIntervall.Enabled = true;
                    this.rdSaa1064TriggerIntervall.Enabled = true;
                    this.rdSaa1064TriggerSingle.Enabled = true;
                }
            }
        }

        /// <summary>
        /// Gets the intervall for triggering the two chips of I2C-4DLED.
        /// </summary>
        /// <param name="isSaa1064">SAA1064 or MCP9801 of I2C-4DLED is demanded.</param>
        /// <returns>Intervall from gui.</returns>
        private int GetIntervallForTimerTrigger(bool isSaa1064)
        {
            string intervallAsString;
            Regex regex = new Regex("^[0-9]*$");

            if (isSaa1064)
            {
                intervallAsString = this.tbSaa1064TriggerIntervall.Text;
                this.tbSaa1064TriggerIntervall.BackColor = Color.Red;
            }
            else
            {
                intervallAsString = this.tbMCP9801TriggerIntervall.Text;
                this.tbMCP9801TriggerIntervall.BackColor = Color.Red;
            }

            if (!regex.IsMatch(intervallAsString))
            {
                return -1;
            }

            try
            {
                int intervall = Convert.ToInt32(intervallAsString);
                if (intervall <= 0)
                {
                    return -1;
                }

                if (isSaa1064)
                {
                    this.tbSaa1064TriggerIntervall.BackColor = Color.White;
                }
                else
                {
                    this.tbMCP9801TriggerIntervall.BackColor = Color.White;
                }

                return intervall;
            }
            catch
            {
                return -1;
            }
        }

        /// <summary>
        /// Sends a command to the I2C-4DLED-SAA1064 based on the current gui.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TriggerSaa1064(object sender, EventArgs e)
        {
            this.SetDataSaa1064();
            this.I2C4DLed.Saa1064.SendCommand();
        }

        /// <summary>
        /// Sets settings of I2C-4DLED-SAA1064 based on current gui.
        /// </summary>
        private void SetDataSaa1064()
        {
            Saa1064.UiElements uiElements = new Saa1064.UiElements();
            uiElements.CbSlaveAdress = this.comboBoxSaa1064SlaveAddress;
            uiElements.PbDisplay = this.Saa1064GetDisplay();
            uiElements.CbCurrent = this.comboBoxSaa1064CurrentChoice;
            uiElements.ListBoxOutput = this.listBoxOutput;
            uiElements.ListBoxInput = this.listBoxInput;
            this.I2C4DLed.Saa1064.ImportUiElements(uiElements);
        }

        /// <summary>
        /// Gets the current wished display of SAA1064 as list of PictureBox'es.
        /// </summary>
        /// <returns>List of PictureBox'es.</returns>
        private PictureBox[] Saa1064GetDisplay()
        {
            PictureBox[] chosenNumbers = new PictureBox[(4 * 8) - 1]; // 8 Zeichen je Ziffer, vierte Ziffer ohne Punkt

            // 1. Ziffer
            chosenNumbers[0] = this.ptSaa11;
            chosenNumbers[1] = this.ptSaa12;
            chosenNumbers[2] = this.ptSaa13;
            chosenNumbers[3] = this.ptSaa14;
            chosenNumbers[4] = this.ptSaa15;
            chosenNumbers[5] = this.ptSaa16;
            chosenNumbers[6] = this.ptSaa17;
            chosenNumbers[7] = this.ptSaa18;

            // 2. Ziffer
            chosenNumbers[8] = this.ptSaa21;
            chosenNumbers[9] = this.ptSaa22;
            chosenNumbers[10] = this.ptSaa23;
            chosenNumbers[11] = this.ptSaa24;
            chosenNumbers[12] = this.ptSaa25;
            chosenNumbers[13] = this.ptSaa26;
            chosenNumbers[14] = this.ptSaa27;
            chosenNumbers[15] = this.ptSaa28;

            // 3. Ziffer
            chosenNumbers[16] = this.ptSaa31;
            chosenNumbers[17] = this.ptSaa32;
            chosenNumbers[18] = this.ptSaa33;
            chosenNumbers[19] = this.ptSaa34;
            chosenNumbers[20] = this.ptSaa35;
            chosenNumbers[21] = this.ptSaa36;
            chosenNumbers[22] = this.ptSaa37;
            chosenNumbers[23] = this.ptSaa38;

            // 4. Ziffer
            chosenNumbers[24] = this.ptSaa41;
            chosenNumbers[25] = this.ptSaa42;
            chosenNumbers[26] = this.ptSaa43;
            chosenNumbers[27] = this.ptSaa44;
            chosenNumbers[28] = this.ptSaa45;
            chosenNumbers[29] = this.ptSaa46;
            chosenNumbers[30] = this.ptSaa47;

            return chosenNumbers;
        }

        /// <summary>
        /// Reads out chosen option to display on I2C-4DLED-SAA1064 and changes gui.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ComboBoxSaa1064NumbersPatternChoice_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex)
            {
                // manuelles Muster
                case 0:
                    this.DeactivateTimerShowTimeOnSaa1064();
                    this.DeactivateCustomNumbers();
                    break;

                // Uhrzeit
                case 1:
                    this.DeactivateCustomNumbers();

                    this.ActivateTimerShowTimeOnSaa1064();
                    break;

                // Temperaturwert
                case 2:
                    this.DeactivateTimerShowTimeOnSaa1064();
                    this.DeactivateCustomNumbers();

                    this.DrawTemperature();
                    break;

                // Zahleneingabe
                case 3:
                    this.DeactivateTimerShowTimeOnSaa1064();
                    this.ActivateCustomNumbers();
                    break;

                default:
                    break;
            }
        }

        /// <summary>
        /// Activates the timer for showing the current time on I2C-4DLED-SAA1064.
        /// </summary>
        private void ActivateTimerShowTimeOnSaa1064()
        {
            this.timerShowTimeOnSaa1064 = new Timer();
            this.timerShowTimeOnSaa1064.Interval = 500;
            this.timerShowTimeOnSaa1064.Tick += new EventHandler(this.TimerShowTimeOnSaa1064_Tick);
            this.timerShowTimeOnSaa1064.Start();
        }

        /// <summary>
        /// Tick-method for the timer for showing the current time on I2C-4DLED-SAA1064.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TimerShowTimeOnSaa1064_Tick(object sender, EventArgs e)
        {
            this.ShowClockInNumberSample();
        }

        /// <summary>
        /// Deactivates the timer for showing the current time on I2C-4DLED-SAA1064.
        /// </summary>
        private void DeactivateTimerShowTimeOnSaa1064()
        {
            if (this.timerShowTimeOnSaa1064 != null)
            {
                this.timerShowTimeOnSaa1064.Dispose();
                this.timerShowTimeOnSaa1064 = null;
            }
        }

        /// <summary>
        /// Activates clickable numbers field for I2C-4DLED-SAA1064.
        /// </summary>
        private void ActivateCustomNumbers()
        {
            this.tbSaa1064CustomNumbers.Visible = true;
            this.btSaa1064CommitNumbers.Visible = true;
        }

        /// <summary>
        /// Deactivates clickable numbers field for I2C-4DLED-SAA1064.
        /// </summary>
        private void DeactivateCustomNumbers()
        {
            this.tbSaa1064CustomNumbers.Visible = false;
            this.btSaa1064CommitNumbers.Visible = false;
        }

        /// <summary>
        /// Shows current time on numbers field for I2C-4DLED-SAA1064.
        /// </summary>
        private void ShowClockInNumberSample()
        {
            int hours = DateTime.Now.Hour;
            int minutes = DateTime.Now.Minute;

            int[] numbers = new int[4];
            numbers[0] = hours / 10;
            numbers[1] = hours % 10;
            numbers[2] = minutes / 10;
            numbers[3] = minutes % 10;

            bool[] dots = new bool[3];
            dots[0] = false;
            dots[1] = true;
            dots[2] = false;

            this.DrawNumbers(numbers, dots);
        }

        /// <summary>
        /// Draws a specific number on the numbers field for I2C-4DLED-SAA1064.
        /// </summary>
        /// <param name="numbers">4 single numbers (0-9).</param>
        /// <param name="dots">3 bool-values for enabling/disabling the dots on the field.</param>
        private void DrawNumbers(int[] numbers, bool[] dots)
        {
            PictureBox[] pictureBoxes = new PictureBox[(4 * 8) - 1]; // 8 token each number, the 4. number does not have a dot
            bool[] pattern;

            // 1. Ziffer
            pictureBoxes[0] = this.ptSaa11;
            pictureBoxes[1] = this.ptSaa12;
            pictureBoxes[2] = this.ptSaa13;
            pictureBoxes[3] = this.ptSaa14;
            pictureBoxes[4] = this.ptSaa15;
            pictureBoxes[5] = this.ptSaa16;
            pictureBoxes[6] = this.ptSaa17;
            pictureBoxes[7] = this.ptSaa18;

            // 2. Ziffer
            pictureBoxes[8] = this.ptSaa21;
            pictureBoxes[9] = this.ptSaa22;
            pictureBoxes[10] = this.ptSaa23;
            pictureBoxes[11] = this.ptSaa24;
            pictureBoxes[12] = this.ptSaa25;
            pictureBoxes[13] = this.ptSaa26;
            pictureBoxes[14] = this.ptSaa27;
            pictureBoxes[15] = this.ptSaa28;

            // 3. Ziffer
            pictureBoxes[16] = this.ptSaa31;
            pictureBoxes[17] = this.ptSaa32;
            pictureBoxes[18] = this.ptSaa33;
            pictureBoxes[19] = this.ptSaa34;
            pictureBoxes[20] = this.ptSaa35;
            pictureBoxes[21] = this.ptSaa36;
            pictureBoxes[22] = this.ptSaa37;
            pictureBoxes[23] = this.ptSaa38;

            // 4. Ziffer
            pictureBoxes[24] = this.ptSaa41;
            pictureBoxes[25] = this.ptSaa42;
            pictureBoxes[26] = this.ptSaa43;
            pictureBoxes[27] = this.ptSaa44;
            pictureBoxes[28] = this.ptSaa45;
            pictureBoxes[29] = this.ptSaa46;
            pictureBoxes[30] = this.ptSaa47;

            this.ActivateAllThePictureBoxes();
            int numberCounter = 1;
            int pictureBoxCounter = 0;
            int amountLines;
            bool withDot = false;

            foreach (int number in numbers)
            {
                if (numberCounter <= 3)
                {
                    if (dots[numberCounter - 1])
                    {
                        withDot = true;
                    }
                    else
                    {
                        withDot = false;
                    }
                }

                pattern = this.GetNumberPattern(number, withDot);
                amountLines = (numberCounter < 4) ? 8 : 7;

                for (int i = 0; i < amountLines; i++)
                {
                    if (!pattern[i])
                    {
                        pictureBoxes[pictureBoxCounter].Image = null;
                    }

                    pictureBoxCounter++;
                }

                numberCounter++;
            }
        }

        /// <summary>
        /// Gets the pattern of a single number for the field for I2C-4DLED-SAA1064.
        /// </summary>
        /// <param name="number">Number to parse.</param>
        /// <param name="isPoint">Activate or deactivate the point/dot.</param>
        /// <returns>Pattern as bool list.</returns>
        private bool[] GetNumberPattern(int number, bool isPoint)
        {
            bool[] numberPattern;

            switch (number)
            {
                case 0:
                    numberPattern = new bool[8] { true, true, true, true, true, true, false, isPoint };
                    break;
                case 1:
                    numberPattern = new bool[8] { false, true, true, false, false, false, false, isPoint };
                    break;
                case 2:
                    numberPattern = new bool[8] { true, true, false, true, true, false, true, isPoint };
                    break;
                case 3:
                    numberPattern = new bool[8] { true, true, true, true, false, false, true, isPoint };
                    break;
                case 4:
                    numberPattern = new bool[8] { false, true, true, false, false, true, true, isPoint };
                    break;
                case 5:
                    numberPattern = new bool[8] { true, false, true, true, false, true, true, isPoint };
                    break;
                case 6:
                    numberPattern = new bool[8] { true, false, true, true, true, true, true, isPoint };
                    break;
                case 7:
                    numberPattern = new bool[8] { true, true, true, false, false, false, false, isPoint };
                    break;
                case 8:
                    numberPattern = new bool[8] { true, true, true, true, true, true, true, isPoint };
                    break;
                case 9:
                    numberPattern = new bool[8] { true, true, true, true, false, true, true, isPoint };
                    break;
                default:
                    throw new ArgumentException();
            }

            return numberPattern;
        }

        /// <summary>
        /// Activates the whole field for I2C-4DLED-SAA1064.
        /// </summary>
        private void ActivateAllThePictureBoxes()
        {
            this.ptSaa11.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa12.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa13.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa14.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa15.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa16.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa17.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa18.Image = Eq3.misc.USBI2C.Properties.Resources.PunktWeiss;

            // 2. Ziffer
            this.ptSaa21.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa22.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa23.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa24.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa25.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa26.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa27.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa28.Image = Eq3.misc.USBI2C.Properties.Resources.PunktWeiss;

            // 3. Ziffer
            this.ptSaa31.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa32.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa33.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa34.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa35.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa36.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa37.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa38.Image = Eq3.misc.USBI2C.Properties.Resources.PunktWeiss;

            // 4. Ziffer
            this.ptSaa41.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa42.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa43.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa44.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
            this.ptSaa45.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa46.Image = Eq3.misc.USBI2C.Properties.Resources.NummerVertikalWeiss;
            this.ptSaa47.Image = Eq3.misc.USBI2C.Properties.Resources.NummerHorizontalWeiss;
        }

        /// <summary>
        /// Shows the chosen numbers on the field for I2C-4DLED-SAA1064.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtSaa1064CommitNumbers_Click(object sender, EventArgs e)
        {
            string uncheckedNumbers = this.tbSaa1064CustomNumbers.Text;
            Regex regex = new Regex("^[0-9]{4}$");

            if (!regex.IsMatch(uncheckedNumbers))
            {
                this.tbSaa1064CustomNumbers.BackColor = Color.Red;
                return;
            }

            this.tbSaa1064CustomNumbers.BackColor = Color.White;
            int[] numbers = new int[4];
            for (int i = 0; i < uncheckedNumbers.Length; i++)
            {
                numbers[i] = Convert.ToInt32(uncheckedNumbers[i].ToString());
            }

            bool[] dots = new bool[3];
            for (int i = 0; i < dots.Length; i++)
            {
                dots[i] = false;
            }

            this.DrawNumbers(numbers, dots);
        }

        /// <summary>
        /// Gets the temperature from MCP9801 and shows it on the field for I2C-4DLED-SAA1064.
        /// </summary>
        private void DrawTemperature()
        {
            int intTemperature;
            if (!this.connectedWithMCP9801)
            {
                this.BtMCPTrigger_Click(null, null);
                return;
            }

            intTemperature = (int)(Math.Round(this.GetTemperatureInRightMeasure(), 2) * 100);
            int[] numbers = new int[4];
            bool[] dots = new bool[3];

            if (intTemperature > 10000)
            {
                intTemperature = intTemperature /= 10;

                dots[0] = false;
                dots[1] = false;
                dots[2] = true;
            }
            else
            {
                dots[0] = false;
                dots[1] = true;
                dots[2] = false;
            }

            int counter = 3;
            while (intTemperature != 0)
            {
                numbers[counter] = intTemperature % 10;
                intTemperature /= 10;
                counter--;
            }

            this.DrawNumbers(numbers, dots);
        }

        #endregion
        #region 4DLED-MCP9801
        /// <summary>
        /// Click-function of button for triggering of 4DLED-SAA1064.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void BtMCPTrigger_Click(object sender, EventArgs e)
        {
            if (this.rdMCP9801TriggerOne.Checked)
            {
                this.TriggerMcp9801(null, null);
            }
            else
            {
                if (this.timerTriggerMcp9801 == null || !this.timerTriggerMcp9801.Enabled)
                {
                    int intervall = this.GetIntervallForTimerTrigger(false);
                    if (intervall == -1)
                    {
                        return;
                    }

                    this.timerTriggerMcp9801 = new Timer();
                    this.timerTriggerMcp9801.Interval = intervall;
                    this.timerTriggerMcp9801.Tick += new EventHandler(this.TriggerMcp9801);
                    this.timerTriggerMcp9801.Start();

                    this.btMCPTrigger.Text = "Stopp";
                    this.tbMCP9801TriggerIntervall.Enabled = false;
                    this.rdMCP9801TriggerOne.Enabled = false;
                    this.rdMCP9801TriggerIntervall.Enabled = false;
                }
                else
                {
                    this.timerTriggerMcp9801.Stop();

                    this.btMCPTrigger.Text = "Senden";
                    this.tbMCP9801TriggerIntervall.Enabled = true;
                    this.rdMCP9801TriggerOne.Enabled = true;
                    this.rdMCP9801TriggerIntervall.Enabled = true;
                }
            }
        }

        /// <summary>
        /// Triggers 4DLED-MCP9801.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TriggerMcp9801(object sender, EventArgs e)
        {
            this.SetDataMcp9801();
            this.I2C4DLed.Mcp9801.SendCommand();

            if (this.I2C4DLed.Mcp9801.TemperatureCelsius != MCP9801.ErrorTemperature)
            {
                this.temperatureCelsius = this.I2C4DLed.Mcp9801.TemperatureCelsius;
                this.tbMCPShowTemperature.Text = this.GetTemperatureInRightMeasureWithFormat();
                this.connectedWithMCP9801 = true;

                if (this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex == 2)
                {
                    this.DrawTemperature();
                }
            }

            this.alert = this.I2C4DLed.Mcp9801.Alert;
            this.ShowAlert();
        }

        /// <summary>
        /// Sets the current settings of I2C-4DLED-MCP9801.
        /// </summary>
        private void SetDataMcp9801()
        {
            MCP9801.UiElements uiElements = new MCP9801.UiElements();
            uiElements.ComboBoxMCP9801SlaveAddress = this.comboBoxMCP9801SlaveAddress;
            uiElements.SettingsByte = this.Mcp9801GetSettings();
            uiElements.TbMCPAlarmValue = this.tbMCPAlarmValue;
            uiElements.TbMCPAlarmValueWrite = this.tbMCPAlarmValueWrite;
            uiElements.RdMCPCelsius = this.rdMCPCelsius;
            uiElements.RdMCPFahrenheit = this.rdMCPFahrenheit;
            uiElements.LbMCPAlarmValueError = this.lbMCPAlarmValueError;
            uiElements.ListBoxOutput = this.listBoxOutput;
            uiElements.ListBoxInput = this.listBoxInput;
            this.I2C4DLed.Mcp9801.ImportUiElements(uiElements);
        }

        /// <summary>
        /// Gets the current measured temperature in the right measure unit.
        /// </summary>
        /// <returns>Current measured temperature in the right measure unit.</returns>
        private double GetTemperatureInRightMeasure()
        {
            this.CalculateTemperatures();

            if (this.rdMCPCelsius.Checked)
            {
                return this.temperatureCelsius;
            }
            else if (this.rdMCPFahrenheit.Checked)
            {
                return this.temperatureFahrenheit;
            }
            else
            {
                return this.temperatureKelvin;
            }
        }

        /// <summary>
        /// Formats the temperature.
        /// </summary>
        /// <returns>Formatted temperature.</returns>
        private string GetTemperatureInRightMeasureWithFormat()
        {
            return String.Format("{0:0.00} {1}", this.GetTemperatureInRightMeasure(), this.GetMeasureUnit());
        }

        /// <summary>
        /// Gets the current measure unit from gui.
        /// </summary>
        /// <returns>Current measure unit.</returns>
        private string GetMeasureUnit()
        {
            if (this.rdMCPCelsius.Checked)
            {
                return "°C";
            }
            else if (this.rdMCPFahrenheit.Checked)
            {
                return "°F";
            }
            else
            {
                return "K";
            }
        }

        /// <summary>
        /// Calculates the current temperature from Celsius to Fahrenheit and Kelvin.
        /// </summary>
        private void CalculateTemperatures()
        {
            this.temperatureFahrenheit = TemperatureUtils.ConvertCelsiusToFahrenheit(this.temperatureCelsius);
            this.temperatureKelvin = TemperatureUtils.ConvertCelsiusToKelvin(this.temperatureCelsius);
        }

        /// <summary>
        /// Shows current temperature in new wished measure unit.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TemperatureSelectionChanged(object sender, EventArgs e)
        {
            string measureUnit = this.GetMeasureUnit();
            this.lbMCPCurrentAlarmUnit.Text = measureUnit;
            this.lbMCPWantedAlarmUnit.Text = measureUnit;

            if (this.connectedWithMCP9801)
            {
                this.tbMCPShowTemperature.Text = this.GetTemperatureInRightMeasureWithFormat();
                this.ShowAlert();

                if (this.comboBoxSaa1064NumbersPatternChoice.SelectedIndex == 2)
                {
                    this.DrawTemperature();
                }
            }
        }

        /// <summary>
        /// Shows a error if a wished alarm value cannot be written to the device.
        /// </summary>
        private void ShowErrorTemperaturNotInRange()
        {
            MessageBox.Show("Der zu schreibende Alarmwert muss zwischen -127,5°C und +127,5°C liegen (0,5-Schritte, eine Nachkommastelle).", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        /// <summary>
        /// Displays the measured temperature on the alert of I2C-4DLED-MCP9801.
        /// </summary>
        private void ShowAlert()
        {
            double alertValueInRightMeasureUnit;

            switch (this.GetMeasureUnit())
            {
                case "°C":
                    alertValueInRightMeasureUnit = this.alert;
                    break;
                case "°F":
                    alertValueInRightMeasureUnit = TemperatureUtils.ConvertCelsiusToFahrenheit(this.alert);
                    break;
                case "K":
                    alertValueInRightMeasureUnit = TemperatureUtils.ConvertCelsiusToKelvin(this.alert);
                    break;
                default:
                    alertValueInRightMeasureUnit = 0.0;
                    break;
            }

            this.tbMCPAlarmValue.Text = string.Format("{0:0.0}", alertValueInRightMeasureUnit);
        }

        /// <summary>
        /// Gets the current settings for I2C-4DLED-MCP9801 from gui.
        /// </summary>
        /// <returns>Current settings.</returns>
        private byte Mcp9801GetSettings()
        {
            byte settings = 0x00;

            settings += (byte)(this.cbMCP9801OneShot.Checked ? 0x80 : 0x00);
            settings += this.GetAdcResolutionToSettingsByte();
            settings += this.GetFaultQueueToSettingsByte();

            settings += (byte)(this.cbMCP9801AlertPolarity.Checked ? 0x04 : 0x00);
            settings += (byte)(this.cbMCP9801CompInt.Checked ? 0x02 : 0x00);
            settings += (byte)(this.cbMCP9801Shutdown.Checked ? 0x01 : 0x00);

            return settings;
        }

        /// <summary>
        /// Gets the current settings for I2C-4DLED-MCP9801 from gui.
        /// </summary>
        /// <returns>Current settings.</returns>
        private byte GetFaultQueueToSettingsByte()
        {
            int faultQueueChoice = this.comboBoxMCP9801FaultQueue.SelectedIndex;
            switch (faultQueueChoice)
            {
                case 0:
                    return (0x10 * 0) + (0x08 * 0);
                case 1:
                    return (0x10 * 0) + (0x08 * 1);
                case 2:
                    return (0x10 * 1) + (0x08 * 0);
                case 3:
                    return (0x10 * 1) + (0x08 * 1);
                default:
                    throw new InvalidOperationException();
            }
        }

        /// <summary>
        /// Gets adc resolution from gui and saves it in a byte for a register.
        /// </summary>
        /// <returns>Adc resolution as byte.</returns>
        private byte GetAdcResolutionToSettingsByte()
        {
            int adcResolutioChoice = this.comboBoxMCP9801ADCResolution.SelectedIndex;
            switch (adcResolutioChoice)
            {
                case 0:
                    return (0x40 * 0) + (0x20 * 0);
                case 1:
                    return (0x40 * 0) + (0x20 * 1);
                case 2:
                    return (0x40 * 1) + (0x20 * 0);
                case 3:
                    return (0x40 * 1) + (0x20 * 1);
                default:
                    throw new InvalidOperationException();
            }
        }

        /// <summary>
        /// Toggles possibility of input a custom intervall.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void RdSaa1064TriggerIntervall_CheckedChanged(object sender, EventArgs e)
        {
            if (this.rdSaa1064TriggerIntervall.Checked)
            {
                this.tbSaa1064TriggerIntervall.Enabled = true;
            }
            else
            {
                this.tbSaa1064TriggerIntervall.Enabled = false;
                this.tbSaa1064TriggerIntervall.BackColor = Color.White;
            }
        }

        /// <summary>
        /// Toggles possibility of input a custom intervall.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void RdMCP9801TriggerIntervall_CheckedChanged(object sender, EventArgs e)
        {
            if (this.rdMCP9801TriggerIntervall.Checked)
            {
                this.tbMCP9801TriggerIntervall.Enabled = true;
            }
            else
            {
                this.tbMCP9801TriggerIntervall.Enabled = false;
                this.tbMCP9801TriggerIntervall.BackColor = Color.White;
            }
        }

        #endregion
        #endregion

        #region 6D-BS
        #region Linear acceleration sensor
        /// <summary>
        /// Sets general settings of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsLaSetSettings_Click(object sender, EventArgs e)
        {
            try
            {
                this.SetDataRateI2C6DBSLa();
                this.SetFilterI2C6DBSLa();
                this.SetScaleI2C6DBSLa();
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht gesetzt werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Sets slave address of I2C-6DBS-LA (click function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void SelectSlaveAddressLa_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.SetSlaveAdressI2C6DBSLa();
        }

        /// <summary>
        /// Sets slave address of I2C-6DBS-LA.
        /// </summary>
        private void SetSlaveAdressI2C6DBSLa()
        {
            LinearAccelerationSensor.SlaveAdress slaveAdress;
            switch (this.comboBox6DbsLaSelectSlaveAdress.SelectedIndex)
            {
                case 0:
                    slaveAdress = LinearAccelerationSensor.SlaveAdress.x30x31;
                    break;
                case 1:
                    slaveAdress = LinearAccelerationSensor.SlaveAdress.x32x33;
                    break;
                default:
                    throw new ArgumentException();
            }

            if (this.I2C6Dbs != null)
            {
                this.I2C6Dbs.LinearAcceleration.SetSlaveAdress(slaveAdress);
            }
        }

        /// <summary>
        /// Sets data rate of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        private void SetDataRateI2C6DBSLa()
        {
            LinearAccelerationSensor.DataRate dataRate;
            switch (this.comboBox6DbsLaSelectDataRate.SelectedIndex)
            {
                case 0:
                    dataRate = LinearAccelerationSensor.DataRate.PowerDownMode;
                    break;
                case 1:
                    dataRate = LinearAccelerationSensor.DataRate.Hz1;
                    break;
                case 2:
                    dataRate = LinearAccelerationSensor.DataRate.Hz10;
                    break;
                case 3:
                    dataRate = LinearAccelerationSensor.DataRate.Hz25;
                    break;
                case 4:
                    dataRate = LinearAccelerationSensor.DataRate.Hz50;
                    break;
                case 5:
                    dataRate = LinearAccelerationSensor.DataRate.Hz100;
                    break;
                case 6:
                    dataRate = LinearAccelerationSensor.DataRate.Hz200;
                    break;
                case 7:
                    dataRate = LinearAccelerationSensor.DataRate.Hz400;
                    break;
                case 8:
                    dataRate = LinearAccelerationSensor.DataRate.LowPowerMode;
                    break;
                case 9:
                    dataRate = LinearAccelerationSensor.DataRate.KHz1344;
                    break;
                default:
                    throw new ArgumentException();
            }

            bool enablePowerDownMode = this.cb6DbsLaEnablePowerDownMode.Checked;
            this.I2C6Dbs.LinearAcceleration.SetDataRate(dataRate, enablePowerDownMode);
        }

        /// <summary>
        /// Sets filter mode of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        private void SetFilterI2C6DBSLa()
        {
            const bool EnableFilterForClickFunction = false;
            const bool EnableFilterForAoiFunction1 = false;
            const bool EnableFilterForAoiFunction2 = false;

            I2C6DBS.FilterMode filterMode;
            switch (this.comboBox6DbsLaSelectFilterMode.SelectedIndex)
            {
                case 0:
                    filterMode = I2C6DBS.FilterMode.NormalModeWithReset;
                    break;
                case 1:
                    filterMode = I2C6DBS.FilterMode.ReferenceSignal;
                    break;
                case 2:
                    filterMode = I2C6DBS.FilterMode.NormalMode;
                    break;
                case 3:
                    filterMode = I2C6DBS.FilterMode.AutoresetOnInterruptEvent;
                    break;
                default:
                    throw new ArgumentException();
            }

            bool cutOffFrequency1 = false;
            bool cutOffFrequency2 = false;
            switch (this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex)
            {
                case 0:
                    break;
                case 1:
                    cutOffFrequency2 = true;
                    break;
                case 2:
                    cutOffFrequency1 = true;
                    break;
                case 3:
                    cutOffFrequency1 = true;
                    cutOffFrequency2 = true;
                    break;
            }

            this.I2C6Dbs.LinearAcceleration.SetFilter(
                filterMode,
                cutOffFrequency1,
                cutOffFrequency2,
                this.cb6DbsLaEnableFilteredDataSelection.Checked,
                EnableFilterForClickFunction,
                EnableFilterForAoiFunction1,
                EnableFilterForAoiFunction2);
        }

        /// <summary>
        /// Sets scale settings of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        private void SetScaleI2C6DBSLa()
        {
            I2C6DBS.EndianSelection endian;
            LinearAccelerationSensor.FullScale fullScale;
            LinearAccelerationSensor.SpiMode spiMode;

            switch (this.comboBox6DbsLaSelectEndianMode.SelectedIndex)
            {
                case 0:
                    endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                    break;
                case 1:
                    endian = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
                    break;
                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsLaFullScale.SelectedIndex)
            {
                case 0:
                    fullScale = LinearAccelerationSensor.FullScale.TwoG;
                    break;
                case 1:
                    fullScale = LinearAccelerationSensor.FullScale.FourG;
                    break;
                case 2:
                    fullScale = LinearAccelerationSensor.FullScale.EightG;
                    break;
                case 3:
                    fullScale = LinearAccelerationSensor.FullScale.SixteenG;
                    break;
                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsLaSelectSpiMode.SelectedIndex)
            {
                case 0:
                    spiMode = LinearAccelerationSensor.SpiMode.ThreeWireInterface;
                    break;
                case 1:
                    spiMode = LinearAccelerationSensor.SpiMode.FourWireInterface;
                    break;
                default:
                    throw new ArgumentException();
            }

            this.I2C6Dbs.LinearAcceleration.SetScale(endian, fullScale, spiMode, this.cb6DbsLaEnableHighResolutionOutputMode.Checked);
        }

        /// <summary>
        /// Reads out settings of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsLaReadSettings_Click(object sender, EventArgs e)
        {
            try
            {
                LinearAccelerationSensor.DataRate dataRate;
                bool enablePowerDownMode;
                this.I2C6Dbs.LinearAcceleration.GetBasicSettings(out dataRate, out enablePowerDownMode);
                this.I2C6DBSLaSetBasicSettingsGui(dataRate, enablePowerDownMode);

                I2C6DBS.FilterMode filterMode;
                bool cutOffFrequency1;
                bool cutOffFrequency2;
                bool filteredDataSelection;
                this.I2C6Dbs.LinearAcceleration.GetFilter(out filterMode, out cutOffFrequency1, out cutOffFrequency2, out filteredDataSelection);
                this.I2C6DBSLaSetFilterGui(filterMode, cutOffFrequency1, cutOffFrequency2, filteredDataSelection);

                I2C6DBS.EndianSelection endian;
                LinearAccelerationSensor.FullScale fullScale;
                bool highResolutionOutputMode;
                LinearAccelerationSensor.SpiMode spiMode;
                this.I2C6Dbs.LinearAcceleration.GetScale(out endian, out fullScale, out highResolutionOutputMode, out spiMode);
                this.I2C6DBSLaSetScaleGui(endian, fullScale, highResolutionOutputMode, spiMode);
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht ausgelesen werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Sets basic settings of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="dataRate">Chosen data rate.</param>
        /// <param name="enablePowerDownMode">Enable or disable power down mode.</param>
        private void I2C6DBSLaSetBasicSettingsGui(LinearAccelerationSensor.DataRate dataRate, bool enablePowerDownMode)
        {
            switch (dataRate)
            {
                case LinearAccelerationSensor.DataRate.PowerDownMode:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 0;
                    break;
                case LinearAccelerationSensor.DataRate.Hz1:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 1;
                    break;
                case LinearAccelerationSensor.DataRate.Hz10:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 2;
                    break;
                case LinearAccelerationSensor.DataRate.Hz25:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 3;
                    break;
                case LinearAccelerationSensor.DataRate.Hz50:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 4;
                    break;
                case LinearAccelerationSensor.DataRate.Hz100:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 5;
                    break;
                case LinearAccelerationSensor.DataRate.Hz200:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 6;
                    break;
                case LinearAccelerationSensor.DataRate.Hz400:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 7;
                    break;
                case LinearAccelerationSensor.DataRate.LowPowerMode:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 8;
                    break;
                case LinearAccelerationSensor.DataRate.KHz1344:
                    this.comboBox6DbsLaSelectDataRate.SelectedIndex = 9;
                    break;
                default:
                    break;
            }

            this.cb6DbsLaEnablePowerDownMode.Checked = enablePowerDownMode;
        }

        /// <summary>
        /// Set filter gui of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="filterMode">Chosen filter mode.</param>
        /// <param name="cutOffFrequency1">Enable cut off frequency 1.</param>
        /// <param name="cutOffFrequency2">Enable cut off frequency 2.</param>
        /// <param name="filteredDataSelection">Chosen filtered data selection.</param>
        private void I2C6DBSLaSetFilterGui(I2C6DBS.FilterMode filterMode, bool cutOffFrequency1, bool cutOffFrequency2, bool filteredDataSelection)
        {
            switch (filterMode)
            {
                case I2C6DBS.FilterMode.NormalModeWithReset:
                    this.comboBox6DbsLaSelectFilterMode.SelectedIndex = 0;
                    break;
                case I2C6DBS.FilterMode.ReferenceSignal:
                    this.comboBox6DbsLaSelectFilterMode.SelectedIndex = 1;
                    break;
                case I2C6DBS.FilterMode.NormalMode:
                    this.comboBox6DbsLaSelectFilterMode.SelectedIndex = 2;
                    break;
                case I2C6DBS.FilterMode.AutoresetOnInterruptEvent:
                    this.comboBox6DbsLaSelectFilterMode.SelectedIndex = 3;
                    break;
                default:
                    break;
            }

            if (!cutOffFrequency1 && !cutOffFrequency2)
            {
                this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex = 0;
            }
            else if (!cutOffFrequency1 && cutOffFrequency2)
            {
                this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex = 2;
            }
            else if (cutOffFrequency1 && !cutOffFrequency2)
            {
                this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex = 1;
            }
            else if (cutOffFrequency1 && cutOffFrequency2)
            {
                this.comboBox6DbsLaSelectCutOffFrequency.SelectedIndex = 3;
            }

            this.cb6DbsLaEnableFilteredDataSelection.Checked = filteredDataSelection;
        }

        /// <summary>
        /// Set scale gui of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="endian">Chosen endian.</param>
        /// <param name="fullScale">Chosen full scale.</param>
        /// <param name="highResolutionOutputMode">Enable or disable high resolution mode.</param>
        /// <param name="spiMode">Chosen spi mode.</param>
        private void I2C6DBSLaSetScaleGui(I2C6DBS.EndianSelection endian, LinearAccelerationSensor.FullScale fullScale, bool highResolutionOutputMode, LinearAccelerationSensor.SpiMode spiMode)
        {
            switch (endian)
            {
                case I2C6DBS.EndianSelection.DataLsbAtLowerAddress:
                    this.comboBox6DbsLaSelectEndianMode.SelectedIndex = 0;
                    break;
                case I2C6DBS.EndianSelection.DataMsbAtLowerAddress:
                    this.comboBox6DbsLaSelectEndianMode.SelectedIndex = 1;
                    break;
                default:
                    break;
            }

            switch (fullScale)
            {
                case LinearAccelerationSensor.FullScale.TwoG:
                    this.comboBox6DbsLaFullScale.SelectedIndex = 0;
                    break;
                case LinearAccelerationSensor.FullScale.FourG:
                    this.comboBox6DbsLaFullScale.SelectedIndex = 1;
                    break;
                case LinearAccelerationSensor.FullScale.EightG:
                    this.comboBox6DbsLaFullScale.SelectedIndex = 2;
                    break;
                case LinearAccelerationSensor.FullScale.SixteenG:
                    this.comboBox6DbsLaFullScale.SelectedIndex = 3;
                    break;
                default:
                    break;
            }

            this.cb6DbsLaEnableHighResolutionOutputMode.Checked = highResolutionOutputMode;

            switch (spiMode)
            {
                case LinearAccelerationSensor.SpiMode.ThreeWireInterface:
                    this.comboBox6DbsLaSelectSpiMode.SelectedIndex = 0;
                    break;
                case LinearAccelerationSensor.SpiMode.FourWireInterface:
                    this.comboBox6DbsLaSelectSpiMode.SelectedIndex = 1;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// Reset settings of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsLaResetSettings_Click(object sender, EventArgs e)
        {
            try
            {
                LinearAccelerationSensor.DataRate dataRate = LinearAccelerationSensor.DataRate.KHz1344;
                bool enablePowerDownMode = false;
                this.I2C6Dbs.LinearAcceleration.SetDataRate(dataRate, enablePowerDownMode);
                this.I2C6DBSLaSetBasicSettingsGui(dataRate, enablePowerDownMode);

                I2C6DBS.FilterMode filterMode = I2C6DBS.FilterMode.NormalModeWithReset;
                bool cutOffFrequency1 = false;
                bool cutOffFrequency2 = false;
                bool filteredDataSelection = false;
                this.I2C6Dbs.LinearAcceleration.SetFilter(filterMode, cutOffFrequency1, cutOffFrequency2, filteredDataSelection, false, false, false);
                this.I2C6DBSLaSetFilterGui(filterMode, cutOffFrequency1, cutOffFrequency2, filteredDataSelection);

                I2C6DBS.EndianSelection endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                LinearAccelerationSensor.FullScale fullScale = LinearAccelerationSensor.FullScale.TwoG;
                bool highResolutionOutputMode = false;
                LinearAccelerationSensor.SpiMode spiMode = LinearAccelerationSensor.SpiMode.FourWireInterface;
                this.I2C6Dbs.LinearAcceleration.SetScale(endian, fullScale, spiMode, highResolutionOutputMode);
                this.I2C6DBSLaSetScaleGui(endian, fullScale, highResolutionOutputMode, spiMode);
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht resettet werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Reads status of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6dbsLaReadStatus_Click(object sender, EventArgs e)
        {
            try
            {
                this.SetStatusGuiI2C6DBSLa();
            }
            catch (Exception)
            {
                MessageBox.Show("Der Status konnte nicht gesetzt werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Set status gui of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        private void SetStatusGuiI2C6DBSLa()
        {
            LinearAccelerationSensor.PositionStatus currentStatus = this.I2C6Dbs.LinearAcceleration.Status;

            this.cb6dbsLaOverrunX.Checked = currentStatus.OverrunX;
            this.cb6dbsLaOverrunY.Checked = currentStatus.OverrunY;
            this.cb6dbsLaOverrunZ.Checked = currentStatus.OverrunZ;

            this.cb6dbsLaNewDataX.Checked = currentStatus.NewDataX;
            this.cb6dbsLaNewDataY.Checked = currentStatus.NewDataY;
            this.cb6dbsLaNewDataZ.Checked = currentStatus.NewDataZ;
        }

        /// <summary>
        /// Initialises the chart controls for visualising measures of I2C-6DBS.
        /// </summary>
        private void InitCharts()
        {
            const int MaximalPoints = 50;

            List<Chart> charts = new List<Chart>();
            charts.AddRange(
                new Chart[]
                {
                    this.chart6DbsLaHistory,
                    this.chart6DbsArHistory
                });

            foreach (Chart chart in charts)
            {
                chart.Series.Clear();
                chart.Series.Add("X");
                chart.Series.Add("Y");
                chart.Series.Add("Z");
                foreach (Series item in chart.Series)
                {
                    item.ChartType = SeriesChartType.Line;
                }

                chart.ChartAreas[0].AxisX.MinorGrid.Interval = 5;
                chart.ChartAreas[0].AxisX.Minimum = 0;
                chart.ChartAreas[0].AxisX.Maximum = MaximalPoints;

                chart.ChartAreas[0].AxisY.MinorGrid.Interval = 10000;
                chart.ChartAreas[0].AxisY.Minimum = -25000;
                chart.ChartAreas[0].AxisY.Maximum = 25000;

                foreach (Series series in chart.Series)
                {
                    for (int i = 1; i < MaximalPoints + 1; i++)
                    {
                        series.Points.AddXY(i, 0.0);
                    }
                }

                chart.Series[0].Color = Color.Red;
                chart.Series[1].Color = Color.Green;
                chart.Series[2].Color = Color.Blue;

                foreach (var legend in chart.Legends)
                {
                    legend.BackColor = Color.Transparent;
                }

                foreach (var area in chart.ChartAreas)
                {
                    area.BackColor = Color.Transparent;
                }

                chart.BackColor = Color.Transparent;
            }
        }

        /// <summary>
        /// Toggles refreshing of the chart of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6dbsLaToggleChartRefreshing_Click(object sender, EventArgs e)
        {
            if (this.timerTriggerChart6dbsAr != null && this.timerTriggerChart6dbsAr.Enabled)
            {
                MessageBox.Show("Es kann nur eine automatische Ansteuerung gleichzeitig aktiviert werden (Gyro wird bereits angesteuert).", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (this.timerTriggerChart6dbsLa == null || !this.timerTriggerChart6dbsLa.Enabled)
            {
                int refreshingIntervall;
                try
                {
                    refreshingIntervall = this.GetRefreshingIntervall(this.tb6dbsLaToggleChartRefreshingIntervall);
                }
                catch (ArgumentException ex)
                {
                    MessageBox.Show(ex.Message, "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                this.btn6dbsLaToggleChartRefreshing.Text = "stoppen";
                this.tb6dbsLaToggleChartRefreshingIntervall.Enabled = false;
                this.I2C6Dbs.LinearAcceleration.DataChart = this.chart6DbsLaHistory;

                this.timerTriggerChart6dbsLa = new Timer();
                this.timerTriggerChart6dbsLa.Interval = refreshingIntervall;
                this.timerTriggerChart6dbsLa.Tick += new EventHandler(this.TimerTriggerChart6dbsLa_Tick);
                this.timerTriggerChart6dbsLa.Start();
            }
            else
            {
                this.btn6dbsLaToggleChartRefreshing.Text = "starten";
                this.tb6dbsLaToggleChartRefreshingIntervall.Enabled = true;
                this.timerTriggerChart6dbsLa.Stop();
            }
        }

        /// <summary>
        /// Gets the refreshing intervall of a chart based on a TextBox.
        /// </summary>
        /// <param name="textBox">Textbox which contains the intervall.</param>
        /// <returns>Refreshing intervall.</returns>
        private int GetRefreshingIntervall(TextBox textBox)
        {
            const int MinIntervall = 500; // ms

            int refreshingIntervall;
            try
            {
                refreshingIntervall = int.Parse(textBox.Text);

                if (refreshingIntervall < MinIntervall)
                {
                    throw new Exception();
                }
            }
            catch
            {
                throw new ArgumentException(string.Format("Das Intervall für die automatische Chart-Aktualisierung muss mindestens {0} ms betragen.", MinIntervall));
            }

            return refreshingIntervall;
        }

        /// <summary>
        /// Tick-Method of I2C-6D-BS' chart.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TimerTriggerChart6dbsLa_Tick(object sender, EventArgs e)
        {
            try
            {
                this.I2C6Dbs.LinearAcceleration.UpdateChart();

                this.tb6dbsLaCurrentX.Text = this.I2C6Dbs.LinearAcceleration.X.ToString();
                this.tb6dbsLaCurrentY.Text = this.I2C6Dbs.LinearAcceleration.Y.ToString();
                this.tb6dbsLaCurrentZ.Text = this.I2C6Dbs.LinearAcceleration.Z.ToString();
            }
            catch (Exception)
            {
                this.Btn6dbsLaToggleChartRefreshing_Click(null, null);
                MessageBox.Show("Die aktuellen Bewegungsdaten des Beschleunigungssensor konnten nicht ausgelesen werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Changes endian selection of I2C-6DBS' linear acceleration sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void ChangeSelectedEndian6dbsLa_Changed(object sender, EventArgs e)
        {
            switch (this.comboBox6DbsLaSelectEndianMode.SelectedIndex)
            {
                case 0:
                    this.cb6DbsLaEnableHighResolutionOutputMode.Enabled = true;
                    break;
                case 1:
                    this.cb6DbsLaEnableHighResolutionOutputMode.Enabled = false;
                    this.cb6DbsLaEnableHighResolutionOutputMode.Checked = true;
                    break;
            }
        }
        #endregion
        #region Angular rate sensor / Gyro
        /// <summary>
        /// Sets general settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsArSetSettings_Click(object sender, EventArgs e)
        {
            try
            {
                this.SetSettingsI2C6DBSAr();
                this.SetFilterI2C6DBSAr();
                this.SetScaleI2C6DBSAr();
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht gesetzt werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Sets slave address of I2C-6DBS' angular rate sensor (click function).
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void SelectSlaveAddressAr_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.SetSlaveAdressI2C6DBSAr();
        }

        /// <summary>
        /// Sets slave address of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetSlaveAdressI2C6DBSAr()
        {
            AngularRateSensor.SlaveAdress slaveAdress;
            switch (this.comboBox6DbsArSelectSlaveAdress.SelectedIndex)
            {
                case 0:
                    slaveAdress = AngularRateSensor.SlaveAdress.xD4xD5;
                    break;
                case 1:
                    slaveAdress = AngularRateSensor.SlaveAdress.xD6xD7;
                    break;
                default:
                    throw new ArgumentException();
            }

            if (this.I2C6Dbs != null)
            {
                this.I2C6Dbs.AngularRate.SetSlaveAdress(slaveAdress);
            }
        }

        /// <summary>
        /// Sets settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetSettingsI2C6DBSAr()
        {
            AngularRateSensor.Odr odr;
            int cutOff;
            AngularRateSensor.Mode mode;

            switch (this.comboBox6DbsArSelectOdr.SelectedIndex)
            {
                case 0:
                case 1:
                    odr = AngularRateSensor.Odr._95;
                    break;

                case 2:
                case 3:
                case 4:
                case 5:
                    odr = AngularRateSensor.Odr._190;
                    break;

                case 6:
                case 7:
                case 8:
                case 9:
                    odr = AngularRateSensor.Odr._380;
                    break;

                case 10:
                case 11:
                case 12:
                case 13:
                    odr = AngularRateSensor.Odr._760;
                    break;

                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsArSelectOdr.SelectedIndex)
            {
                case 0:
                    cutOff = 0;
                    break;

                case 1:
                    cutOff = 1;
                    break;

                default:
                    cutOff = (this.comboBox6DbsArSelectOdr.SelectedIndex - 2) % 4;
                    break;
            }

            mode = this.cb6DbsArEnablePowerDownMode.Checked ? AngularRateSensor.Mode.PowerDownMode : AngularRateSensor.Mode.NormalOrSleepMode;

            this.I2C6Dbs.AngularRate.SetSettings(odr, cutOff, mode);
        }

        /// <summary>
        /// Sets data rate of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetFilterI2C6DBSAr()
        {
            I2C6DBS.FilterMode filterMode;
            switch (this.comboBox6DbsArSelectFilterMode.SelectedIndex)
            {
                case 0:
                    filterMode = I2C6DBS.FilterMode.NormalModeWithReset;
                    break;
                case 1:
                    filterMode = I2C6DBS.FilterMode.ReferenceSignal;
                    break;
                case 2:
                    filterMode = I2C6DBS.FilterMode.NormalMode;
                    break;
                case 3:
                    filterMode = I2C6DBS.FilterMode.AutoresetOnInterruptEvent;
                    break;
                default:
                    throw new ArgumentException();
            }

            AngularRateSensor.DataFilter dataFilter;
            switch (this.comboBox6DbsArSelectDataFilter.SelectedIndex)
            {
                case 0:
                    dataFilter = AngularRateSensor.DataFilter.Normal;
                    break;
                case 1:
                    dataFilter = AngularRateSensor.DataFilter.Hpf;
                    break;
                case 2:
                    dataFilter = AngularRateSensor.DataFilter.Lpf2;
                    break;
                case 3:
                    dataFilter = AngularRateSensor.DataFilter.HpfLpf2;
                    break;
                default:
                    throw new ArgumentException();
            }

            this.I2C6Dbs.AngularRate.SetFilter(
                this.cb6DbsArEnableEdgeSensitiveTrigger.Checked,
                this.cb6DbsArEnableLevelSensitiveTrigger.Checked,
                filterMode,
                this.comboBox6DbsArSelectCutOffFrequency.SelectedIndex,
                dataFilter);
        }

        /// <summary>
        /// Sets scale settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetScaleI2C6DBSAr()
        {
            AngularRateSensor.BlockDataUpdate blockDataUpdate;
            I2C6DBS.EndianSelection endianSelection;
            AngularRateSensor.FullScale fullScale;
            AngularRateSensor.Sim spiMode;

            switch (this.comboBox6DbsArSelectBlockDataUpdate.SelectedIndex)
            {
                case 0:
                    blockDataUpdate = AngularRateSensor.BlockDataUpdate.ContinuousUpdate;
                    break;
                case 1:
                    blockDataUpdate = AngularRateSensor.BlockDataUpdate.OutputRegistersNotUpdatedUntilMSbAndLSbReading;
                    break;
                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsArSelectEndian.SelectedIndex)
            {
                case 0:
                    endianSelection = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                    break;
                case 1:
                    endianSelection = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
                    break;
                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsArSelectFullScale.SelectedIndex)
            {
                case 0:
                    fullScale = AngularRateSensor.FullScale._250dps;
                    break;
                case 1:
                    fullScale = AngularRateSensor.FullScale._500dps;
                    break;
                case 2:
                    fullScale = AngularRateSensor.FullScale._2000dps;
                    break;
                default:
                    throw new ArgumentException();
            }

            switch (this.comboBox6DbsArSelectSpiMode.SelectedIndex)
            {
                case 0:
                    spiMode = AngularRateSensor.Sim.ThreeWireReadModeDisabled;
                    break;
                case 1:
                    spiMode = AngularRateSensor.Sim.ThreeWireReadModeEnabled;
                    break;
                default:
                    throw new ArgumentException();
            }

            this.I2C6Dbs.AngularRate.SetScale(blockDataUpdate, endianSelection, fullScale, spiMode);
        }

        /// <summary>
        /// Reads out settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsArReadSettings_Click(object sender, EventArgs e)
        {
            try
            {
                AngularRateSensor.Odr odr;
                int bandwithIndex;
                bool enablePowerDownMode;
                this.I2C6Dbs.AngularRate.GetBasicSettings(out odr, out bandwithIndex, out enablePowerDownMode);
                this.I2C6DBSArSetBasicSettingsGui(odr, bandwithIndex, enablePowerDownMode);

                I2C6DBS.EndianSelection endian;
                AngularRateSensor.FullScale fullScale;
                AngularRateSensor.BlockDataUpdate blockDataUpdate;
                AngularRateSensor.Sim sim;
                this.I2C6Dbs.AngularRate.GetScale(out endian, out fullScale, out blockDataUpdate, out sim);
                this.I2C6DBSArSetScaleGui(endian, fullScale, blockDataUpdate, sim);

                bool edgeSensitiveTrigger;
                bool levelSensitiveTrigger;
                I2C6DBS.FilterMode filterMode;
                int cutOffFrequencyIndex;
                AngularRateSensor.DataFilter dataFilter;
                this.I2C6Dbs.AngularRate.GetFilter(out edgeSensitiveTrigger, out levelSensitiveTrigger, out filterMode, out cutOffFrequencyIndex, out dataFilter);
                this.I2C6DBSArSetFilterGui(edgeSensitiveTrigger, levelSensitiveTrigger, filterMode, cutOffFrequencyIndex, dataFilter);
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht ausgelesen werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Sets basic settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="odr">Chosen odr.</param>
        /// <param name="bandwithIndex">Chosen bandwith.</param>
        /// <param name="enablePowerDownMode">Enable or disable power down mode.</param>
        private void I2C6DBSArSetBasicSettingsGui(AngularRateSensor.Odr odr, int bandwithIndex, bool enablePowerDownMode)
        {
            int odrIndex;
            switch (odr)
            {
                case AngularRateSensor.Odr._95:
                    odrIndex = 0;
                    break;
                case AngularRateSensor.Odr._190:
                    odrIndex = 1;
                    break;
                case AngularRateSensor.Odr._380:
                    odrIndex = 2;
                    break;
                case AngularRateSensor.Odr._760:
                    odrIndex = 3;
                    break;
                default:
                    throw new ArgumentException();
            }

            odrIndex *= 4;
            odrIndex += bandwithIndex;

            switch (odrIndex)
            {
                case 0:
                    this.comboBox6DbsArSelectOdr.SelectedIndex = 0;
                    break;

                case 1:
                case 2:
                case 3:
                    this.comboBox6DbsArSelectOdr.SelectedIndex = 1;
                    break;

                case 4:
                    this.comboBox6DbsArSelectOdr.SelectedIndex = odrIndex - 3;
                    break;

                default:
                    this.comboBox6DbsArSelectOdr.SelectedIndex = odrIndex - 2;
                    break;
            }

            this.cb6DbsArEnablePowerDownMode.Checked = enablePowerDownMode;
        }

        /// <summary>
        /// Sets scale of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="endian">Chosen endian.</param>
        /// <param name="fullScale">Chosen full scale.</param>
        /// <param name="blockDataUpdate">Chosen block data update.</param>
        /// <param name="sim">Chosen sim.</param>
        private void I2C6DBSArSetScaleGui(I2C6DBS.EndianSelection endian, AngularRateSensor.FullScale fullScale, AngularRateSensor.BlockDataUpdate blockDataUpdate, AngularRateSensor.Sim sim)
        {
            switch (endian)
            {
                case I2C6DBS.EndianSelection.DataLsbAtLowerAddress:
                    this.comboBox6DbsArSelectEndian.SelectedIndex = 0;
                    break;
                case I2C6DBS.EndianSelection.DataMsbAtLowerAddress:
                    this.comboBox6DbsArSelectEndian.SelectedIndex = 1;
                    break;
                default:
                    break;
            }

            switch (fullScale)
            {
                case AngularRateSensor.FullScale._250dps:
                    this.comboBox6DbsArSelectFullScale.SelectedIndex = 0;
                    break;
                case AngularRateSensor.FullScale._500dps:
                    this.comboBox6DbsArSelectFullScale.SelectedIndex = 1;
                    break;
                case AngularRateSensor.FullScale._1000dps:
                    break;
                case AngularRateSensor.FullScale._2000dps:
                    this.comboBox6DbsArSelectFullScale.SelectedIndex = 2;
                    break;
                default:
                    break;
            }

            switch (blockDataUpdate)
            {
                case AngularRateSensor.BlockDataUpdate.ContinuousUpdate:
                    this.comboBox6DbsArSelectBlockDataUpdate.SelectedIndex = 0;
                    break;
                case AngularRateSensor.BlockDataUpdate.OutputRegistersNotUpdatedUntilMSbAndLSbReading:
                    this.comboBox6DbsArSelectBlockDataUpdate.SelectedIndex = 1;
                    break;
                default:
                    break;
            }

            switch (sim)
            {
                case AngularRateSensor.Sim.ThreeWireReadModeDisabled:
                    this.comboBox6DbsArSelectSpiMode.SelectedIndex = 0;
                    break;
                case AngularRateSensor.Sim.ThreeWireReadModeEnabled:
                    this.comboBox6DbsArSelectSpiMode.SelectedIndex = 1;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// Sets filter of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="edgeSensitiveTrigger">Enable or disable edge sensitive trigger.</param>
        /// <param name="levelSensitiveTrigger">Enable or disable level sensitive trigger.</param>
        /// <param name="filterMode">Chosen filter mode.</param>
        /// <param name="cutOffFrequencyIndex">Chosen cut-off frequency.</param>
        /// <param name="dataFilter">Chosen data filter.</param>
        private void I2C6DBSArSetFilterGui(bool edgeSensitiveTrigger, bool levelSensitiveTrigger, I2C6DBS.FilterMode filterMode, int cutOffFrequencyIndex, AngularRateSensor.DataFilter dataFilter)
        {
            this.cb6DbsArEnableEdgeSensitiveTrigger.Checked = edgeSensitiveTrigger;
            this.cb6DbsArEnableLevelSensitiveTrigger.Checked = levelSensitiveTrigger;

            switch (filterMode)
            {
                case I2C6DBS.FilterMode.NormalModeWithReset:
                    this.comboBox6DbsArSelectFilterMode.SelectedIndex = 0;
                    break;
                case I2C6DBS.FilterMode.ReferenceSignal:
                    this.comboBox6DbsArSelectFilterMode.SelectedIndex = 1;
                    break;
                case I2C6DBS.FilterMode.NormalMode:
                    this.comboBox6DbsArSelectFilterMode.SelectedIndex = 2;
                    break;
                case I2C6DBS.FilterMode.AutoresetOnInterruptEvent:
                    this.comboBox6DbsArSelectFilterMode.SelectedIndex = 3;
                    break;
                default:
                    break;
            }

            this.comboBox6DbsArSelectCutOffFrequency.SelectedIndex = cutOffFrequencyIndex;

            switch (dataFilter)
            {
                case AngularRateSensor.DataFilter.Normal:
                    this.comboBox6DbsArSelectDataFilter.SelectedIndex = 0;
                    break;
                case AngularRateSensor.DataFilter.Hpf:
                    this.comboBox6DbsArSelectDataFilter.SelectedIndex = 1;
                    break;
                case AngularRateSensor.DataFilter.Lpf2:
                    this.comboBox6DbsArSelectDataFilter.SelectedIndex = 2;
                    break;
                case AngularRateSensor.DataFilter.HpfLpf2:
                    this.comboBox6DbsArSelectDataFilter.SelectedIndex = 3;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// Resets the settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6DbsArResetSettings_Click(object sender, EventArgs e)
        {
            try
            {
                AngularRateSensor.Odr odr = AngularRateSensor.Odr._95;
                int bandwithIndex = 0;
                bool enablePowerDownMode = false;
                AngularRateSensor.Mode mode = enablePowerDownMode ? AngularRateSensor.Mode.PowerDownMode : AngularRateSensor.Mode.NormalOrSleepMode;
                this.I2C6Dbs.AngularRate.SetSettings(odr, bandwithIndex, mode);
                this.I2C6DBSArSetBasicSettingsGui(odr, bandwithIndex, enablePowerDownMode);

                I2C6DBS.EndianSelection endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                AngularRateSensor.FullScale fullScale = AngularRateSensor.FullScale._250dps;
                AngularRateSensor.BlockDataUpdate blockDataUpdate = AngularRateSensor.BlockDataUpdate.ContinuousUpdate;
                AngularRateSensor.Sim sim = AngularRateSensor.Sim.ThreeWireReadModeDisabled;
                this.I2C6Dbs.AngularRate.SetScale(blockDataUpdate, endian, fullScale, sim);
                this.I2C6DBSArSetScaleGui(endian, fullScale, blockDataUpdate, sim);

                bool edgeSensitiveTrigger = false;
                bool levelSensitiveTrigger = false;
                I2C6DBS.FilterMode filterMode = I2C6DBS.FilterMode.NormalModeWithReset;
                int cutOffFrequencyIndex = 0;
                AngularRateSensor.DataFilter dataFilter = AngularRateSensor.DataFilter.Normal;
                this.I2C6DBSArSetFilterGui(edgeSensitiveTrigger, levelSensitiveTrigger, filterMode, cutOffFrequencyIndex, dataFilter);
                this.I2C6Dbs.AngularRate.SetFilter(edgeSensitiveTrigger, levelSensitiveTrigger, filterMode, cutOffFrequencyIndex, dataFilter);
            }
            catch (Exception)
            {
                MessageBox.Show("Die Einstellungen konnten nicht resettet werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Reads out settings of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6dbsArReadStatus_Click(object sender, EventArgs e)
        {
            try
            {
                this.SetStatusGuiI2C6DBSAr();
                this.SetChipIdGuiI2C6DBSAr();
                this.SetTemperatureGuiI2C6DBSAr();
            }
            catch (Exception)
            {
                MessageBox.Show("Der Status konnte nicht ausgelesen werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Sets status gui of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetStatusGuiI2C6DBSAr()
        {
            AngularRateSensor.PositionStatus currentStatus = this.I2C6Dbs.AngularRate.Status;

            this.cb6dbsArOverrunX.Checked = currentStatus.OverrunX;
            this.cb6dbsArOverrunY.Checked = currentStatus.OverrunY;
            this.cb6dbsArOverrunZ.Checked = currentStatus.OverrunZ;

            this.cb6dbsArNewDataX.Checked = currentStatus.NewDataX;
            this.cb6dbsArNewDataY.Checked = currentStatus.NewDataY;
            this.cb6dbsArNewDataZ.Checked = currentStatus.NewDataZ;
        }

        /// <summary>
        /// Sets chip id gui of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetChipIdGuiI2C6DBSAr()
        {
            this.lb6dbsArChipID.Text = this.I2C6Dbs.AngularRate.ChipID.ToString("X2");
        }

        /// <summary>
        /// Sets temperature gui of I2C-6DBS' angular rate sensor.
        /// </summary>
        private void SetTemperatureGuiI2C6DBSAr()
        {
            this.lb6dbsArTemperature.Text = string.Format("{0} °C", this.I2C6Dbs.AngularRate.Temperature);
        }

        /// <summary>
        /// Toggles refreshing of the chart of I2C-6DBS' angular rate sensor.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void Btn6dbsArToggleChartRefreshing_Click(object sender, EventArgs e)
        {
            if (this.timerTriggerChart6dbsLa != null && this.timerTriggerChart6dbsLa.Enabled)
            {
                MessageBox.Show("Es kann nur eine automatische Ansteuerung gleichzeitig aktiviert werden (Beschleunigungssensor wird bereits angesteuert).", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (this.timerTriggerChart6dbsAr == null || !this.timerTriggerChart6dbsAr.Enabled)
            {
                int refreshingIntervall;
                try
                {
                    refreshingIntervall = this.GetRefreshingIntervall(this.tb6dbsArToggleChartRefreshingIntervall);
                }
                catch (ArgumentException ex)
                {
                    MessageBox.Show(ex.Message, "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }

                this.btn6dbsArToggleChartRefreshing.Text = "stoppen";
                this.tb6dbsArToggleChartRefreshingIntervall.Enabled = false;
                this.I2C6Dbs.AngularRate.DataChart = this.chart6DbsArHistory;

                this.timerTriggerChart6dbsAr = new Timer();
                this.timerTriggerChart6dbsAr.Interval = refreshingIntervall;
                this.timerTriggerChart6dbsAr.Tick += new EventHandler(this.TimerTriggerChart6dbsAr_Tick);
                this.timerTriggerChart6dbsAr.Start();
            }
            else
            {
                this.btn6dbsArToggleChartRefreshing.Text = "starten";
                this.tb6dbsArToggleChartRefreshingIntervall.Enabled = true;
                this.timerTriggerChart6dbsAr.Stop();
            }
        }

        /// <summary>
        /// Tick-Method of I2C-6D-BS' chart.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void TimerTriggerChart6dbsAr_Tick(object sender, EventArgs e)
        {
            try
            {
                this.I2C6Dbs.AngularRate.UpdateChart();

                this.tb6dbsArCurrentX.Text = this.I2C6Dbs.AngularRate.X.ToString();
                this.tb6dbsArCurrentY.Text = this.I2C6Dbs.AngularRate.Y.ToString();
                this.tb6dbsArCurrentZ.Text = this.I2C6Dbs.AngularRate.Z.ToString();
            }
            catch (Exception)
            {
                this.Btn6dbsArToggleChartRefreshing_Click(null, null);
                MessageBox.Show("Die aktuellen Bewegungsdaten des Gyros konnten nicht ausgelesen werden.", "Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Refreshes the list of cut-off frequencies if a new ODR-rate was chosen.
        /// </summary>
        /// <param name="sender">Standard parameter sender.</param>
        /// <param name="e">Standard parameter e.</param>
        private void OdrChoiceI2C6DBSAr_SelectedIndexChanged(object sender, EventArgs e)
        {
            this.SetCutOffFrequencyBasedOnI2C6DBSArOdr();
        }

        /// <summary>
        /// Sets the items of the combo box which contains the selectable cut-off frequencies based on the chosen ODR-rate.
        /// </summary>
        private void SetCutOffFrequencyBasedOnI2C6DBSArOdr()
        {
            if (this.odrList == null)
            {
                this.odrList = new List<List<string>>();
                this.odrList.Add(new List<string>(new string[]
                {
                    "7.2",
                    "3.5",
                    "1.8",
                    "0.9",
                    "0.45",
                    "0.18",
                    "0.09",
                    "0.045",
                    "0.018",
                    "0.009"
                }));
                this.odrList.Add(new List<string>(new string[]
                {
                    "13.5",
                    "7.2",
                    "3.5",
                    "1.8",
                    "0.9",
                    "0.45",
                    "0.18",
                    "0.09",
                    "0.045",
                    "0.018",
                }));
                this.odrList.Add(new List<string>(new string[]
                {
                    "27",
                    "13.5",
                    "7.2",
                    "3.5",
                    "1.8",
                    "0.9",
                    "0.45",
                    "0.18",
                    "0.09",
                    "0.045",
                }));
                this.odrList.Add(new List<string>(new string[]
                {
                    "51.4",
                    "27",
                    "13.5",
                    "7.2",
                    "3.5",
                    "1.8",
                    "0.9",
                    "0.45",
                    "0.18",
                    "0.09",
                }));
            }

            this.comboBox6DbsArSelectCutOffFrequency.Items.Clear();
            int odrIndex = this.comboBox6DbsArSelectOdr.SelectedIndex / 4;
            for (int i = 0; i < this.odrList[0].Count; i++)
            {
                this.comboBox6DbsArSelectCutOffFrequency.Items.Add(this.odrList[odrIndex][i]);
            }

            this.comboBox6DbsArSelectCutOffFrequency.SelectedIndex = 0;
        }

        #endregion
        #endregion
    }
}
