﻿//-----------------------------------------------------------------------------
// <copyright file="LCD.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Class-Representation of I2C-LCD
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    /// <summary>
    /// Class-Representation of I2C-LCD.
    /// </summary>
    public class LCD : AbstractDevice
    {
        /// <summary>
        /// Imported UI-Elements of MCP9801.
        /// </summary>
        private UiElements uiElements;

        /// <summary>
        /// Initializes a new instance of the LCD class.
        /// </summary>
        /// <param name="usbI2C">I2C Comport-device.</param>
        public LCD(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.uiElements = new UiElements();
        }

        /// <summary>
        /// Initializes the device.
        /// </summary>
        public void InitDevice()
        {
            string[] commandList = new string[10];
            commandList[0] = "S76 04 00 B5 00 00 00 FF 00 P";
            commandList[1] = "S76 04 00 BD 00 P";
            commandList[2] = "S76 04 00 B3 FF P";
            commandList[3] = "S76 03 00 0A A5 F1 P";
            commandList[4] = "S76 04 00 0B 14 00 P";
            commandList[5] = "S76 04 00 0B 15 00 P";
            commandList[6] = "S76 04 00 0B 16 0C P";
            commandList[7] = "S76 04 00 0B 14 80 P";
            commandList[8] = "S76 04 00 95 0E 08 9F 1F 00 0C 00 P";
            commandList[9] = "S76 04 00 A0 01 P";

            foreach (string command in commandList)
            {
                this.uiElements.ListBoxOutput.Items.Add(command);
                this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
                this.UsbI2C.SendCommand(command);
            }
        }

        /// <summary>
        /// Sets the LEDs of the I2C-LCD.
        /// </summary>
        /// <param name="ledState">Wanted state of the LCDs.</param>
        public void SetLeds(LedKeyState ledState)
        {
            string command = "S76 04 00 B3 ";
            byte ledStatusByte = 0xF0;
            if (ledState.Object1)
            {
                ledStatusByte += 1;
            }

            if (ledState.Object2)
            {
                ledStatusByte += 2;
            }

            if (ledState.Object3)
            {
                ledStatusByte += 4;
            }

            if (ledState.Object4)
            {
                ledStatusByte += 8;
            }

            command += ledStatusByte.ToString("X2");
            command += " P";

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
            this.UsbI2C.SendCommand(command);
        }

        /// <summary>
        /// Gets the state of the keys.
        /// </summary>
        /// <returns>State of the keys.</returns>
        public LedKeyState GetKeyState()
        {
            string command = "S76 02 00 D4 R01 P";

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
            int result = this.UsbI2C.SendReceiveByteCommand(command);

            this.uiElements.ListBoxInput.Items.Add(result.ToString("X2"));
            this.uiElements.ListBoxInput.TopIndex = this.uiElements.ListBoxInput.Items.Count - 1;

            LCD.LedKeyState keyState = new LedKeyState();
            keyState.Object1 = (result & 0x80) == 0;
            keyState.Object2 = (result & 0x40) == 0;
            keyState.Object3 = (result & 0x20) == 0;
            keyState.Object4 = (result & 0x10) == 0;

            return keyState;
        }

        /// <summary>
        /// Triggers the display.
        /// </summary>
        /// <param name="lcdState">State of the display.</param>
        public void TriggerLcd(string lcdState)
        {
            string command = string.Format("S76 04 00 81 {0} P", lcdState);
            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
            this.UsbI2C.SendCommand(command);
        }

        /// <summary>
        /// Clears the display.
        /// </summary>
        public void ClearDisplay()
        {
            string command = "S76 04 00 81 00 00 00 00 00 00 00 00 00 00 P";
            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
            this.UsbI2C.SendCommand(command);
        }

        /// <summary>
        /// Imports the UI-Elements of the I2C-LCD.
        /// </summary>
        /// <param name="uiElements">UI-Elements of the I2C-LCD.</param>
        public void ImportUiElements(LCD.UiElements uiElements)
        {
            this.uiElements = uiElements;
        }

        /// <summary>
        /// UI-Elements of the I2C-LCD.
        /// </summary>
        public struct UiElements
        {
            /// <summary>
            /// Gets or sets output-command list.
            /// </summary>
            public System.Windows.Forms.ListBox ListBoxOutput;

            /// <summary>
            /// Gets or sets input-command list.
            /// </summary>
            public System.Windows.Forms.ListBox ListBoxInput;
        }

        /// <summary>
        /// Struct of four bools, used for state of keys or LEDs.
        /// </summary>
        public struct LedKeyState
        {
            /// <summary>
            /// State of Object 1.
            /// </summary>
            public bool Object1;

            /// <summary>
            /// State of Object 2.
            /// </summary>
            public bool Object2;

            /// <summary>
            /// State of Object 3.
            /// </summary>
            public bool Object3;

            /// <summary>
            /// State of Object 4.
            /// </summary>
            public bool Object4;
        }
    }
}