﻿//-----------------------------------------------------------------------------
// <copyright file="LinearAccelerationSensor.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Linear acceleration sensor of the I2C-6D-BS.
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using System.Windows.Forms.DataVisualization.Charting;

    /// <summary>
    /// Linear acceleration sensor of the I2C-6D-BS.
    /// </summary>
    public class LinearAccelerationSensor : AbstractDevice
    {
        /// <summary>
        /// I2C-address of the linear acceleration senslor of the I2C-6D-BS.
        /// </summary>
        private byte address;

        /// <summary>
        /// General UI-Elements.
        /// </summary>
        private I2C6DBS.UiElements uiElements;

        /// <summary>
        /// Determines whether the low or power down mode is enabled or not.
        /// </summary>
        private bool isLowPowerModeEnabled = false;

        /// <summary>
        /// Determines whether the high resolution mode is enabled or not.
        /// </summary>
        private bool isHighResolutionModeEnabled = false;

        /// <summary>
        /// Determines whether the flags for the modes are initialized or not.
        /// </summary>
        private bool areModesInitialized = false;

        /// <summary>
        /// Initializes a new instance of the LinearAccelerationSensor class.
        /// </summary>
        /// <param name="usbI2C">I2C comport-device.</param>
        public LinearAccelerationSensor(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.uiElements = new I2C6DBS.UiElements();
            this.SetSlaveAdress(SlaveAdress.x30x31);
        }

        /// <summary>
        /// List of all available register.
        /// </summary>
        public enum Register
        {
            /// <summary>
            /// Settings register.
            /// </summary>
            Settings = 0x20,

            /// <summary>
            /// Filter options register.
            /// </summary>
            FilterOptions = 0x21,

            /// <summary>
            /// Scale options register.
            /// </summary>
            ScaleOptions = 0x23,

            /// <summary>
            /// Status register.
            /// </summary>
            Status = 0x27,

            /// <summary>
            /// First possible slave address register.
            /// </summary>
            AdressX30X31 = 0x30,

            /// <summary>
            /// Second possible slave address register.
            /// </summary>
            AdressX32X33 = 0x32
        }

        /// <summary>
        /// List of all possible slave addresses.
        /// </summary>
        public enum SlaveAdress
        {
            /// <summary>
            /// First possible slave address: 0x30/0x31.
            /// </summary>
            x30x31,

            /// <summary>
            /// Second possible slave address: 0x32/0x33.
            /// </summary>
            x32x33
        }

        /// <summary>
        /// Choice of data rate.
        /// </summary>
        public enum DataRate
        {
            /// <summary>
            /// Power-down mode.
            /// </summary>
            PowerDownMode,

            /// <summary>
            /// Normal / Low power mode (1 Hz).
            /// </summary>
            Hz1,

            /// <summary>
            /// Normal / Low power mode (10 Hz).
            /// </summary>
            Hz10,

            /// <summary>
            /// Normal / Low power mode (25 Hz).
            /// </summary>
            Hz25,

            /// <summary>
            /// Normal / Low power mode (50 Hz).
            /// </summary>
            Hz50,

            /// <summary>
            /// Normal / Low power mode (100 Hz).
            /// </summary>
            Hz100,

            /// <summary>
            /// Normal / Low power mode (200 Hz).
            /// </summary>
            Hz200,

            /// <summary>
            /// Normal / Low power mode (400 Hz).
            /// </summary>
            Hz400,

            /// <summary>
            /// Low power mode (1.620 kHz).
            /// </summary>
            LowPowerMode,

            /// <summary>
            /// Normal (1.344 kHz) / Low power mode (5.376 kHz).
            /// </summary>
            KHz1344
        }

        /// <summary>
        /// Choice of full scale.
        /// </summary>
        public enum FullScale
        {
            /// <summary>
            /// Full Scale of +/- 2G.
            /// </summary>
            TwoG,

            /// <summary>
            /// Full Scale of +/- 4G.
            /// </summary>
            FourG,

            /// <summary>
            /// Full Scale of +/- 8G.
            /// </summary>
            EightG,

            /// <summary>
            /// Full Scale of +/- 16G.
            /// </summary>
            SixteenG
        }

        /// <summary>
        /// Choice of spi-mode.
        /// </summary>
        public enum SpiMode
        {
            /// <summary>
            /// Three-wire interface.
            /// </summary>
            ThreeWireInterface,

            /// <summary>
            /// Four-wire interface.
            /// </summary>
            FourWireInterface
        }

        /// <summary>
        /// Gets or sets the chart of the sensor.
        /// </summary>
        /// <value>Chart of the sensor.</value>
        public Chart DataChart { get; set; }

        /// <summary>
        /// Gets or sets current x-value of read data.
        /// </summary>
        /// <value>Read x-value.</value>
        public int X { get; set; }

        /// <summary>
        /// Gets or sets current y-value of read data.
        /// </summary>
        /// <value>Read y-value.</value>
        public int Y { get; set; }

        /// <summary>
        /// Gets or sets current z-value of read data.
        /// </summary>
        /// <value>Read z-value.</value>
        public int Z { get; set; }

        /// <summary>
        /// Gets or sets current endian selection.
        /// </summary>
        /// <value>Current endian selection.</value>
        public I2C6DBS.EndianSelection Endian { get; set; }

        /// <summary>
        /// Gets current status of the device. 
        /// </summary>
        /// <value>Current status of the angular rate sensor.</value>
        public LinearAccelerationSensor.PositionStatus Status
        {
            get
            {
                return this.GetCurrentStatus();
            }
        }

        /// <summary>
        /// Imports general UI-Elements.
        /// </summary>
        /// <param name="uiElements">UI-Elements to import.</param>
        public void ImportUiElements(I2C6DBS.UiElements uiElements)
        {
            this.uiElements = uiElements;
        }

        /// <summary>
        /// Sets the slave address of the device.
        /// </summary>
        /// <param name="slaveAdress">Wanted slave address.</param>
        public void SetSlaveAdress(LinearAccelerationSensor.SlaveAdress slaveAdress)
        {
            switch (slaveAdress)
            {
                case SlaveAdress.x30x31:
                    this.address = (byte) LinearAccelerationSensor.Register.AdressX30X31;
                    break;
                case SlaveAdress.x32x33:
                    this.address = (byte)LinearAccelerationSensor.Register.AdressX32X33;
                    break;
                default:
                    throw new ArgumentException();
            }
        }

        /// <summary>
        /// Sets the data rate of the linear acceleration sensor.
        /// </summary>
        /// <param name="dataRate">Chosen data rate.</param>
        /// <param name="enablePowerDownMode">Enable or disable power down mode.</param>
        public void SetDataRate(LinearAccelerationSensor.DataRate dataRate, bool enablePowerDownMode)
        {
            byte data = 0;

            switch (dataRate)
            {
                case DataRate.PowerDownMode:
                    break;

                case DataRate.Hz1:
                    data += BitOperationUtils.Bit4;
                    break;

                case DataRate.Hz10:
                    data += BitOperationUtils.Bit5;
                    break;

                case DataRate.Hz25:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit5;
                    break;

                case DataRate.Hz50:
                    data += BitOperationUtils.Bit6;
                    break;

                case DataRate.Hz100:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit6;
                    break;

                case DataRate.Hz200:
                    data += BitOperationUtils.Bit5;
                    data += BitOperationUtils.Bit6;
                    break;

                case DataRate.Hz400:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit5;
                    data += BitOperationUtils.Bit6;
                    break;

                case DataRate.LowPowerMode:
                    data += BitOperationUtils.Bit7;
                    break;

                case DataRate.KHz1344:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit7;
                    break;
            }

            if (enablePowerDownMode)
            {
                data += BitOperationUtils.Bit3;
            }

            if (dataRate == DataRate.LowPowerMode || dataRate == DataRate.PowerDownMode || enablePowerDownMode)
            {
                this.isLowPowerModeEnabled = true;
            }
            else
            {
                this.isLowPowerModeEnabled = false;
            }

            this.areModesInitialized = true;

            // activate x-, y- and z-axis
            data += BitOperationUtils.Bit2;
            data += BitOperationUtils.Bit1;
            data += BitOperationUtils.Bit0;

            this.WriteRegister((byte) LinearAccelerationSensor.Register.Settings, data);
        }

        /// <summary>
        /// Sets the filter options of the linear acceleration sensor.
        /// </summary>
        /// <param name="filterMode">Chosen filter mode.</param>
        /// <param name="filterCutOffFrequency1">Enable first bit of the filter cut-off frequency.</param>
        /// <param name="filterCutOffFrequency2">Enable second bit of the filter cut-off frequency.</param>
        /// <param name="filteredDataSelection">Enabled = data from internal filter sent to output register and FIFO; Disabled: internal filter bypassed.</param>
        /// <param name="highPassFilterForClickFunction">Enable filter for CLICK function.</param>
        /// <param name="highPassFilterForAoiFunctionInterrupt2">Enable filter for AOI function on interrupt 2.</param>
        /// <param name="highPassFilterForAoiFunctionInterrupt1">Enable filter for AOI function on interrupt 1.</param>
        public void SetFilter(
            I2C6DBS.FilterMode filterMode,
            bool filterCutOffFrequency1,
            bool filterCutOffFrequency2,
            bool filteredDataSelection,
            bool highPassFilterForClickFunction,
            bool highPassFilterForAoiFunctionInterrupt2,
            bool highPassFilterForAoiFunctionInterrupt1)
        {
            byte filterByte = 0x00;

            switch (filterMode)
            {
                case I2C6DBS.FilterMode.NormalModeWithReset:
                    break;
                case I2C6DBS.FilterMode.ReferenceSignal:
                    filterByte += BitOperationUtils.Bit6;
                    break;
                case I2C6DBS.FilterMode.NormalMode:
                    filterByte += BitOperationUtils.Bit7;
                    break;
                case I2C6DBS.FilterMode.AutoresetOnInterruptEvent:
                    filterByte += BitOperationUtils.Bit6;
                    filterByte += BitOperationUtils.Bit7;
                    break;
            }

            if (filterCutOffFrequency1)
            {
                filterByte += BitOperationUtils.Bit5;
            }

            if (filterCutOffFrequency2)
            {
                filterByte += BitOperationUtils.Bit4;
            }

            if (filteredDataSelection)
            {
                filterByte += BitOperationUtils.Bit3;
            }

            if (highPassFilterForClickFunction)
            {
                filterByte += BitOperationUtils.Bit2;
            }

            if (highPassFilterForAoiFunctionInterrupt2)
            {
                filterByte += BitOperationUtils.Bit1;
            }

            if (highPassFilterForAoiFunctionInterrupt1)
            {
                filterByte += BitOperationUtils.Bit0;
            }

            this.WriteRegister((byte) LinearAccelerationSensor.Register.FilterOptions, filterByte);
        }

        /// <summary>
        /// Sets the scale options of the linear acceleration sensor.
        /// </summary>
        /// <param name="endian">Choice of big/small endian.</param>
        /// <param name="fullScale">Choice of full scale.</param>
        /// <param name="spiMode">Choice of spi-mode.</param>
        /// <param name="highResolutionOutputMode">Enable high resolution output mode.</param>
        public void SetScale(
            I2C6DBS.EndianSelection endian,
            LinearAccelerationSensor.FullScale fullScale,
            LinearAccelerationSensor.SpiMode spiMode,
            bool highResolutionOutputMode)
        {
            byte scale = 0x00;

            switch (endian)
            {
                case I2C6DBS.EndianSelection.DataLsbAtLowerAddress:
                    this.Endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                    break;
                case I2C6DBS.EndianSelection.DataMsbAtLowerAddress:
                    scale += BitOperationUtils.Bit6;

                    this.Endian = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
                    break;
            }

            switch (fullScale)
            {
                case LinearAccelerationSensor.FullScale.TwoG:
                    break;
                case LinearAccelerationSensor.FullScale.FourG:
                    scale += BitOperationUtils.Bit4;
                    break;
                case LinearAccelerationSensor.FullScale.EightG:
                    scale += BitOperationUtils.Bit5;
                    break;
                case LinearAccelerationSensor.FullScale.SixteenG:
                    scale += BitOperationUtils.Bit4;
                    scale += BitOperationUtils.Bit5;
                    break;
            }

            switch (spiMode)
            {
                case LinearAccelerationSensor.SpiMode.ThreeWireInterface:
                    scale += BitOperationUtils.Bit0;
                    break;
                case LinearAccelerationSensor.SpiMode.FourWireInterface:
                    break;
            }

            this.isHighResolutionModeEnabled = highResolutionOutputMode;
            this.areModesInitialized = true;

            this.WriteRegister((byte) LinearAccelerationSensor.Register.ScaleOptions, scale);
        }

        /// <summary>
        /// Updates the chart of the linear acceleration sensor.
        /// </summary>
        public void UpdateChart()
        {
            ChartUtils.MoveSeriesPointsToRight(this.DataChart.Series);
            this.SetChartAxisMinMax();

            List<int> linearAcceleration = this.GetLinearAcceleration();

            for (int i = 0; i < this.DataChart.Series.Count; i++)
            {
                this.DataChart.Series[i].Points[0].YValues = new double[] { linearAcceleration[i] };
            }

            this.X = linearAcceleration[0];
            this.Y = linearAcceleration[1];
            this.Z = linearAcceleration[2];
        }

        /// <summary>
        /// Gets the basic settings from the device.
        /// </summary>
        /// <param name="dataRate">Read data rate.</param>
        /// <param name="enablePowerDownMode">Read flag for enabling power down mode.</param>
        public void GetBasicSettings(out DataRate dataRate, out bool enablePowerDownMode)
        {
            byte settings = this.ReadRegister((byte) LinearAccelerationSensor.Register.Settings);

            if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) == 0)
            {
                dataRate = DataRate.PowerDownMode;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) != 0)
            {
                dataRate = DataRate.Hz1;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) != 0 &&
                (settings & BitOperationUtils.Bit4) == 0)
            {
                dataRate = DataRate.Hz10;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) != 0 &&
                (settings & BitOperationUtils.Bit4) != 0)
            {
                dataRate = DataRate.Hz25;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) != 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) == 0)
            {
                dataRate = DataRate.Hz50;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) != 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) != 0)
            {
                dataRate = DataRate.Hz100;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) != 0 &&
                (settings & BitOperationUtils.Bit5) != 0 &&
                (settings & BitOperationUtils.Bit4) == 0)
            {
                dataRate = DataRate.Hz200;
            }
            else if ((settings & BitOperationUtils.Bit7) == 0 &&
                (settings & BitOperationUtils.Bit6) != 0 &&
                (settings & BitOperationUtils.Bit5) != 0 &&
                (settings & BitOperationUtils.Bit4) != 0)
            {
                dataRate = DataRate.Hz400;
            }
            else if ((settings & BitOperationUtils.Bit7) != 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) == 0)
            {
                dataRate = DataRate.LowPowerMode;
            }
            else if ((settings & BitOperationUtils.Bit7) != 0 &&
                (settings & BitOperationUtils.Bit6) == 0 &&
                (settings & BitOperationUtils.Bit5) == 0 &&
                (settings & BitOperationUtils.Bit4) != 0)
            {
                dataRate = DataRate.KHz1344;
            }
            else
            {
                throw new ArgumentException();
            }

            if ((settings & BitOperationUtils.Bit3) != 0)
            {
                enablePowerDownMode = true;
            }
            else
            {
                enablePowerDownMode = false;
            }

            this.isLowPowerModeEnabled = enablePowerDownMode;
            this.areModesInitialized = true;
        }

        /// <summary>
        /// Gets the filter from the device.
        /// </summary>
        /// <param name="filterMode">Read filter mode.</param>
        /// <param name="cutOffFrequency1">Read cut-off frequency (first bit).</param>
        /// <param name="cutOffFrequency2">Read cut-off frequency (second bit).</param>
        /// <param name="filteredDataSelection">Read filtered data selection.</param>
        public void GetFilter(out I2C6DBS.FilterMode filterMode, out bool cutOffFrequency1, out bool cutOffFrequency2, out bool filteredDataSelection)
        {
            byte filter = this.ReadRegister((byte) LinearAccelerationSensor.Register.FilterOptions);

            if ((filter & BitOperationUtils.Bit7) == 0 && (filter & BitOperationUtils.Bit6) == 0)
            {
                filterMode = I2C6DBS.FilterMode.NormalModeWithReset;
            }
            else if ((filter & BitOperationUtils.Bit7) == 0 && (filter & BitOperationUtils.Bit6) != 0)
            {
                filterMode = I2C6DBS.FilterMode.ReferenceSignal;
            }
            else if ((filter & BitOperationUtils.Bit7) != 0 && (filter & BitOperationUtils.Bit6) == 0)
            {
                filterMode = I2C6DBS.FilterMode.NormalMode;
            }
            else if ((filter & BitOperationUtils.Bit7) != 0 && (filter & BitOperationUtils.Bit6) != 0)
            {
                filterMode = I2C6DBS.FilterMode.AutoresetOnInterruptEvent;
            }
            else
            {
                throw new ArgumentException();
            }

            cutOffFrequency2 = (filter & BitOperationUtils.Bit5) != 0 ? true : false;
            cutOffFrequency1 = (filter & BitOperationUtils.Bit4) != 0 ? true : false;
            filteredDataSelection = (filter & BitOperationUtils.Bit3) != 0 ? true : false;
        }

        /// <summary>
        /// Gets the scale from the device.
        /// </summary>
        /// <param name="endian">Read endian selection.</param>
        /// <param name="fullScale">Read full scale.</param>
        /// <param name="highResolutionOutputMode">Read flag for enabling high resolution mode.</param>
        /// <param name="spiMode">Read spi mode.</param>
        public void GetScale(out I2C6DBS.EndianSelection endian, out FullScale fullScale, out bool highResolutionOutputMode, out SpiMode spiMode)
        {
            byte scale = this.ReadRegister((byte)LinearAccelerationSensor.Register.ScaleOptions);

            if ((scale & BitOperationUtils.Bit6) != 0)
            {
                endian = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
            }
            else
            {
                endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
            }

            if ((scale & BitOperationUtils.Bit5) == 0 && (scale & BitOperationUtils.Bit4) == 0)
            {
                fullScale = FullScale.TwoG;
            }
            else if ((scale & BitOperationUtils.Bit5) == 0 && (scale & BitOperationUtils.Bit4) != 0)
            {
                fullScale = FullScale.FourG;
            }
            else if ((scale & BitOperationUtils.Bit5) != 0 && (scale & BitOperationUtils.Bit4) == 0)
            {
                fullScale = FullScale.EightG;
            }
            else if ((scale & BitOperationUtils.Bit5) != 0 && (scale & BitOperationUtils.Bit4) != 0)
            {
                fullScale = FullScale.SixteenG;
            }
            else
            {
                throw new ArgumentException();
            }

            highResolutionOutputMode = ((scale & BitOperationUtils.Bit3) != 0) ? true : false;
            this.isHighResolutionModeEnabled = highResolutionOutputMode;
            this.areModesInitialized = true;

            spiMode = ((scale & BitOperationUtils.Bit0) != 0) ? SpiMode.ThreeWireInterface : SpiMode.FourWireInterface;
        }

        /// <summary>
        /// Gets the value of a axis (x/y/z) of one of I2C-6D-BS sensors.
        /// </summary>
        /// <param name="axis">Axis (x/y/z).</param>
        /// <returns>Value of the axis.</returns>
        public int GetSingleAxis(I2C6DBS.Axis axis)
        {
            const byte AddressX = 0x28;
            const byte AddressY = 0x2A;
            const byte AddressZ = 0x2C;

            byte address;
            switch (axis)
            {
                case I2C6DBS.Axis.X:
                    address = AddressX;
                    break;
                case I2C6DBS.Axis.Y:
                    address = AddressY;
                    break;
                case I2C6DBS.Axis.Z:
                    address = AddressZ;
                    break;
                default:
                    throw new ArgumentException();
            }

            List<byte> readResults = new List<byte>();
            readResults.Add(this.ReadRegister(address));
            readResults.Add(this.ReadRegister((byte)(address + 1)));

            // reverse order of read values if msb is at lower address (= little endian)
            if (this.Endian == I2C6DBS.EndianSelection.DataMsbAtLowerAddress)
            {
                readResults.Reverse();
            }

            short axisValue;
            try
            {
                axisValue = BitConverter.ToInt16(readResults.ToArray(), 0);
            }
            catch
            {
                throw new InvalidOperationException("Das Auslesen der aktuellen Bewegungsdaten war nicht erfolgreich.");
            }

            if (!this.areModesInitialized)
            {
                DataRate dr;
                bool b;
                this.GetBasicSettings(out dr, out b);

                I2C6DBS.EndianSelection endian;
                FullScale fullScale;
                bool highResolutionOutputMode;
                SpiMode spiMode;
                this.GetScale(out endian, out fullScale, out highResolutionOutputMode, out spiMode);
            }

            sbyte highByte = (sbyte)(axisValue / 256);
            sbyte lowByte = (sbyte)(axisValue % 256);
            if (!this.isLowPowerModeEnabled && this.isHighResolutionModeEnabled)
            {
                return ((highByte * 256) + lowByte) >> 4;
            }
            else if (!this.isLowPowerModeEnabled && !this.isHighResolutionModeEnabled)
            {
                return ((highByte * 256) + lowByte) >> 6;
            }
            else
            {
                return highByte;
            }
        }

        /// <summary>
        /// Sets the maximum and minimum of the chart. Also sets a matching intervall.
        /// </summary>
        private void SetChartAxisMinMax()
        {
            if (!this.isLowPowerModeEnabled && this.isHighResolutionModeEnabled)
            {
                this.DataChart.ChartAreas[0].AxisY.Maximum = 2200;
                this.DataChart.ChartAreas[0].AxisY.Minimum = -2100;

                this.DataChart.ChartAreas[0].AxisY.Interval = 1050;
            }
            else if (!this.isLowPowerModeEnabled && !this.isHighResolutionModeEnabled)
            {
                this.DataChart.ChartAreas[0].AxisY.Maximum = 550;
                this.DataChart.ChartAreas[0].AxisY.Minimum = -550;

                this.DataChart.ChartAreas[0].AxisY.Interval = 275;
            }
            else
            {
                this.DataChart.ChartAreas[0].AxisY.Maximum = 130;
                this.DataChart.ChartAreas[0].AxisY.Minimum = -130;

                this.DataChart.ChartAreas[0].AxisY.Interval = 65;
            }
        }

        /// <summary>
        /// Gets the current linear acceleration of the I2C-6D-BS.
        /// </summary>
        /// <returns>Current linear acceleration of the I2C-6D-BS.</returns>
        private List<int> GetLinearAcceleration()
        {
            List<int> result = new List<int>();
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.X));
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.Y));
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.Z));

            return result;
        }

        /// <summary>
        /// Sets a register of the device with a specific byte.
        /// </summary>
        /// <param name="register">Register to set.</param>
        /// <param name="data">Data to use.</param>
        private void WriteRegister(byte register, byte data)
        {
            string command = string.Format(
                "S{0} {1} {2} P",
                this.address.ToString("X2"),
                register.ToString("X2"),
                data.ToString("X2"));

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.UsbI2C.SendCommand(command);
        }

        /// <summary>
        /// Reads a register of the device.
        /// </summary>
        /// <param name="register">Wanted register.</param>
        /// <returns>Read value.</returns>
        private byte ReadRegister(byte register)
        {
            string command = string.Format(
                   "S{0} {1} R01 P",
                   this.address.ToString("X2"),
                   register.ToString("X2"));

            this.uiElements.ListBoxOutput.Items.Add(command);

            byte result;
            string read = this.UsbI2C.SendReceiveCommand(command);
            try
            {
                result = byte.Parse(read.Trim(), System.Globalization.NumberStyles.AllowHexSpecifier);
            }
            catch (FormatException)
            {
                this.uiElements.ListBoxInput.Items.Add(read);
                throw new InvalidOperationException();
            }

            this.uiElements.ListBoxInput.Items.Add(result.ToString("X2"));
            return result;
        }

        /// <summary>
        /// Reads the current status from the device.
        /// </summary>
        /// <returns>Current status from the device.</returns>
        private LinearAccelerationSensor.PositionStatus GetCurrentStatus()
        {
            LinearAccelerationSensor.PositionStatus boolWrapper = new PositionStatus();
            byte status = this.ReadRegister((byte) LinearAccelerationSensor.Register.Status);

            boolWrapper.OverrunZ = ((status & BitOperationUtils.Bit6) != 0) ? true : false;
            boolWrapper.OverrunY = ((status & BitOperationUtils.Bit5) != 0) ? true : false;
            boolWrapper.OverrunX = ((status & BitOperationUtils.Bit4) != 0) ? true : false;

            boolWrapper.NewDataZ = ((status & BitOperationUtils.Bit2) != 0) ? true : false;
            boolWrapper.NewDataY = ((status & BitOperationUtils.Bit1) != 0) ? true : false;
            boolWrapper.NewDataX = ((status & BitOperationUtils.Bit0) != 0) ? true : false;

            return boolWrapper;
        }

        /// <summary>
        /// Current status of the angular rate sensor.
        /// </summary>
        public struct PositionStatus
        {
            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the x-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the x-value.</value>
            public bool OverrunX { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the y-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the y-value.</value>
            public bool OverrunY { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the z-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the z-value.</value>
            public bool OverrunZ { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the x-value is set or not.
            /// </summary>
            /// <value>New data flag for the x-value.</value>
            public bool NewDataX { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the y-value is set or not.
            /// </summary>
            /// <value>New data flag for the y-value.</value>
            public bool NewDataY { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the z-value is set or not.
            /// </summary>
            /// <value>New data flag for the z-value.</value>
            public bool NewDataZ { get; set; }
        }
    }
}
