﻿//-----------------------------------------------------------------------------
// <copyright file="I2C6DBS.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// I2C-6D-BS
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;

    /// <summary>
    /// Wrapper of I2C-6D-BS.
    /// </summary>
    public class I2C6DBS
        : AbstractDevice
    {
        /// <summary>
        /// Initializes a new instance of the I2C6DBS class.
        /// </summary>
        /// <param name="usbI2C">I2C comport-device.</param>
        public I2C6DBS(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.LinearAcceleration = new LinearAccelerationSensor(usbI2C);

            this.AngularRate = new AngularRateSensor(usbI2C);
        }

        /// <summary>
        /// Choice of different Axis for charts.
        /// </summary>
        public enum Axis
        {
            /// <summary>
            /// X-axis of a sensor.
            /// </summary>
            X,

            /// <summary>
            /// Y-axis of a sensor.
            /// </summary>
            Y,

            /// <summary>
            /// Z-axis of a sensor.
            /// </summary>
            Z
        }

        /// <summary>
        /// Choice of filter mode.
        /// </summary>
        public enum FilterMode
        {
            /// <summary>
            /// Normal mode (reset reading HP_RESET_FILTER).
            /// </summary>
            NormalModeWithReset,

            /// <summary>
            /// Reference signal for filtering.
            /// </summary>
            ReferenceSignal,

            /// <summary>
            /// Normal mode.
            /// </summary>
            NormalMode,

            /// <summary>
            /// Autoreset on interrupt event.
            /// </summary>
            AutoresetOnInterruptEvent
        }

        /// <summary>
        /// Big/little endian selection.
        /// </summary>
        public enum EndianSelection
        {
            /// <summary>
            /// Data LSb @ lower address.
            /// </summary>
            DataLsbAtLowerAddress,

            /// <summary>
            /// Data MSb @ lower address.
            /// </summary>
            DataMsbAtLowerAddress,
        }

        /// <summary>
        /// Gets or sets linear acceleration sensor of I2C' 6D-BS.
        /// </summary>
        /// <value>Linear acceleration sensor of I2C' 6D-BS.</value>
        public LinearAccelerationSensor LinearAcceleration { get; set; }

        /// <summary>
        /// Gets or sets angular rate sensor of I2C' 6D-BS.
        /// </summary>
        /// <value>Angular rate sensor of I2C' 6D-BS.</value>
        public AngularRateSensor AngularRate { get; set; }

        /// <summary>
        /// Gets the value of a axis (x/y/z) of one of I2C-6D-BS sensors.
        /// </summary>
        /// <param name="axis">Axis (x/y/z).</param>
        /// <param name="usbI2c">I2C comport-device.</param>
        /// <param name="deviceAddress">Address of the I2C-device.</param>
        /// <param name="uiElements">UI-Elements of the I2C-device.</param>
        /// <returns>Value of the axis.</returns>
        public static int GetSingleAxis(I2C6DBS.Axis axis, UsbI2C usbI2c, byte deviceAddress, I2C6DBS.UiElements uiElements)
        {
            const int AdressX = 0x28;
            const int AdressY = 0x2A;
            const int AdressZ = 0x2C;

            List<string> readResults = new List<string>();
            int adress = 0;
            switch (axis)
            {
                case I2C6DBS.Axis.X:
                    adress = AdressX;
                    break;
                case I2C6DBS.Axis.Y:
                    adress = AdressY;
                    break;
                case I2C6DBS.Axis.Z:
                    adress = AdressZ;
                    break;
            }

            List<string> commandList = new List<string>();
            commandList.Add(string.Format(
                "S{0} {1} {2} P",
                deviceAddress.ToString("X2"),
                adress.ToString("X2"),
                "R01"));
            commandList.Add(string.Format(
                 "S{0} {1} {2} P",
                 deviceAddress.ToString("X2"),
                 (adress + 1).ToString("X2"),
                 "R01"));

            foreach (string command in commandList)
            {
                uiElements.ListBoxOutput.Items.Add(command);
                readResults.Add(usbI2c.SendReceiveCommand(command));
            }

            for (int i = 0; i < readResults.Count; i++)
            {
                readResults[i] = readResults[i].Trim();
                uiElements.ListBoxInput.Items.Add(readResults[i]);
            }

            short result;
            try
            {
                result = BitConverter.ToInt16(
                    new byte[]
                {
                    byte.Parse(readResults[0], System.Globalization.NumberStyles.AllowHexSpecifier),
                    byte.Parse(readResults[1], System.Globalization.NumberStyles.AllowHexSpecifier)
                },
                    0);
            }
            catch
            {
                throw new InvalidOperationException("Das Auslesen der aktuellen Bewegungsdaten war nicht erfolgreich.");
            }

            return result;
        }

        /// <summary>
        /// General UI-Elements.
        /// </summary>
        public struct UiElements
        {
            /// <summary>
            /// Gets or sets output-command list.
            /// </summary>
            /// <value>Output-command list.</value>
            public ListBox ListBoxOutput { get; set; }

            /// <summary>
            /// Gets or sets input-command list.
            /// </summary>
            /// <value>Input-command list.</value>
            public ListBox ListBoxInput { get; set; }
        }
    }
}
