﻿//-----------------------------------------------------------------------------
// <copyright file="AngularRateSensor.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Angular rate sensor of the I2C-6DBS.
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms.DataVisualization.Charting;

    /// <summary>
    /// Angular rate sensor of the I2C-6D-BS.
    /// </summary>
    public class AngularRateSensor : AbstractDevice
    {
        /// <summary>
        /// I2C-address of the angular rate sensor of the I2C-6D-BS.
        /// </summary>
        private byte address = 0xD4;

        /// <summary>
        /// General UI-Elements.
        /// </summary>
        private I2C6DBS.UiElements uiElements;
        
        /// <summary>
        /// Initializes a new instance of the AngularRateSensor class.
        /// </summary>
        /// <param name="usbI2C">I2C Comport-device.</param>
        public AngularRateSensor(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.uiElements = new I2C6DBS.UiElements();

            this.Endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
            this.SetSlaveAdress(SlaveAdress.xD4xD5);
        }

        /// <summary>
        /// List of all available register.
        /// </summary>
        public enum Register
        {
            /// <summary>
            /// Chip id register.
            /// </summary>
            ChipID = 0x0F,

            /// <summary>
            /// Settings register.
            /// </summary>
            Settings = 0x20,

            /// <summary>
            /// Filter register 1.
            /// </summary>
            FilterSettings1 = 0x21,

            /// <summary>
            /// Scale register.
            /// </summary>
            Scale = 0x23,

            /// <summary>
            /// Filter register 2.
            /// </summary>
            FilterSettings2 = 0x24,

            /// <summary>
            /// Temperature register.
            /// </summary>
            Temperature = 0x26,

            /// <summary>
            /// Status register.
            /// </summary>
            Status = 0x27,

            /// <summary>
            /// First possible slave address register.
            /// </summary>
            AddressXD4XD5 = 0xD4,

            /// <summary>
            /// Second possible slave address register.
            /// </summary>
            AdressXD6XD7 = 0xD6
        }

        /// <summary>
        /// List of all possible slave addresses.
        /// </summary>
        public enum SlaveAdress
        {
            /// <summary>
            /// First possible slave address: 0xD4/0xD5.
            /// </summary>
            xD4xD5,

            /// <summary>
            /// Second possible slave address: 0xD6/0xD7.
            /// </summary>
            xD6xD7
        }

        /// <summary>
        /// Choice of ODR.
        /// </summary>
        public enum Odr
        {
            /// <summary>
            /// Frequence of 95 Hz.
            /// </summary>
            _95,

            /// <summary>
            /// Frequence of 190 Hz.
            /// </summary>
            _190,

            /// <summary>
            /// Frequence of 380 Hz.
            /// </summary>
            _380,

            /// <summary>
            /// Frequence of 760 Hz.
            /// </summary>
            _760
        }

        /// <summary>
        /// Choice of mode.
        /// </summary>
        public enum Mode
        {
            /// <summary>
            /// Normal mode or Sleep mode.
            /// </summary>
            NormalOrSleepMode,

            /// <summary>
            /// Power-down mode.
            /// </summary>
            PowerDownMode
        }

        /// <summary>
        /// Choice of block data update.
        /// </summary>
        public enum BlockDataUpdate
        {
            /// <summary>
            /// Continuous update.
            /// </summary>
            ContinuousUpdate,

            /// <summary>
            /// Output registers not updated until MSb and LSb reading.
            /// </summary>
            OutputRegistersNotUpdatedUntilMSbAndLSbReading
        }

        /// <summary>
        /// Choice of full scale.
        /// </summary>
        public enum FullScale
        {
            /// <summary>
            /// Scale of 250 dps.
            /// </summary>
            _250dps,

            /// <summary>
            /// Scale of 500 dps.
            /// </summary>
            _500dps,

            /// <summary>
            /// Scale of 1000 dps.
            /// </summary>
            _1000dps,

            /// <summary>
            /// Scale of 2000 dps.
            /// </summary>
            _2000dps
        }

        /// <summary>
        /// Choice of sim.
        /// </summary>
        public enum Sim
        {
            /// <summary>
            /// Three-wire SPI Serial interface read mode diabled.
            /// </summary>
            ThreeWireReadModeDisabled,

            /// <summary>
            /// Three-wire SPI Serial interface read mode enabled.
            /// </summary>
            ThreeWireReadModeEnabled,
        }

        /// <summary>
        /// List of possible data filter.
        /// </summary>
        public enum DataFilter
        {
            /// <summary>
            /// Normal: no additional filter activated.
            /// </summary>
            Normal,

            /// <summary>
            /// Use filter option hpf.
            /// </summary>
            Hpf,

            /// <summary>
            /// Use filter option lpf.
            /// </summary>
            Lpf2,

            /// <summary>
            /// Use filter option hpf and lpf.
            /// </summary>
            HpfLpf2
        }

        /// <summary>
        /// Gets or sets the chart of the sensor.
        /// </summary>
        /// <value>Chart of the sensor.</value>
        public Chart DataChart { get; set; }

        /// <summary>
        /// Gets or sets current x-value of read data.
        /// </summary>
        /// <value>Read x-value.</value>
        public int X { get; set; }

        /// <summary>
        /// Gets or sets current y-value of read data.
        /// </summary>
        /// <value>Read y-value.</value>
        public int Y { get; set; }

        /// <summary>
        /// Gets or sets current z-value of read data.
        /// </summary>
        /// <value>Read z-value.</value>
        public int Z { get; set; }

        /// <summary>
        /// Gets current status of the angular rate sensor. 
        /// </summary>
        /// <value>Current status of the angular rate sensor.</value>
        public AngularRateSensor.PositionStatus Status
        {
            get
            {
                return this.GetCurrentStatus();
            }
        }

        /// <summary>
        /// Gets chip id of the angular rate sensor.
        /// </summary>
        /// <value>Chip id of the angular rate sensor.</value>
        public byte ChipID
        {
            get
            {
                return this.GetChipID();
            }
        }

        /// <summary>
        /// Gets current temperature of the angular rate sensor.
        /// </summary>
        /// <value>Current temperature of the angular rate sensor..</value>
        public int Temperature
        {
            get
            {
                return this.GetTemperature();
            }
        }

        /// <summary>
        /// Gets or sets current endian selection.
        /// </summary>
        /// <value>Current endian selection.</value>
        public I2C6DBS.EndianSelection Endian { get; set; }

        /// <summary>
        /// Import general UI-Elements.
        /// </summary>
        /// <param name="uiElements">General UI-Elements.</param>
        public void ImportUiElements(I2C6DBS.UiElements uiElements)
        {
            this.uiElements = uiElements;
        }

        /// <summary>
        /// Updates the chart of the linear acceleration sensor.
        /// </summary>
        public void UpdateChart()
        {
            ChartUtils.MoveSeriesPointsToRight(this.DataChart.Series);
            List<int> angularRate = this.GetAngularRate();

            for (int i = 0; i < this.DataChart.Series.Count; i++)
            {
                this.DataChart.Series[i].Points[0].YValues = new double[] { angularRate[i] };
            }

            this.X = angularRate[0];
            this.Y = angularRate[1];
            this.Z = angularRate[2];
        }

        /// <summary>
        /// Sets the slave address of the device.
        /// </summary>
        /// <param name="slaveAdress">Wished slave address.</param>
        public void SetSlaveAdress(AngularRateSensor.SlaveAdress slaveAdress)
        {
            switch (slaveAdress)
            {
                case SlaveAdress.xD4xD5:
                    this.address = (byte)AngularRateSensor.Register.AddressXD4XD5;
                    break;
                case SlaveAdress.xD6xD7:
                    this.address = (byte)AngularRateSensor.Register.AdressXD6XD7;
                    break;
                default:
                    throw new ArgumentException();
            }
        }

        /// <summary>
        /// Sets the general settings of the angular rate sensor.
        /// </summary>
        /// <param name="odr">Choice of odr.</param>
        /// <param name="cutOff">Choice of cut-off frequency.</param>
        /// <param name="mode">Choice of mode.</param>
        public void SetSettings(
            AngularRateSensor.Odr odr,
            int cutOff,
            AngularRateSensor.Mode mode)
        {
            byte data = 0x00;

            switch (odr)
            {
                case Odr._95:
                    break;
                case Odr._190:
                    data += BitOperationUtils.Bit6;
                    break;
                case Odr._380:
                    data += BitOperationUtils.Bit7;
                    break;
                case Odr._760:
                    data += BitOperationUtils.Bit6;
                    data += BitOperationUtils.Bit7;
                    break;
            }

            switch (cutOff)
            {
                case 0:
                    break;
                case 1:
                    data += BitOperationUtils.Bit4;
                    break;
                case 2:
                    data += BitOperationUtils.Bit5;
                    break;
                case 3:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit5;
                    break;
            }

            if (mode == Mode.NormalOrSleepMode)
            {
                data += BitOperationUtils.Bit3;
            }

            // activate x-, y- and z-axis
            data += BitOperationUtils.Bit2;
            data += BitOperationUtils.Bit1;
            data += BitOperationUtils.Bit0;

            this.WriteRegister((byte) AngularRateSensor.Register.Settings, data);
        }

        /// <summary>
        /// Sets the filter options of the angular rate sensor.
        /// </summary>
        /// <param name="isEnabledEdgeSensitiveTrigger">Enable edge-sensitive trigger.</param>
        /// <param name="isEnabledLevelSensitiveTrigger">Enable level-sensitive trigger.</param>
        /// <param name="filterMode">Choice of filter-mode.</param>
        /// <param name="cutOffFrequencySelectionIndex">Index of cut-off frequency.</param>
        /// <param name="dataFilter">Data filter choice.</param>
        public void SetFilter(
            bool isEnabledEdgeSensitiveTrigger,
            bool isEnabledLevelSensitiveTrigger,
            I2C6DBS.FilterMode filterMode,
            int cutOffFrequencySelectionIndex,
            AngularRateSensor.DataFilter dataFilter)
        {
            byte data = 0x00;
            if (isEnabledEdgeSensitiveTrigger)
            {
                data += BitOperationUtils.Bit7;
            }

            if (isEnabledLevelSensitiveTrigger)
            {
                data += BitOperationUtils.Bit6;
            }

            switch (filterMode)
            {
                case I2C6DBS.FilterMode.NormalModeWithReset:
                    break;
                case I2C6DBS.FilterMode.ReferenceSignal:
                    data += BitOperationUtils.Bit4;
                    break;
                case I2C6DBS.FilterMode.NormalMode:
                    data += BitOperationUtils.Bit5;
                    break;
                case I2C6DBS.FilterMode.AutoresetOnInterruptEvent:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit5;
                    break;
            }

            switch (cutOffFrequencySelectionIndex)
            {
                case 0:
                    break;
                case 1:
                    data += BitOperationUtils.Bit0;
                    break;
                case 2:
                    data += BitOperationUtils.Bit1;
                    break;
                case 3:
                    data += BitOperationUtils.Bit0;
                    data += BitOperationUtils.Bit1;
                    break;
                case 4:
                    data += BitOperationUtils.Bit2;
                    break;
                case 5:
                    data += BitOperationUtils.Bit0;
                    data += BitOperationUtils.Bit2;
                    break;
                case 6:
                    data += BitOperationUtils.Bit1;
                    data += BitOperationUtils.Bit2;
                    break;
                case 7:
                    data += BitOperationUtils.Bit0;
                    data += BitOperationUtils.Bit1;
                    data += BitOperationUtils.Bit2;
                    break;
                case 8:
                    data += BitOperationUtils.Bit3;
                    break;
                case 9:
                    data += BitOperationUtils.Bit0;
                    data += BitOperationUtils.Bit3;
                    break;
            }

            this.WriteRegister((byte)AngularRateSensor.Register.FilterSettings1, data);

            data = 0x00;
            byte formerValue = this.ReadRegister((byte)AngularRateSensor.Register.FilterSettings2);
            if ((formerValue & BitOperationUtils.Bit2) != 0)
            {
                data += BitOperationUtils.Bit2;
            }

            if ((formerValue & BitOperationUtils.Bit3) != 0)
            {
                data += BitOperationUtils.Bit3;
            }

            if ((formerValue & BitOperationUtils.Bit6) != 0)
            {
                data += BitOperationUtils.Bit6;
            }

            if ((formerValue & BitOperationUtils.Bit7) != 0)
            {
                data += BitOperationUtils.Bit7;
            }

            switch (dataFilter)
            {
                case DataFilter.Normal:
                    break;
                case DataFilter.Hpf:
                    data += BitOperationUtils.Bit0;
                    break;
                case DataFilter.Lpf2:
                    data += BitOperationUtils.Bit1;
                    break;
                case DataFilter.HpfLpf2:
                    data += BitOperationUtils.Bit1;
                    data += BitOperationUtils.Bit5;
                    break;
            }

            this.WriteRegister((byte)AngularRateSensor.Register.FilterSettings2, data);
        }

        /// <summary>
        /// Sets the scale options of the angular rate sensor.
        /// </summary>
        /// <param name="blockDataUpdate">Choice of block data update.</param>
        /// <param name="endianSelection">Choice of endian selection.</param>
        /// <param name="fullScale">Choice of full scale.</param>
        /// <param name="spiMode">Choice of spi-mode.</param>
        public void SetScale(
            AngularRateSensor.BlockDataUpdate blockDataUpdate,
            I2C6DBS.EndianSelection endianSelection,
            AngularRateSensor.FullScale fullScale,
            AngularRateSensor.Sim spiMode)
        {
            byte data = 0x00;

            switch (blockDataUpdate)
            {
                case BlockDataUpdate.ContinuousUpdate:
                    break;
                case BlockDataUpdate.OutputRegistersNotUpdatedUntilMSbAndLSbReading:
                    data += BitOperationUtils.Bit7;
                    break;
            }

            switch (endianSelection)
            {
                case I2C6DBS.EndianSelection.DataLsbAtLowerAddress:
                    this.Endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
                    break;
                case I2C6DBS.EndianSelection.DataMsbAtLowerAddress:
                    data += BitOperationUtils.Bit6;

                    this.Endian = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
                    break;
            }

            switch (fullScale)
            {
                case FullScale._250dps:
                    break;
                case FullScale._500dps:
                    data += BitOperationUtils.Bit4;
                    break;
                case FullScale._1000dps:
                    data += BitOperationUtils.Bit5;
                    break;
                case FullScale._2000dps:
                    data += BitOperationUtils.Bit4;
                    data += BitOperationUtils.Bit5;
                    break;
            }

            switch (spiMode)
            {
                case AngularRateSensor.Sim.ThreeWireReadModeDisabled:
                    break;
                case AngularRateSensor.Sim.ThreeWireReadModeEnabled:
                    data += BitOperationUtils.Bit0;
                    break;
            }

            this.WriteRegister((byte) AngularRateSensor.Register.Scale, data);
        }

        /// <summary>
        /// Gets the basic settings from the device.
        /// </summary>
        /// <param name="odr">Read odr from the device.</param>
        /// <param name="bandwithIndex">Read bandwith.</param>
        /// <param name="enablePowerDownMode">Read flag for enabling power down mode.</param>
        public void GetBasicSettings(out AngularRateSensor.Odr odr, out int bandwithIndex, out bool enablePowerDownMode)
        {
            byte basicSettings = this.ReadRegister((byte)AngularRateSensor.Register.Settings);

            if ((basicSettings & BitOperationUtils.Bit7) == 0 && (basicSettings & BitOperationUtils.Bit6) == 0)
            {
                odr = Odr._95;
            }
            else if ((basicSettings & BitOperationUtils.Bit7) == 0 && (basicSettings & BitOperationUtils.Bit6) != 0)
            {
                odr = Odr._190;
            }
            else if ((basicSettings & BitOperationUtils.Bit7) != 0 && (basicSettings & BitOperationUtils.Bit6) == 0)
            {
                odr = Odr._380;
            }
            else if ((basicSettings & BitOperationUtils.Bit7) != 0 && (basicSettings & BitOperationUtils.Bit6) != 0)
            {
                odr = Odr._760;
            }
            else
            {
                throw new ArgumentException();
            }

            if ((basicSettings & BitOperationUtils.Bit5) == 0 && (basicSettings & BitOperationUtils.Bit4) == 0)
            {
                bandwithIndex = 0;
            }
            else if ((basicSettings & BitOperationUtils.Bit5) == 0 && (basicSettings & BitOperationUtils.Bit4) != 0)
            {
                bandwithIndex = 1;
            }
            else if ((basicSettings & BitOperationUtils.Bit5) != 0 && (basicSettings & BitOperationUtils.Bit4) == 0)
            {
                bandwithIndex = 2;
            }
            else if ((basicSettings & BitOperationUtils.Bit5) != 0 && (basicSettings & BitOperationUtils.Bit4) != 0)
            {
                bandwithIndex = 3;
            }
            else
            {
                throw new ArgumentException();
            }

            if ((basicSettings & BitOperationUtils.Bit3) != 0)
            {
                enablePowerDownMode = false;
            }
            else
            {
                enablePowerDownMode = true;
            }
        }

        /// <summary>
        /// Gets the scale from the device.
        /// </summary>
        /// <param name="endian">Read endian.</param>
        /// <param name="fullScale">Read full scale option.</param>
        /// <param name="blockDataUpdate">Read block data update option.</param>
        /// <param name="sim">Read sim from the device.</param>
        public void GetScale(out I2C6DBS.EndianSelection endian, out FullScale fullScale, out BlockDataUpdate blockDataUpdate, out Sim sim)
        {
            byte scale = this.ReadRegister((byte)AngularRateSensor.Register.Scale);

            if ((scale & BitOperationUtils.Bit7) != 0)
            {
                blockDataUpdate = AngularRateSensor.BlockDataUpdate.OutputRegistersNotUpdatedUntilMSbAndLSbReading;
            }
            else
            {
                blockDataUpdate = AngularRateSensor.BlockDataUpdate.ContinuousUpdate;
            }

            if ((scale & BitOperationUtils.Bit6) != 0)
            {
                endian = I2C6DBS.EndianSelection.DataMsbAtLowerAddress;
            }
            else
            {
                endian = I2C6DBS.EndianSelection.DataLsbAtLowerAddress;
            }

            if ((scale & BitOperationUtils.Bit5) == 0 && (scale & BitOperationUtils.Bit4) == 0)
            {
                fullScale = FullScale._250dps;
            }
            else if ((scale & BitOperationUtils.Bit5) == 0 && (scale & BitOperationUtils.Bit4) != 0)
            {
                fullScale = FullScale._500dps;
            }
            else if ((scale & BitOperationUtils.Bit5) != 0 && (scale & BitOperationUtils.Bit4) == 0)
            {
                fullScale = FullScale._1000dps;
            }
            else if ((scale & BitOperationUtils.Bit5) != 0 && (scale & BitOperationUtils.Bit4) != 0)
            {
                fullScale = FullScale._2000dps;
            }
            else
            {
                throw new ArgumentException();
            }

            if ((scale & BitOperationUtils.Bit0) != 0)
            {
                sim = Sim.ThreeWireReadModeEnabled;
            }
            else
            {
                sim = Sim.ThreeWireReadModeDisabled;
            }
        }

        /// <summary>
        /// Gets the filter from the device.
        /// </summary>
        /// <param name="edgeSensitiveTrigger">Read flag for enabling edge sensitive trigger.</param>
        /// <param name="levelSensitiveTrigger">Read flag for enabling level sensitive trigger.</param>
        /// <param name="modeOfHp">Read hp mode.</param>
        /// <param name="cutOffFrequencyIndex">Read cut-off frequency.</param>
        /// <param name="dataFilter">Read data filter.</param>
        public void GetFilter(out bool edgeSensitiveTrigger, out bool levelSensitiveTrigger, out I2C6DBS.FilterMode modeOfHp, out int cutOffFrequencyIndex, out DataFilter dataFilter)
        {
            byte filterSettings1 = this.ReadRegister((byte)AngularRateSensor.Register.FilterSettings1);
            byte filterSettings2 = this.ReadRegister((byte)AngularRateSensor.Register.FilterSettings2);

            if ((filterSettings1 & BitOperationUtils.Bit7) != 0)
            {
                edgeSensitiveTrigger = true;
            }
            else
            {
                edgeSensitiveTrigger = false;
            }

            if ((filterSettings1 & BitOperationUtils.Bit6) != 0)
            {
                levelSensitiveTrigger = true;
            }
            else
            {
                levelSensitiveTrigger = false;
            }

            if ((filterSettings1 & BitOperationUtils.Bit5) == 0 && (filterSettings1 & BitOperationUtils.Bit4) == 0)
            {
                modeOfHp = I2C6DBS.FilterMode.NormalModeWithReset;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit5) == 0 && (filterSettings1 & BitOperationUtils.Bit4) != 0)
            {
                modeOfHp = I2C6DBS.FilterMode.ReferenceSignal;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit5) != 0 && (filterSettings1 & BitOperationUtils.Bit4) == 0)
            {
                modeOfHp = I2C6DBS.FilterMode.NormalMode;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit5) != 0 && (filterSettings1 & BitOperationUtils.Bit4) != 0)
            {
                modeOfHp = I2C6DBS.FilterMode.AutoresetOnInterruptEvent;
            }
            else
            {
                throw new ArgumentException();
            }

            if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) == 0)
            {
                cutOffFrequencyIndex = 0;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) != 0)
            {
                cutOffFrequencyIndex = 1;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) == 0)
            {
                cutOffFrequencyIndex = 2;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) != 0)
            {
                cutOffFrequencyIndex = 3;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) == 0)
            {
                cutOffFrequencyIndex = 4;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) != 0)
            {
                cutOffFrequencyIndex = 5;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) == 0)
            {
                cutOffFrequencyIndex = 6;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) != 0)
            {
                cutOffFrequencyIndex = 7;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) == 0)
            {
                cutOffFrequencyIndex = 8;
            }
            else if ((filterSettings1 & BitOperationUtils.Bit3) != 0 &&
                (filterSettings1 & BitOperationUtils.Bit2) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings1 & BitOperationUtils.Bit0) != 0)
            {
                cutOffFrequencyIndex = 9;
            }
            else
            {
                cutOffFrequencyIndex = 0;
            }

            if ((filterSettings2 & BitOperationUtils.Bit5) == 0 &&
                (filterSettings2 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings2 & BitOperationUtils.Bit0) == 0)
            {
                dataFilter = DataFilter.Normal;
            }
            else if ((filterSettings2 & BitOperationUtils.Bit5) == 0 &&
                (filterSettings2 & BitOperationUtils.Bit1) == 0 &&
                (filterSettings2 & BitOperationUtils.Bit0) != 0)
            {
                dataFilter = DataFilter.Hpf;
            }
            else if ((filterSettings2 & BitOperationUtils.Bit5) == 0 &&
                (filterSettings2 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings2 & BitOperationUtils.Bit0) == 0)
            {
                dataFilter = DataFilter.Lpf2;
            }
            else if ((filterSettings2 & BitOperationUtils.Bit5) != 0 &&
                (filterSettings2 & BitOperationUtils.Bit1) != 0 &&
                (filterSettings2 & BitOperationUtils.Bit0) == 0)
            {
                dataFilter = DataFilter.HpfLpf2;
            }
            else
            {
                dataFilter = DataFilter.Normal;
            }
        }

        /// <summary>
        /// Gets the value of a axis (x/y/z) of one of I2C-6D-BS sensors.
        /// </summary>
        /// <param name="axis">Axis (x/y/z).</param>
        /// <returns>Value of the axis.</returns>
        public int GetSingleAxis(I2C6DBS.Axis axis)
        {
            const byte AddressX = 0x28;
            const byte AddressY = 0x2A;
            const byte AddressZ = 0x2C;

            byte address;
            switch (axis)
            {
                case I2C6DBS.Axis.X:
                    address = AddressX;
                    break;
                case I2C6DBS.Axis.Y:
                    address = AddressY;
                    break;
                case I2C6DBS.Axis.Z:
                    address = AddressZ;
                    break;
                default:
                    throw new ArgumentException();
            }

            List<byte> readResults = new List<byte>();
            readResults.Add(this.ReadRegister(address));
            readResults.Add(this.ReadRegister((byte)(address + 1)));

            // reverse order of read values if msb is at lower address (= little endian)
            if (this.Endian == I2C6DBS.EndianSelection.DataMsbAtLowerAddress)
            {
                readResults.Reverse();
            }

            short result;
            try
            {
                result = BitConverter.ToInt16(readResults.ToArray(), 0);
            }
            catch
            {
                throw new InvalidOperationException("Das Auslesen der aktuellen Bewegungsdaten war nicht erfolgreich.");
            }

            return result;
        }

        /// <summary>
        /// Reads the current status from the device.
        /// </summary>
        /// <returns>Current status from the device.</returns>
        private AngularRateSensor.PositionStatus GetCurrentStatus()
        {
            AngularRateSensor.PositionStatus boolWrapper = new PositionStatus();
            byte status = this.ReadRegister((byte) AngularRateSensor.Register.Status);

            boolWrapper.OverrunZ = ((status & BitOperationUtils.Bit6) != 0) ? true : false;
            boolWrapper.OverrunY = ((status & BitOperationUtils.Bit5) != 0) ? true : false;
            boolWrapper.OverrunX = ((status & BitOperationUtils.Bit4) != 0) ? true : false;

            boolWrapper.NewDataZ = ((status & BitOperationUtils.Bit2) != 0) ? true : false;
            boolWrapper.NewDataY = ((status & BitOperationUtils.Bit1) != 0) ? true : false;
            boolWrapper.NewDataX = ((status & BitOperationUtils.Bit0) != 0) ? true : false;

            return boolWrapper;
        }

        /// <summary>
        /// Reads the chip id from the device.
        /// </summary>
        /// <returns>Chip id from the device.</returns>
        private byte GetChipID()
        {
            byte chipID = this.ReadRegister((byte) AngularRateSensor.Register.ChipID);
            return chipID;
        }

        /// <summary>
        /// Reads the current temperature from the device.
        /// </summary>
        /// <returns>Current temperature from the device.</returns>
        private int GetTemperature()
        {
            byte temperature = this.ReadRegister((byte) AngularRateSensor.Register.Temperature);

            return (int) temperature;
        }

        /// <summary>
        /// Sets a register of the device with a specific byte.
        /// </summary>
        /// <param name="register">Register to set.</param>
        /// <param name="data">Data to use.</param>
        private void WriteRegister(byte register, byte data)
        {
            string command = string.Format(
                "S{0} {1} {2} P",
                this.address.ToString("X2"),
                register.ToString("X2"),
                data.ToString("X2"));

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.UsbI2C.SendCommand(command);
        }

        /// <summary>
        /// Reads a register of the device.
        /// </summary>
        /// <param name="register">Wanted register.</param>
        /// <returns>Read value.</returns>
        private byte ReadRegister(byte register)
        {
            string command = string.Format(
                   "S{0} {1} R01 P",
                   this.address.ToString("X2"),
                   register.ToString("X2"));

            this.uiElements.ListBoxOutput.Items.Add(command);

            byte result;
            string read = this.UsbI2C.SendReceiveCommand(command);
            try
            {
                result = byte.Parse(read.Trim(), System.Globalization.NumberStyles.AllowHexSpecifier);
            }
            catch (FormatException)
            {
                this.uiElements.ListBoxInput.Items.Add(read);
                throw new InvalidOperationException();
            }

            this.uiElements.ListBoxInput.Items.Add(result.ToString("X2"));
            return result;
        }

        /// <summary>
        /// Gets the current angular rate of the I2C-6D-BS.
        /// </summary>
        /// <returns>Current angular rate of the I2C-6D-BS.</returns>
        private List<int> GetAngularRate()
        {
            List<int> result = new List<int>();
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.X));
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.Y));
            result.Add(this.GetSingleAxis(I2C6DBS.Axis.Z));

            return result;
        }

        /// <summary>
        /// Current status of the angular rate sensor.
        /// </summary>
        public struct PositionStatus
        {
            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the x-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the x-value.</value>
            public bool OverrunX { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the y-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the y-value.</value>
            public bool OverrunY { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the overrun flag for the z-value is set or not.
            /// </summary>
            /// <value>Overrun flag for the z-value.</value>
            public bool OverrunZ { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the x-value is set or not.
            /// </summary>
            /// <value>New data flag for the x-value.</value>
            public bool NewDataX { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the y-value is set or not.
            /// </summary>
            /// <value>New data flag for the y-value.</value>
            public bool NewDataY { get; set; }

            /// <summary>
            /// Gets or sets a value indicating whether the new data flag for the z-value is set or not.
            /// </summary>
            /// <value>New data flag for the z-value.</value>
            public bool NewDataZ { get; set; }
        }
    }
}
