﻿//-----------------------------------------------------------------------------
// <copyright file="Saa1064.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Class-representation of 4DLED-SAA1064.
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Text;
    using System.Windows.Forms;

    /// <summary>
    /// Class-Representation of 4DLED-SAA1064.
    /// </summary>
    public class Saa1064
        : AbstractDevice
    {
        /// <summary>
        /// Imported UI-Elements of SAA1064.
        /// </summary>
        private Saa1064.UiElements uiElements;

        /// <summary>
        /// Initializes a new instance of the Saa1064 class.
        /// </summary>
        /// <param name="usbI2C">I2C Comport-device.</param>
        public Saa1064(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.uiElements = new UiElements();
        }

        /// <summary>
        /// Sends command based on the UI to the MCP9801.
        /// </summary>
        public void SendCommand()
        {
            byte slaveAdressByte = this.GetSlaveAdressByte();
            byte instructionByte = 0x00;
            byte controlByte = this.GetControlByte();
            byte[] displayBytes = this.GetDisplayBytes();

            StringBuilder command = new StringBuilder();
            command.Append(String.Format("S{0} {1} {2}", slaveAdressByte.ToString("X2"), instructionByte.ToString("X2"), controlByte.ToString("X2")));
            foreach (byte b in displayBytes)
            {
                command.Append(String.Format(" {0}", b.ToString("X2")));
            }

            this.UsbI2C.SendCommand(command.ToString());

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
        }

        /// <summary>
        /// Imports the UI-Elements of the SAA1064.
        /// </summary>
        /// <param name="uiElements">UI-Elements of the SAA1064.</param>
        public void ImportUiElements(Saa1064.UiElements uiElements)
        {
            this.uiElements = uiElements;
        }

        /// <summary>
        /// Gets the slave adress byte based on the imported UI.
        /// </summary>
        /// <returns>Slave adress as byte.</returns>
        private byte GetSlaveAdressByte()
        {
            return byte.Parse(this.uiElements.CbSlaveAdress.SelectedItem.ToString(), System.Globalization.NumberStyles.AllowHexSpecifier);
        }

        /// <summary>
        /// Gets the control byte based on the imported UI.
        /// </summary>
        /// <returns>Control byte.</returns>
        private byte GetControlByte()
        {
            byte controlByte = 0x00;

            controlByte += 0x01;
            controlByte += 0x02;
            controlByte += 0x04;

            string current = this.uiElements.CbCurrent.SelectedItem.ToString().Split(' ')[0];

            switch (current)
            {
                case "0":
                    break;
                case "3":
                    controlByte += 0x10;
                    break;
                case "6":
                    controlByte += 0x20;
                    break;
                case "9":
                    controlByte += 0x10;
                    controlByte += 0x20;
                    break;
                case "12":
                    controlByte += 0x40;
                    break;
                case "15":
                    controlByte += 0x10;
                    controlByte += 0x40;
                    break;
                case "18":
                    controlByte += 0x20;
                    controlByte += 0x40;
                    break;
                case "21":
                    controlByte += 0x10;
                    controlByte += 0x20;
                    controlByte += 0x40;
                    break;
                default:
                    break;
            }

            return controlByte;
        }

        /// <summary>
        /// Gets the display bytes based on the imported UI.
        /// </summary>
        /// <returns>Display content as byte array.</returns>
        private byte[] GetDisplayBytes()
        {
            byte[] numbers = new byte[4];

            int counter = 0;
            foreach (PictureBox pb in this.uiElements.PbDisplay)
            {
                if (pb.Image != null)
                {
                    numbers[counter / 8] += (byte)Math.Pow(2, (counter % 8));
                }

                counter++;
            }

            return numbers;
        }

        /// <summary>
        /// UI-Elements of the MCP9801.
        /// </summary>
        public struct UiElements
        {
            /// <summary>
            /// Gets or sets slave adress.
            /// </summary>
            /// <value>Slave adress.</value>
            public ComboBox CbSlaveAdress { get; set; }

            /// <summary>
            /// Gets or sets the display content.
            /// </summary>
            /// <value>Display content.</value>
            public PictureBox[] PbDisplay { get; set; }

            /// <summary>
            /// Gets or sets control to set the wanted current.
            /// </summary>
            /// <value>Wanted current.</value>
            public ComboBox CbCurrent { get; set; }

            /// <summary>
            /// Gets or sets output-command list.
            /// </summary>
            /// <value>Output-command list.</value>
            public ListBox ListBoxOutput { get; set; }

            /// <summary>
            /// Gets or sets input-command list.
            /// </summary>
            /// <value>Input-command list.</value>
            public ListBox ListBoxInput { get; set; }
        }
    }
}