﻿//-----------------------------------------------------------------------------
// <copyright file="MCP9801.cs" company="eQ-3 Entwicklung GmbH">
//  Copyright (c) 2013 eQ-3 Entwicklung GmbH
// </copyright>
// <summary>
// Class-Representation of 4DLED-MCP9801
// </summary>
//-----------------------------------------------------------------------------
namespace Eq3.misc.USBI2C
{
    using System;
    using System.Drawing;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;

    /// <summary>
    /// Class-Representation of 4DLED-MCP9801.
    /// </summary>
    public class MCP9801
        : AbstractDevice
    {
        /// <summary>
        /// Value which indicates that the read temperature is not valid.
        /// </summary>
        public const double ErrorTemperature = 150.0;

        /// <summary>
        /// Imported UI-Elements of MCP9801.
        /// </summary>
        private MCP9801.UiElements uiElements;

        /// <summary>
        /// Initializes a new instance of the MCP9801 class.
        /// </summary>
        /// <param name="usbI2C">I2C Comport-device.</param>
        public MCP9801(UsbI2C usbI2C)
            : base(usbI2C)
        {
            this.uiElements = new UiElements();
        }

        /// <summary>
        /// Gets or sets read temperature.
        /// </summary>
        /// <value>Read temperature.</value>
        public double TemperatureCelsius { get; set; }

        /// <summary>
        /// Gets or sets read alert.
        /// </summary>
        /// <value>Read alert.</value>
        public double Alert { get; set; }

        /// <summary>
        /// Sends command based on the UI to the MCP9801.
        /// </summary>
        public void SendCommand()
        {
            this.HandleSettings();
            this.HandleTemperature();
            this.HandleAlert();
        }

        /// <summary>
        /// Imports the UI-Elements of the MCP9801.
        /// </summary>
        /// <param name="uiElements">UI-Elements of the MCP9801.</param>
        public void ImportUiElements(UiElements uiElements)
        {
            this.uiElements = uiElements;
        }

        /// <summary>
        /// Managing settings command in regard to the settings of the MCP9801.
        /// </summary>
        private void HandleSettings()
        {
            string command = String.Format("S{0} 01 {1} P", this.uiElements.ComboBoxMCP9801SlaveAddress.SelectedItem.ToString(), this.uiElements.SettingsByte.ToString("X2"));
            this.UsbI2C.SendCommand(command);

            this.uiElements.ListBoxOutput.Items.Add(command);
            this.uiElements.ListBoxOutput.TopIndex = this.uiElements.ListBoxOutput.Items.Count - 1;
        }

        /// <summary>
        /// Managing temperature command in regard to the settings of the MCP9801.
        /// </summary>
        private void HandleTemperature()
        {
            string command = String.Format("S{0} 00 R02 P", this.uiElements.ComboBoxMCP9801SlaveAddress.SelectedItem.ToString());

            string temperatureHex = this.UsbI2C.SendReceiveCommand(command.ToString());
            this.uiElements.ListBoxOutput.Items.Add(command);

            if (!this.CheckHexString(temperatureHex) || string.IsNullOrEmpty(temperatureHex))
            {
                this.uiElements.ListBoxInput.Items.Add("--> Err:TWI READ@Temperature");
                this.TemperatureCelsius = MCP9801.ErrorTemperature;
                return;
            }

            this.TemperatureCelsius = this.HexToTemperature(temperatureHex);
        }

        /// <summary>
        /// Checks whether a string is a valid Hexstring or not.
        /// </summary>
        /// <param name="alertHex">Hexstring as string.</param>
        /// <returns>Result of the action.</returns>
        private bool CheckHexString(string alertHex)
        {
            Regex regex = new Regex("^[0-9A-Fa-f ]*$");

            if (regex.IsMatch(alertHex))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Converts a Temperature as Hexstring to a double value.
        /// </summary>
        /// <param name="temperatureHexString">Temperature as Hexstring.</param>
        /// <returns>Temperature as double.</returns>
        private double HexToTemperature(string temperatureHexString)
        {
            int temperatureHex;

            try
            {
                temperatureHex = Int32.Parse(temperatureHexString.Replace(" ", String.Empty), System.Globalization.NumberStyles.HexNumber);
            }
            catch
            {
                return 0.0;
            }

            int temperatureInt;
            int readValue;
            int[] readData = new int[2];

            readData[0] = temperatureHex / 256;
            readData[1] = temperatureHex % 256;

            readValue = (readData[0] << 4) | (readData[1] >> 4);

            if ((readData[0] & 0x80) == 1)
            {
                readValue |= 0xf000;
            }

            temperatureInt = 625 * readValue;
            double result = (double)temperatureInt / 10000;

            return result;
        }

        /// <summary>
        /// Managing commands in regard of the alert of the MCP9801.
        /// </summary>
        private void HandleAlert()
        {
            this.WriteAlert();
            this.GetAlert();
        }

        /// <summary>
        /// Writes the alert to the MCP9801.
        /// </summary>
        private void WriteAlert()
        {
            string writeTemperatureString = this.uiElements.TbMCPAlarmValueWrite.Text;
            int externalDecimalPlaces;
            bool withDecimal = false;
            Regex regex = new Regex("^[-]?[0-9][0-9]?[0-9]?([.,][0-9])?$");

            if (writeTemperatureString == String.Empty)
            {
                return;
            }

            this.uiElements.TbMCPAlarmValueWrite.BackColor = Color.Red;
            if (!regex.IsMatch(writeTemperatureString))
            {
                this.ShowErrorTemperaturNotInRange(true);
                return;
            }

            if (!this.uiElements.RdMCPCelsius.Checked)
            {
                double parsedAlert = double.Parse(writeTemperatureString), convertedAlert;
                if (this.uiElements.RdMCPFahrenheit.Checked)
                {
                    convertedAlert = TemperatureUtils.ConvertFahrenheitToCelsius(parsedAlert);
                }
                else
                {
                    convertedAlert = TemperatureUtils.ConvertKelvinToCelsius(parsedAlert);
                }

                convertedAlert = Math.Round(convertedAlert, 0);
                writeTemperatureString = convertedAlert.ToString();
            }

            try
            {
                externalDecimalPlaces = Convert.ToInt32(writeTemperatureString);
            }
            catch
            {
                string[] temp = writeTemperatureString.Split(new char[] { ',', '.' });
                try
                {
                    externalDecimalPlaces = Convert.ToInt32(temp[0]);
                    if (temp[1] == "5")
                    {
                        withDecimal = true;
                    }
                    else if (temp[1] == "0")
                    {
                        withDecimal = false;
                    }
                    else
                    {
                        throw new InvalidOperationException();
                    }
                }
                catch
                {
                    this.ShowErrorTemperaturNotInRange(true);
                    return;
                }
            }

            if (externalDecimalPlaces > 127 || externalDecimalPlaces < -127)
            {
                this.ShowErrorTemperaturNotInRange(true);
                return;
            }

            this.ShowErrorTemperaturNotInRange(false);
            this.uiElements.TbMCPAlarmValueWrite.BackColor = Color.White;

            if (externalDecimalPlaces < 0)
            {
                externalDecimalPlaces = Math.Abs(externalDecimalPlaces);
                externalDecimalPlaces += 0x80;
            }

            string command = String.Format(
                "S{0} 03 {1} {2} P",
                this.uiElements.ComboBoxMCP9801SlaveAddress.SelectedItem.ToString(),
                externalDecimalPlaces.ToString("X2"),
                withDecimal ? "80" : "00");

            this.UsbI2C.SendCommand(command);
            this.uiElements.ListBoxOutput.Items.Add(command);
        }

        /// <summary>
        /// Gets the alert from the MCP9801.
        /// </summary>
        private void GetAlert()
        {
            string command = String.Format("S{0} 03 R02 P", this.uiElements.ComboBoxMCP9801SlaveAddress.SelectedItem.ToString());
            string alertHex = this.UsbI2C.SendReceiveCommand(command.ToString());
            this.uiElements.ListBoxOutput.Items.Add(command);

            if (!this.CheckHexString(alertHex) || string.IsNullOrEmpty(alertHex) || alertHex == " ")
            {
                this.uiElements.ListBoxInput.Items.Add("--> Err:TWI READ@Alert");
                return;
            }

            int alertHexInt = Int32.Parse(alertHex.Replace(" ", String.Empty), System.Globalization.NumberStyles.HexNumber);
            byte[] bytes = new byte[2] { (byte)(alertHexInt / 256), (byte)(alertHexInt % 256) };
            bool isSigned;

            this.Alert = 0.0;

            isSigned = ((bytes[0] & 0x80) != 0) ? true : false;
            this.Alert += ((bytes[0] & 0x40) != 0) ? 64 : 0;
            this.Alert += ((bytes[0] & 0x20) != 0) ? 32 : 0;
            this.Alert += ((bytes[0] & 0x10) != 0) ? 16 : 0;

            this.Alert += ((bytes[0] & 0x08) != 0) ? 8 : 0;
            this.Alert += ((bytes[0] & 0x04) != 0) ? 4 : 0;
            this.Alert += ((bytes[0] & 0x02) != 0) ? 2 : 0;
            this.Alert += ((bytes[0] & 0x01) != 0) ? 1 : 0;

            this.Alert += ((bytes[1] & 0x80) != 0) ? 0.5 : 0;

            if (isSigned)
            {
                this.Alert *= -1;
            }

            this.uiElements.ListBoxInput.Items.Add(alertHex);
        }

        /// <summary>
        /// Shows a error in a message box if the temperature is invalid.
        /// </summary>
        /// <param name="isShowing">Indicates whether a error should be shown or not.</param>
        private void ShowErrorTemperaturNotInRange(bool isShowing)
        {
            const double MaxTop = 127.5;
            const double MaxBottom = -127.5;

            if (isShowing)
            {
                double top, bottom;
                string measureUnit;

                if (this.uiElements.RdMCPCelsius.Checked)
                {
                    top = MaxTop;
                    bottom = MaxBottom;

                    measureUnit = "°C";
                }
                else if (this.uiElements.RdMCPFahrenheit.Checked)
                {
                    top = Math.Round(TemperatureUtils.ConvertCelsiusToFahrenheit(MaxTop), 0, MidpointRounding.ToEven) - 1.0;
                    bottom = Math.Round(TemperatureUtils.ConvertCelsiusToFahrenheit(MaxBottom), 0, MidpointRounding.AwayFromZero) + 1.0;

                    measureUnit = "°F";
                }
                else
                {
                    top = Math.Round(TemperatureUtils.ConvertCelsiusToKelvin(MaxTop), 0, MidpointRounding.ToEven) - 1.0;
                    bottom = Math.Round(TemperatureUtils.ConvertCelsiusToKelvin(MaxBottom), 0, MidpointRounding.AwayFromZero);

                    measureUnit = "K";
                }

                this.uiElements.LbMCPAlarmValueError.Text = string.Format(
                    "Der zu schreibende Alarmwert muss zwischen {0:0.0}{2} und {1:0.0}{2} liegen{3}.",
                    bottom,
                    top,
                    measureUnit,
                    this.uiElements.RdMCPCelsius.Checked ? " (0,5-Schritte, eine Nachkommastelle)" : string.Empty);
            }
            else
            {
                this.uiElements.LbMCPAlarmValueError.Text = string.Empty;
            }
        }

        /// <summary>
        /// UI-Elements of the MCP9801.
        /// </summary>
        public struct UiElements
        {
            /// <summary>
            /// Gets or sets slave adress.
            /// </summary>
            /// <value>Slave adress.</value>
            public ComboBox ComboBoxMCP9801SlaveAddress { get; set; }

            /// <summary>
            /// Gets or sets settings byte.
            /// </summary>
            /// <value>Settings byte.</value>
            public byte SettingsByte { get; set; }

            /// <summary>
            /// Gets or sets read alarm value.
            /// </summary>
            /// <value>Read alarm value.</value>
            public TextBox TbMCPAlarmValue { get; set; }

            /// <summary>
            /// Gets or sets alarm value to write.
            /// </summary>
            /// <value>Alarm value to write.</value>
            public TextBox TbMCPAlarmValueWrite { get; set; }

            /// <summary>
            /// Gets or sets the control element wheter celsius should be temperature measure unit or not.
            /// </summary>
            /// <value>Control element wheter celsius should be temperature measure unit or not.</value>
            public RadioButton RdMCPCelsius { get; set; }

            /// <summary>
            /// Gets or sets the control element wheter fahrenheit should be temperature measure unit or not.
            /// </summary>
            /// <value>Control element wheter fahrenheit should be temperature measure unit or not.</value>
            public RadioButton RdMCPFahrenheit { get; set; }

            /// <summary>
            /// Gets or sets the label for showing possible errors while writing the alert.
            /// </summary>
            /// <value>Label for showing possible errors while writing the alert.</value>
            public Label LbMCPAlarmValueError { get; set; }

            /// <summary>
            /// Gets or sets output-command list.
            /// </summary>
            /// <value>Output-command list.</value>
            public ListBox ListBoxOutput { get; set; }

            /// <summary>
            /// Gets or sets input-command list.
            /// </summary>
            /// <value>Input-command list.</value>
            public ListBox ListBoxInput { get; set; }
        }
    }
}
