# -*- coding: utf-8 -*-

"""
USB-ESI3 -> MQTT Publisher mit Home Assistant Discovery
- Liest Zeilen im Format:
  "Connector X:[electricity|gas];key=value;key=value;..."
- Publiziert pro Connector JSON-Zustand
- Legt Sensor-Entitäten via MQTT Discovery automatisch in Home Assistant an
- Alle Konfigurationsparameter sind im Script definiert (kein CLI nötig)
- Robuste Beendigung: SIGINT/SIGTERM werden sauber behandelt (systemd-tauglich)
"""

import re
import json
import time
import logging
import signal
import threading
from typing import Dict, Tuple, Optional

import serial                # pyserial
from paho.mqtt import client as mqtt_client

# -------------------------
# Konfiguration
# -------------------------

#______________USB-ESI3 Settings________________

SERIAL_PORT = "/dev/ttyUSB0"  # stabiler udev-Symlink empfohlen
BAUDRATE = 115200
SERIAL_TIMEOUT = 1.0               # Sekunden

MQTT_HOST = "192.168.1.10"
MQTT_PORT = 1883
MQTT_USER = "youruser"
MQTT_PASS = "yourpass"

#___________END USB-ESI3 Settings____________

DEVICE_NAME = "USB-ESI3"
DEVICE_ID = "usb-esi3"
BASE_TOPIC = "sensors/usb-esi3"           # Basis
STATUS_TOPIC = f"{BASE_TOPIC}/status"     # Availability (online/offline)
QOS = 1
RETAIN_STATE = True
RETAIN_DISCOVERY = True

UNITS_ELECTRICITY = {
    "power": "W",
    "energy_import": "kWh",
    "energy_export": "kWh",
    "energy_import_nt": "kWh",
}
UNITS_GAS = {
    "volume_import": "m³",
    "momentary_use": "m³",  # ggf. anpassen
}

# HA-Metadaten konservativ gewählt
HA_META = {
    "electricity": {
        "power": {"device_class": "power", "state_class": "measurement"},
        "energy_import": {"device_class": "energy", "state_class": "total_increasing"},
        "energy_export": {"device_class": "energy", "state_class": "total_increasing"},
        "energy_import_nt": {"device_class": "energy", "state_class": "total_increasing"},
    },
    "gas": {
        "volume_import": {"state_class": "total_increasing"},
        "momentary_use": {"state_class": "measurement"},
    },
}

# Logging
logging.basicConfig(level=logging.INFO, format="%(asctime)s %(levelname)s: %(message)s")
log = logging.getLogger("usb_esi3")

# Regex: "Connector X:TYPE; rest"
HEADER_RE = re.compile(
    r"^\s*Connector\s+(\d+)\s*:\s*(electricity|gas)\s*;?\s*(.*)$",
    re.IGNORECASE
)

# Stop-Event (für sauberes Beenden)
stop_event = threading.Event()

# Cache für bereits publizierte Discovery-Keys
_discovered: Dict[Tuple[int, str], Dict[str, bool]] = {}


# -------------------------
# Parser
# -------------------------

def parse_line(line: str) -> Optional[Tuple[int, str, Dict[str, float]]]:
    """
    Parst eine Zeile vom USB-ESI3:
    Rückgabe: (connector_index, meter_type, data_dict) oder None
    - meter_type: 'electricity' oder 'gas'
    - data_dict: vorhandene key=value Paare (fehlende Keys fehlen einfach)
    """
    line = line.strip()
    if not line:
        return None

    m = HEADER_RE.match(line)
    if not m:
        return None

    conn_idx = int(m.group(1))
    meter_type = m.group(2).lower()
    tail = m.group(3) or ""

    data: Dict[str, float] = {}
    # Schlüssel/Wert-Paare sind mit ';' getrennt
    parts = [p.strip() for p in tail.split(";") if p.strip()]
    for part in parts:
        if "=" not in part:
            continue
        key, val = part.split("=", 1)
        key = key.strip()
        val = val.strip()
        try:
            num = float(val)
            data[key] = num
        except ValueError:
            data[key] = val

    return (conn_idx, meter_type, data)


# -------------------------
# MQTT-Client
# -------------------------

def make_mqtt_client(client_id: str) -> mqtt_client.Client:
    client = mqtt_client.Client(mqtt_client.CallbackAPIVersion.VERSION1, client_id=client_id, clean_session=True)
    if MQTT_USER:
        client.username_pw_set(MQTT_USER, MQTT_PASS or "")

    # Last Will: offline
    client.will_set(STATUS_TOPIC, payload="offline", qos=QOS, retain=True)

    def on_connect(c, userdata, flags, rc):
        if rc == 0:
            log.info("MQTT verbunden")
            c.publish(STATUS_TOPIC, "online", qos=QOS, retain=True)
        else:
            log.error(f"MQTT Verbindungsfehler rc={rc}")

    def on_disconnect(c, userdata, rc):
        log.warning(f"MQTT getrennt rc={rc}")

    client.on_connect = on_connect
    client.on_disconnect = on_disconnect
    client.connect(MQTT_HOST, MQTT_PORT, keepalive=30)
    return client


# -------------------------
# Home Assistant Discovery
# -------------------------

def device_info() -> Dict:
    return {
        "identifiers": [DEVICE_ID],
        "name": DEVICE_NAME,
        "manufacturer": "USB-ESI3",
        "model": "Energy/Gas Serial Interface",
    }

def publish_discovery_for_keys(
    client: mqtt_client.Client,
    conn_idx: int,
    meter_type: str,
    keys: Dict[str, float]
) -> None:
    """
    Publiziert HA-Discovery-Payloads für neu erschienene Keys dieses Connectors.
    """
    state_topic = f"{BASE_TOPIC}/connector/{conn_idx}/state"
    published_for_conn = _discovered.setdefault((conn_idx, meter_type), {})

    unit_map = UNITS_ELECTRICITY if meter_type == "electricity" else UNITS_GAS

    for key in keys.keys():
        if published_for_conn.get(key):
            continue

        unique_id = f"{DEVICE_ID}_conn{conn_idx}_{key}"
        discovery_topic = f"homeassistant/sensor/{unique_id}/config"

        payload = {
            "name": f"{DEVICE_NAME} {meter_type} Kanal {conn_idx} {key}",
            "unique_id": unique_id,
            "state_topic": state_topic,
            "value_template": f"{{{{ value_json.{key} }}}}",
            "availability_topic": STATUS_TOPIC,
            "payload_available": "online",
            "payload_not_available": "offline",
            "device": device_info(),
        }

        unit = unit_map.get(key)
        if unit:
            payload["unit_of_measurement"] = unit

        meta = HA_META.get(meter_type, {}).get(key, {})
        payload.update(meta)

        client.publish(discovery_topic, json.dumps(payload), qos=QOS, retain=RETAIN_DISCOVERY)
        log.info(f"Discovery publiziert: {discovery_topic}")
        published_for_conn[key] = True


# -------------------------
# Serielle Öffnung (mit einfacher Wiederhollogik)
# -------------------------

def open_serial() -> Optional[serial.Serial]:
    try:
        ser = serial.Serial(port=SERIAL_PORT, baudrate=BAUDRATE, timeout=SERIAL_TIMEOUT)
        log.info(f"Serieller Port geöffnet: {ser.port} @ {ser.baudrate}")
        return ser
    except Exception as e:
        log.error(f"Serieller Port {SERIAL_PORT} kann nicht geöffnet werden: {e}")
        return None


# -------------------------
# Hauptlogik
# -------------------------

def run_loop() -> None:
    client = make_mqtt_client(client_id=f"{DEVICE_ID}_publisher")
    client.loop_start()

    ser = open_serial()
    last_serial_attempt = time.time()

    try:
        while not stop_event.is_set():
            # Falls Serial gerade nicht offen ist: zyklisch versuchen
            if ser is None and (time.time() - last_serial_attempt) >= 3.0:
                last_serial_attempt = time.time()
                ser = open_serial()

            if ser is None:
                time.sleep(0.2)
                continue

            try:
                raw_bytes = ser.readline()
                if not raw_bytes:
                    continue

                raw = raw_bytes.decode(errors="ignore").strip()
                if not raw:
                    continue

                parsed = parse_line(raw)
                if not parsed:
                    # optional: log.debug(f"Ignored: {raw}")
                    continue

                conn_idx, meter_type, data = parsed
                if not data:
                    continue

                # Discovery für neue Keys
                publish_discovery_for_keys(client, conn_idx, meter_type, data)

                # State publizieren
                state_topic = f"{BASE_TOPIC}/connector/{conn_idx}/state"
                payload = json.dumps(data)
                res = client.publish(state_topic, payload, qos=QOS, retain=RETAIN_STATE)
                rc = res[0] if isinstance(res, tuple) else getattr(res, "rc", 0)
                if rc == 0:
                    log.info(f"MQTT [{state_topic}] -> {payload}")
                else:
                    log.error(f"Publish fehlgeschlagen rc={rc}")

                # kurze Pause
                time.sleep(0.05)

            except serial.SerialException as e:
                log.error(f"Serielle Ausnahme: {e}")
                try:
                    ser.close()
                except Exception:
                    pass
                ser = None
                time.sleep(0.5)

    finally:
        # Sauber beenden
        try:
            if ser is not None:
                ser.close()
        except Exception:
            pass
        client.loop_stop()
        client.disconnect()
        log.info("Beendet: Ressourcen freigegeben.")


def main():
    # Signale für sauberes Beenden (systemd)
    def _handle_signal(sig, frame):
        log.info(f"Signal empfangen: {sig} -> Stop")
        stop_event.set()

    signal.signal(signal.SIGINT, _handle_signal)
    signal.signal(signal.SIGTERM, _handle_signal)

    run_loop()


# ------------- WICHTIG: Einstiegspunkt -------------
if __name__ == "__main__":
    main()
