/**
* @file scale.c
* @brief Source file for 24Bit-ADC functions.
* @author Marcel Maas, eQ-3 Entwicklung GmbH
**/
/*-----------------------------------------------------------------------------
* Copyright 2012 eQ-3 Entwicklung GmbH
* All Rights Reserved.
*
* The information contained in this file is confidential property of "eQ-3
* Entwicklung GmbH". The use copying, transfer or disclosure is prohibited
* except by written agreement with "eQ-3 Entwicklung GmbH".
*-----------------------------------------------------------------------------*/
/*
* $URL:$
* $Author:      Marcel Maas$
* $Date:        08.05.2025$
* $Revision:    1$
*/
/*----------------------------------------------------------------------------*/

// Includes --------------------------------------------------------------------
#include "scale.h"

#include "Arduino.h"
#include "MAX11210.h"

// Definitions -----------------------------------------------------------------
// Typedefs --------------------------------------------------------------------

// Variables -------------------------------------------------------------------
double d_scale_value    = 1.0f;
int64_t s_offset_value  = 0;

// Prototypes ------------------------------------------------------------------

// Exported functions ----------------------------------------------------------
bool scale_init( void )
{
  bool b_ret = false;  
  
#if SCALE_ADC == SCALE_MAX11210
  max11210_init();
  b_ret = max11210_copro_device_id( COPRO_ADDR );

  max11210_copro_set_input_range( MAX11210_BIPOLAR, COPRO_ADDR );
  delay( 100 );
  max11210_copro_set_format( MAX11210_2SCOMPL, COPRO_ADDR );         // value = { MAX11210_OFFSET, MAX11210_2SCOMPL }
  delay( 100 );
  max11210_set_gain( MAX11210_GAIN16, COPRO_ADDR );
  delay( 100 );
#else
  // ...
#endif
  
  return b_ret;
}

void scale_set_scale( double d_scale )
{
	d_scale_value = d_scale;
}

double scale_get_scale( void )
{
  return d_scale_value;
}

void scale_tare( void )
{
  double d_sum = scale_read_average( SCALE_READ_AVGM_TRIM_MODE );
  
	scale_set_offset( ( int64_t )d_sum );
}

void scale_calibrate( uint32_t u32_weight_in_grams )
{
  double d_calibration_factor = 0;
  double d_weight_in_kg = 0;
  
  if( u32_weight_in_grams < 1000 )
  {
    return;
  }
  
  d_weight_in_kg = u32_weight_in_grams / 1000.0;
  
  d_calibration_factor = ( scale_read_average( SCALE_READ_AVGM_TRIM_MEAN ) - s_offset_value ) / d_weight_in_kg;
  
  scale_set_scale( d_calibration_factor );
}

int64_t scale_read_average( scale_read_avg_mode_t mode )
{
  uint32_t u32_i;
  int32_t s32_data[SCALE_SAMPLES] = { 0 };
  int64_t s64_value = 0;
  
  for( u32_i = 0; u32_i < SCALE_SAMPLES; u32_i++ )
  {
#if SCALE_ADC == SCALE_MAX11210
    max11210_copro_meas_uv( COPRO_ADDR );
    delay( 100 );
    s32_data[u32_i] = max11210_copro_get_uv( COPRO_ADDR );
    delay( 10 );
#else
    s32_data[u32_i] = 0;
#endif
  }

  switch( mode )
  {
    case SCALE_READ_AVGM_TRIM_MEAN:
    {
      s64_value = scale_read_average_trim_mean( s32_data );
    }
    break;
    
    case SCALE_READ_AVGM_TRIM_MODE:
    {
      s64_value = scale_read_average_trim_mode( s32_data );
    }
    break;

    default:
    {
      s64_value = scale_read_average_trim_mean( s32_data );
      
      if( ( ( s64_value - s_offset_value ) / d_scale_value ) <= 1.0f )
      {
        s64_value = scale_read_average( SCALE_READ_AVGM_TRIM_MODE );
      }
    }
    break;
  }

  return s64_value;
}

int64_t scale_read_average_trim_mean( int32_t *s32_adc_data )
{
  uint32_t u32_i, u32_j;
  int64_t s64_total_sum  = 0;
  int32_t s32_data[SCALE_SAMPLES] = { 0 };
  bool b_excluded[SCALE_SAMPLES] = { false };
  int64_t s64_trimmed_sum = 0;

  // 1) make local copy
  for( uint32_t u32_i = 0; u32_i < SCALE_SAMPLES; u32_i++ )
  {
    s32_data[u32_i] = s32_adc_data[u32_i];
  }

  // 2) Compute total sum of all values
  for( u32_i = 0; u32_i < SCALE_SAMPLES; u32_i++ )
  {
    s64_total_sum += s32_data[u32_i];
  }
//  APP_LOG( TS_OFF, VLEVEL_L, "s64_total_sum: %d\r\n", s64_total_sum );
  
  // 3) Identify and exclude min/max values for trimming
  for( uint32_t u32_i = 0; u32_i < SCALE_TRIM_COUNT; u32_i++ )
  {
    // Find minimum unmarked value
    int32_t s32_min_idx = -1;

    for( u32_j = 0; u32_j < SCALE_SAMPLES; u32_j++ )
    {
      if( b_excluded[u32_j] )
      {
        continue;
      }
      if( s32_min_idx < 0 || s32_data[u32_j] < s32_data[s32_min_idx] )
      {
          s32_min_idx = u32_j;
      }
    }

    b_excluded[s32_min_idx] = true;
    s64_trimmed_sum += s32_data[s32_min_idx];

    // Find maximum unmarked value
    int32_t s32_max_idx = -1;

    for( u32_j = 0; u32_j < SCALE_SAMPLES; u32_j++ )
    {
      if( b_excluded[u32_j] )
      {
        continue;
      }
      if( s32_max_idx < 0 || s32_data[u32_j] > s32_data[s32_max_idx] )
      {
          s32_max_idx = u32_j;
      }
    }

    b_excluded[s32_max_idx] = true;
    s64_trimmed_sum += s32_data[s32_max_idx];
  }
  
  // 4) Compute trimmed mean
  uint32_t u32_valid_count = SCALE_SAMPLES - 2 * SCALE_TRIM_COUNT;

  if( u32_valid_count == 0 )
  {
    return 0; // avoid divide-by-zero
  }

  return ( int32_t )( ( s64_total_sum - s64_trimmed_sum ) / u32_valid_count );
}

int64_t scale_read_average_trim_mode( int32_t *s32_adc_data )
{
  uint32_t u32_i;
  int32_t s32_buf[SCALE_SAMPLES] = { 0 };
  
  // 1) make local copy
  for( u32_i = 0; u32_i < SCALE_SAMPLES; u32_i++ )
  {
    s32_buf[u32_i] = s32_adc_data[u32_i];
  }
  
  // 2) sort the copy
  s32_qsort( s32_buf, SCALE_SAMPLES );
  
  // 3) define trimmed range [start .. end)
  uint32_t u32_start = SCALE_TRIM_COUNT;
  uint32_t u32_end   = SCALE_SAMPLES - SCALE_TRIM_COUNT; // exclusive
  
  // 4) scan for mode in the trimmed range
  int32_t  s32_mode       = s32_buf[u32_start];
  uint32_t u32_best_count = 1;
  uint32_t u32_curr_count = 1;

  for( uint32_t u32_i = u32_start + 1; u32_i < u32_end; u32_i++ )
  {
    if( s32_buf[u32_i] == s32_buf[u32_i - 1] )
    {
      u32_curr_count++;
    }
    else
    {
      // check if the run we just closed is the best so far
      if( u32_curr_count > u32_best_count )
      {
        u32_best_count = u32_curr_count;
        s32_mode       = s32_buf[u32_i - 1];
      }

      u32_curr_count = 1;
    }
  }
  
  // final run at the end of the loop
  if( u32_curr_count > u32_best_count )
  {
    s32_mode = s32_buf[u32_end - 1];
  }
  
  return s32_mode;
}

void scale_set_offset( int64_t s64_offset )
{
	s_offset_value = s64_offset;
}

int64_t scale_get_offset( void )
{
  return s_offset_value;
}

double scale_get_units( uint8_t u8_times )
{
  double d_value = 0;
  
  d_value = scale_get_value( u8_times ) / d_scale_value;
  
  if( d_value < 0 )
  {
    d_value = 0;
  }
  
  return d_value;
}

double scale_get_value( uint8_t u8_times )
{
	return scale_read_average( SCALE_READ_AVGM_DEFAULT ) - s_offset_value;
}

void s32_qsort_range( int32_t *s32_data, int32_t s32_lo, int32_t s32_hi )
{
  if( s32_lo >= s32_hi )
  {
    return;
  }

  // choose pivot as middle element
  int32_t s32_pivot = s32_data[( s32_lo + s32_hi ) / 2];
  int32_t s32_i = s32_lo;
  int32_t s32_j = s32_hi;

  // partition loop
  while( s32_i <= s32_j )
  {
    while( s32_data[s32_i] < s32_pivot )
    {
      s32_i++;
    }
    
    while( s32_data[s32_j] > s32_pivot )
    {
      s32_j--;
    }

    if( s32_i <= s32_j )
    {
      // swap s32_data[s32_i] and s32_data[s32_j]
      int32_t s32_tmp = s32_data[s32_i];
      s32_data[s32_i] = s32_data[s32_j];
      s32_data[s32_j] = s32_tmp;
      s32_i++;
      s32_j--;
    }
  }

  // recursively sort the two partitions
  if( s32_lo < s32_j )
  {
    s32_qsort_range( s32_data, s32_lo, s32_j );
  }

  if( s32_i < s32_hi )
  {
    s32_qsort_range( s32_data, s32_i, s32_hi );
  }
}

void s32_qsort( int32_t *s32_data, uint32_t u32_count )
{
  if( u32_count <= 1 )
  {
    return;
  }

  s32_qsort_range( s32_data, 0, ( int32_t )u32_count - 1 );
}