#pragma once
#include <cstdint>
#include <cstddef> // size_t

class MyWire {
public:
    // Function pointer types
    using BeginFunc       = void (*)();
    using BeginTransFunc  = void (*)(uint8_t);
    using WriteFunc       = size_t (*)(uint8_t);
    using EndTransFunc    = uint8_t (*)();
    using RequestFunc     = uint8_t (*)(uint8_t, uint8_t);
    using ReadFunc        = int (*)();
    using SetClockFunc    = void (*)(uint32_t);
    using AvailableFunc   = int (*)();

    // Initialize MyWire with function pointers
    void init(BeginFunc b,
              BeginTransFunc bt,
              WriteFunc w,
              EndTransFunc et,
              RequestFunc r,
              ReadFunc rd,
              SetClockFunc sc,
              AvailableFunc av)
    {
        beginFunc      = b;
        beginTransFunc = bt;
        writeFunc      = w;
        endTransFunc   = et;
        requestFunc    = r;
        readFunc       = rd;
        setClockFunc   = sc;
        availableFunc  = av;
    }

    // Wrapper functions
    void begin() { if(beginFunc) beginFunc(); }
    void beginTransmission(uint8_t addr) { if(beginTransFunc) beginTransFunc(addr); }
    size_t write(uint8_t data) { return writeFunc ? writeFunc(data) : 0; }
    uint8_t endTransmission() { return endTransFunc ? endTransFunc() : 0; }
    uint8_t requestFrom(uint8_t addr, uint8_t qty) { return requestFunc ? requestFunc(addr, qty) : 0; }
    int read() { return readFunc ? readFunc() : -1; }
    void setClock(uint32_t c) { if(setClockFunc) setClockFunc(c); }
    int available() { return availableFunc ? availableFunc() : 0; }

private:
    BeginFunc      beginFunc      = nullptr;
    BeginTransFunc beginTransFunc = nullptr;
    WriteFunc      writeFunc      = nullptr;
    EndTransFunc   endTransFunc   = nullptr;
    RequestFunc    requestFunc    = nullptr;
    ReadFunc       readFunc       = nullptr;
    SetClockFunc   setClockFunc   = nullptr;
    AvailableFunc  availableFunc  = nullptr;
};

// Global extern object
extern MyWire MyWireObj;
