/**
* @file MAX11210.h
* @brief Header file for 24Bit-ADC functions.
* @author Marcel Maas, eQ-3 Entwicklung GmbH
**/
/*-----------------------------------------------------------------------------
* Copyright 2012 eQ-3 Entwicklung GmbH
* All Rights Reserved.
*
* The information contained in this file is confidential property of "eQ-3
* Entwicklung GmbH". The use copying, transfer or disclosure is prohibited
* except by written agreement with "eQ-3 Entwicklung GmbH".
*----------------------------------------------------------------------------*/
/*
* $URL:$
* $Author:      Marcel Maas$
* $Date:        22.05.2025$
* $Revision:    1$
*/

// Define to prevent recursive inclusion ---------------------------------------
#ifndef __MAX11210_H
#define __MAX11210_H

#ifdef __cplusplus
 extern "C" {
#endif

// Includes --------------------------------------------------------------------
#include <stdbool.h>
#include <stdint.h>

// Definitions -----------------------------------------------------------------
#define COPRO_ADDR_1        ( 0x14 >> 1 )
#define COPRO_ADDR_2        ( 0x16 >> 1 )
#define COPRO_ADDR_3        ( 0x18 >> 1 )
#define COPRO_ADDR_4        ( 0x1A >> 1 )

#define COPRO_ADDR          COPRO_ADDR_1

#define MAX11210_COPRO_ID   0x0ADC

// CTRL1
#define MAX11210_50HZ               0x01
#define MAX11210_60HZ               0x00
#define MAX11210_UNIPOLAR           0x01
#define MAX11210_BIPOLAR            0x00
#define MAX11210_EXTCLK             0x01
#define MAX11210_INTCLK             0x00
#define MAX11210_OFFSET             0x01
#define MAX11210_2SCOMPL            0x00
#define MAX11210_SINGLE             0x01
#define MAX11210_CONT               0x00

// CTRL3
#define MAX11210_GAIN1              0x00
#define MAX11210_GAIN2              0x01
#define MAX11210_GAIN4              0x02
#define MAX11210_GAIN8              0x03
#define MAX11210_GAIN16             0x04

// Exported types --------------------------------------------------------------
typedef enum
{
  I2C_UNDEF_REG                   = 0x00,

  I2C_ID_REG,
  
  I2C_MAX11210_VREF_MV_SET_REG,
  I2C_MAX11210_VREF_MV_GET_REG,

  I2C_MAX11210_MEAS_UV_REG,
  I2C_MAX11210_GET_UV_REG,

  I2C_MAX11210_MEAS_RAW_REG,
  I2C_MAX11210_GET_RAW_REG,

  I2C_MAX11210_SELF_CAL_REG,
  I2C_MAX11210_SYS_OFFSET_CAL_REG,
  I2C_MAX11210_SYS_GAIN_CAL_REG,

  I2C_MAX11210_READ_SYS_GAIN_OVER_RANGE_REG,
  I2C_MAX11210_READ_RATE_REG,
  I2C_MAX11210_READ_OVER_RANGE_REG,
  I2C_MAX11210_READ_UNDER_RANGE_REG,
  I2C_MAX11210_READ_MEAS_STAT_REG,
  I2C_MAX11210_READ_READY_REG,

  I2C_MAX11210_GET_SYS_GAIN_OVER_RANGE_REG,
  I2C_MAX11210_GET_RATE_REG,
  I2C_MAX11210_GET_OVER_RANGE_REG,
  I2C_MAX11210_GET_UNDER_RANGE_REG,
  I2C_MAX11210_GET_MEAS_STAT_REG,
  I2C_MAX11210_GET_READY_REG,

  I2C_MAX11210_SET_LINE_FREQ_REG,
  I2C_MAX11210_SET_INPUT_RANGE_REG,
  I2C_MAX11210_SET_CLOCK_SOURCE_REG,
  I2C_MAX11210_SET_ENABLE_REF_BUF_REG,
  I2C_MAX11210_SET_ENABLE_SIG_BUF_REG,
  I2C_MAX11210_SET_FORMAT_REG,
  I2C_MAX11210_SET_CONV_MODE_REG,

  I2C_MAX11210_READ_LINE_FREQ_REG,
  I2C_MAX11210_READ_INPUT_RANGE_REG,
  I2C_MAX11210_READ_CLOCK_SOURCE_REG,
  I2C_MAX11210_READ_ENABLE_REF_BUF_REG,
  I2C_MAX11210_READ_ENABLE_SIG_BUF_REG,
  I2C_MAX11210_READ_FORMAT_REG,
  I2C_MAX11210_READ_CONV_MODE_REG,

  I2C_MAX11210_GET_LINE_FREQ_REG,
  I2C_MAX11210_GET_INPUT_RANGE_REG,
  I2C_MAX11210_GET_CLOCK_SOURCE_REG,
  I2C_MAX11210_GET_ENABLE_REF_BUF_REG,
  I2C_MAX11210_GET_ENABLE_SIG_BUF_REG,
  I2C_MAX11210_GET_FORMAT_REG,
  I2C_MAX11210_GET_CONV_MODE_REG,

  I2C_MAX11210_SET_GAIN_REG,
  I2C_MAX11210_SET_DISABLE_SYS_GAIN_REG,
  I2C_MAX11210_SET_DISABLE_SYS_OFFSET_REG,
  I2C_MAX11210_SET_DISABLE_SELF_CAL_GAIN_REG,
  I2C_MAX11210_SET_DISABLE_SELF_CAL_OFFSET_REG,
  
  I2C_MAX11210_READ_GAIN_REG,
  I2C_MAX11210_READ_DISABLE_SYS_GAIN_REG,
  I2C_MAX11210_READ_DISABLE_SYS_OFFSET_REG,
  I2C_MAX11210_READ_DISABLE_SELF_CAL_GAIN_REG,
  I2C_MAX11210_READ_DISABLE_SELF_CAL_OFFSET_REG,
  
  I2C_MAX11210_GET_GAIN_REG,
  I2C_MAX11210_GET_DISABLE_SYS_GAIN_REG,
  I2C_MAX11210_GET_DISABLE_SYS_OFFSET_REG,
  I2C_MAX11210_GET_DISABLE_SELF_CAL_GAIN_REG,
  I2C_MAX11210_GET_DISABLE_SELF_CAL_OFFSET_REG,

  I2C_MAX11210_SET_SYS_GAIN_CAL_REG,
  I2C_MAX11210_SET_SYS_OFFSET_CAL_REG,
  I2C_MAX11210_SET_SELF_CAL_GAIN_REG,
  I2C_MAX11210_SET_SELF_CAL_OFFSET_REG,
  
  I2C_MAX11210_READ_SYS_GAIN_CAL_REG,
  I2C_MAX11210_READ_SYS_OFFSET_CAL_REG,
  I2C_MAX11210_READ_SELF_CAL_GAIN_REG,
  I2C_MAX11210_READ_SELF_CAL_OFFSET_REG,
  
  I2C_MAX11210_GET_SYS_GAIN_CAL_REG,
  I2C_MAX11210_GET_SYS_OFFSET_CAL_REG,
  I2C_MAX11210_GET_SELF_CAL_GAIN_REG,
  I2C_MAX11210_GET_SELF_CAL_OFFSET_REG,
} I2C_REG_t;

// Exported macro --------------------------------------------------------------

// Exported functions ----------------------------------------------------------
void max11210_init( void );
void max11210_reset_buffers( void );

void max11210_copro_wakeup( uint8_t u8_slave_addr );
bool max11210_copro_device_id( uint8_t u8_slave_addr );

void max11210_copro_set_vref_mv( uint16_t u16_vref_mv, uint8_t u8_slave_addr );
uint16_t max11210_copro_get_vref_mv( uint8_t u8_slave_addr );

void max11210_copro_meas_uv( uint8_t u8_slave_addr );
int32_t max11210_copro_get_uv( uint8_t u8_slave_addr );

void max11210_copro_meas_raw( uint8_t u8_slave_addr );
uint32_t max11210_copro_get_raw( uint8_t u8_slave_addr );

// CMD MODE 0
void max11210_copro_self_cal( uint8_t u8_slave_addr );
void max11210_copro_sys_offset_cal( uint8_t u8_slave_addr );
void max11210_copro_sys_gain_cal( uint8_t u8_slave_addr );

// STAT1
void max11210_copro_read_sys_gain_over_range( uint8_t u8_slave_addr );
void max11210_copro_read_rate( uint8_t u8_slave_addr );
void max11210_copro_read_over_range( uint8_t u8_slave_addr );
void max11210_copro_read_under_range( uint8_t u8_slave_addr );
void max11210_copro_read_meas_stat( uint8_t u8_slave_addr ); 
void max11210_copro_read_ready( uint8_t u8_slave_addr );

bool max11210_copro_get_sys_gain_over_range( uint8_t u8_slave_addr );
uint8_t max11210_copro_get_rate( uint8_t u8_slave_addr );
bool max11210_copro_get_over_range( uint8_t u8_slave_addr );
bool max11210_copro_get_under_range( uint8_t u8_slave_addr );
bool max11210_copro_get_meas_stat( uint8_t u8_slave_addr ); 
bool max11210_copro_get_ready( uint8_t u8_slave_addr );

// CTRL1
void max11210_copro_set_line_freq( uint8_t u8_value, uint8_t u8_slave_addr );      // value = { MAX11210_50HZ, MAX11210_60HZ }
void max11210_copro_set_input_range( uint8_t u8_value, uint8_t u8_slave_addr );    // value = { MAX11210_UNIPOLAR, MAX11210_BIPOLAR }
void max11210_copro_set_clock_source( uint8_t u8_value, uint8_t u8_slave_addr );   // value = { MAX11210_EXTCLK, MAX11210_INTCLK }
void max11210_copro_set_enable_ref_buf( bool b_state, uint8_t u8_slave_addr );
void max11210_copro_set_enable_sig_buf( bool b_state, uint8_t u8_slave_addr );
void max11210_copro_set_format( uint8_t u8_value, uint8_t u8_slave_addr );         // value = { MAX11210_OFFSET, MAX11210_2SCOMPL }
void max11210_copro_set_conv_mode( uint8_t u8_value, uint8_t u8_slave_addr );      // value = { MAX11210_SINGLE, MAX11210_CONT }

void max11210_copro_read_line_freq( uint8_t u8_slave_addr );
void max11210_copro_read_input_range( uint8_t u8_slave_addr );
void max11210_copro_read_clock_source( uint8_t u8_slave_addr );
void max11210_copro_read_enable_ref_buf( uint8_t u8_slave_addr );
void max11210_copro_read_enable_sig_buf( uint8_t u8_slave_addr );
void max11210_copro_read_format( uint8_t u8_slave_addr );
void max11210_copro_read_conv_mode( uint8_t u8_slave_addr );

uint8_t max11210_copro_get_line_freq( uint8_t u8_slave_addr );
uint8_t max11210_copro_get_input_range( uint8_t u8_slave_addr );
uint8_t max11210_copro_get_clock_source( uint8_t u8_slave_addr );
bool max11210_copro_get_enable_ref_buf( uint8_t u8_slave_addr );
bool max11210_copro_get_enable_sig_buf( uint8_t u8_slave_addr );
uint8_t max11210_copro_get_format( uint8_t u8_slave_addr );
uint8_t max11210_copro_get_conv_mode( uint8_t u8_slave_addr );

// CTRL3
void max11210_set_gain( uint8_t u8_gain, uint8_t u8_slave_addr );
void max11210_set_disable_sys_gain( bool b_state, uint8_t u8_slave_addr );
void max11210_set_disable_sys_offset( bool b_state, uint8_t u8_slave_addr );
void max11210_set_disable_self_cal_gain( bool b_state, uint8_t u8_slave_addr );
void max11210_set_disable_self_cal_offset( bool b_state, uint8_t u8_slave_addr );

void max11210_read_gain( uint8_t u8_slave_addr );
void max11210_read_disable_sys_gain( uint8_t u8_slave_addr );
void max11210_read_disable_sys_offset( uint8_t u8_slave_addr );
void max11210_read_disable_self_cal_gain( uint8_t u8_slave_addr );
void max11210_read_disable_self_cal_offset( uint8_t u8_slave_addr );

uint8_t max11210_get_gain( uint8_t u8_slave_addr );
bool max11210_get_disable_sys_gain( uint8_t u8_slave_addr );
bool max11210_get_disable_sys_offset( uint8_t u8_slave_addr );
bool max11210_get_disable_self_cal_gain( uint8_t u8_slave_addr );
bool max11210_get_disable_self_cal_offset( uint8_t u8_slave_addr );

void max11210_set_sys_gain_cal( uint32_t u32_value, uint8_t u8_slave_addr );
void max11210_set_sys_offset_cal( uint32_t u32_value, uint8_t u8_slave_addr );
void max11210_set_self_cal_gain( uint32_t u32_value, uint8_t u8_slave_addr );
void max11210_set_self_cal_offset( uint32_t u32_value, uint8_t u8_slave_addr );

void max11210_read_sys_gain_cal( uint8_t u8_slave_addr );
void max11210_read_sys_offset_cal( uint8_t u8_slave_addr );
void max11210_read_self_cal_gain( uint8_t u8_slave_addr );
void max11210_read_self_cal_offset( uint8_t u8_slave_addr );

uint32_t max11210_get_sys_gain_cal( uint8_t u8_slave_addr );
uint32_t max11210_get_sys_offset_cal( uint8_t u8_slave_addr );
uint32_t max11210_get_self_cal_gain( uint8_t u8_slave_addr );
uint32_t max11210_get_self_cal_offset( uint8_t u8_slave_addr );

#ifdef __cplusplus
}
#endif

/**
  * @}
  */

#endif /* __MAX11210_H */

