/**
* @file MAX11210.c
* @brief Source file for 24Bit-ADC functions.
* @author Marcel Maas, eQ-3 Entwicklung GmbH
**/
/*-----------------------------------------------------------------------------
* Copyright 2012 eQ-3 Entwicklung GmbH
* All Rights Reserved.
*
* The information contained in this file is confidential property of "eQ-3
* Entwicklung GmbH". The use copying, transfer or disclosure is prohibited
* except by written agreement with "eQ-3 Entwicklung GmbH".
*-----------------------------------------------------------------------------*/
/*
* $URL:$
* $Author:      Marcel Maas$
* $Date:        22.05.2025$
* $Revision:    1$
*/
/*----------------------------------------------------------------------------*/

/* Includes ------------------------------------------------------------------*/
#include "MAX11210.h"
#include <cstring>
#include "MyWire.h"

// Definitions -----------------------------------------------------------------
#define Wire MyWireObj
// Typedefs --------------------------------------------------------------------

// Variables -------------------------------------------------------------------
uint8_t u8_copro_rx_buffer[5] = { 0 };
uint8_t u8_copro_tx_buffer[5] = { 0 };
uint8_t u8_copro_trash_buffer[1] = { 0 };

// Prototypes ------------------------------------------------------------------

// Exported functions ----------------------------------------------------------
void max11210_init( void )
{
  Wire.begin();
  Wire.setClock( 100000 );
}

void max11210_reset_buffers( void )
{
  std::memset( u8_copro_rx_buffer, 0, 5 );
  std::memset( u8_copro_tx_buffer, 0, 5 );
  std::memset( u8_copro_trash_buffer, 0, 1 );
}

void max11210_copro_wakeup( uint8_t u8_slave_addr )
{
  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    Wire.read();
  }
}

bool max11210_copro_device_id( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint16_t u16_copro_id = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_ID_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 2 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }

  u16_copro_id = ( ( u8_copro_rx_buffer[1] << 8 )	| ( u8_copro_rx_buffer[0] ) );

  u8_copro_rx_buffer[0] = 0;
  u8_copro_rx_buffer[1] = 0;

  if( u16_copro_id == MAX11210_COPRO_ID )
  {
    return true;
  }

  return false;
}

void max11210_copro_set_vref_mv( uint16_t u16_vref_mv, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  u8_copro_tx_buffer[0] = ( uint8_t )( u16_vref_mv >> 8 );
  u8_copro_tx_buffer[1] = ( uint8_t )( u16_vref_mv >> 0 );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_VREF_MV_SET_REG );
  Wire.write( u8_copro_tx_buffer[0] );
  Wire.write( u8_copro_tx_buffer[1] );
  Wire.endTransmission();

  max11210_reset_buffers();
}

uint16_t max11210_copro_get_vref_mv( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint16_t u16_vref_mv_val = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_VREF_MV_GET_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 2 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u16_vref_mv_val = ( int32_t )( ( u8_copro_rx_buffer[0] << 8 )	| ( u8_copro_rx_buffer[1] ) );
  
  max11210_reset_buffers();

  return u16_vref_mv_val;
}

void max11210_copro_meas_uv( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_MEAS_UV_REG );
  Wire.endTransmission();
}

int32_t max11210_copro_get_uv( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  int32_t s32_max11210_uv = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_UV_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }

  uint32_t u32_raw24 =  ( ( uint32_t )u8_copro_rx_buffer[0] << 16 ) |
                    ( ( uint32_t )u8_copro_rx_buffer[1] << 8 ) |
                    ( ( uint32_t )u8_copro_rx_buffer[2] );

  // shift left to move sign bit (bit23) into bit31, then arithmetic shift right
  s32_max11210_uv = ( int32_t )( u32_raw24 << 8 ) >> 8;

  max11210_reset_buffers();

  return s32_max11210_uv;
}

void max11210_copro_meas_raw( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_MEAS_RAW_REG );
  Wire.endTransmission();
}

uint32_t max11210_copro_get_raw( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint32_t u32_max11210_raw = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_RAW_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u32_max11210_raw = ( int32_t )( ( u8_copro_rx_buffer[0] << 16 ) | ( u8_copro_rx_buffer[1] << 8 )	| (u8_copro_rx_buffer[2] ) );

  max11210_reset_buffers();

  return u32_max11210_raw;
}

// CMD MODE 0
void max11210_copro_self_cal( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SELF_CAL_REG );
  Wire.endTransmission();
}

void max11210_copro_sys_offset_cal( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SYS_OFFSET_CAL_REG );
  Wire.endTransmission();
}

void max11210_copro_sys_gain_cal( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SYS_GAIN_CAL_REG );
  Wire.endTransmission();
}

// STAT1
void max11210_copro_read_sys_gain_over_range( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_SYS_GAIN_OVER_RANGE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_rate( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_RATE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_over_range( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_OVER_RANGE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_under_range( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_UNDER_RANGE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_meas_stat( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_MEAS_STAT_REG );
  Wire.endTransmission();
}

void max11210_copro_read_ready( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_READY_REG );
  Wire.endTransmission();
}

bool max11210_copro_get_sys_gain_over_range( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool b_max11210_sys_gain_over_range = false;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_SYS_GAIN_OVER_RANGE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  b_max11210_sys_gain_over_range = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return b_max11210_sys_gain_over_range;
}

uint8_t max11210_copro_get_rate( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_rate = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_RATE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_rate = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_rate;
}

bool max11210_copro_get_over_range( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool b_max11210_over_range = false;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_OVER_RANGE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  b_max11210_over_range = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return b_max11210_over_range;
}

bool max11210_copro_get_under_range( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool b_max11210_under_range = false;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_UNDER_RANGE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  b_max11210_under_range = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return b_max11210_under_range;
}

bool max11210_copro_get_meas_stat( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool b_max11210_meas_stat = false;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_MEAS_STAT_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  b_max11210_meas_stat = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return b_max11210_meas_stat;
}

bool max11210_copro_get_ready( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool b_max11210_ready = false;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_READY_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  b_max11210_ready = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return b_max11210_ready;
}

// CTRL1
void max11210_copro_set_line_freq( uint8_t u8_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_LINE_FREQ_REG );
  Wire.write( u8_value );
  Wire.endTransmission();
}

void max11210_copro_set_input_range( uint8_t u8_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_INPUT_RANGE_REG );
  Wire.write( u8_value );
  Wire.endTransmission();
}

void max11210_copro_set_clock_source( uint8_t u8_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_CLOCK_SOURCE_REG );
  Wire.write( u8_value );
  Wire.endTransmission();
}

void max11210_copro_set_enable_ref_buf( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_ENABLE_REF_BUF_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_copro_set_enable_sig_buf( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_ENABLE_SIG_BUF_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_copro_set_format( uint8_t u8_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_FORMAT_REG );
  Wire.write( u8_value );
  Wire.endTransmission();
  // uint8_t u8_err = Wire.endTransmission();

  // Serial.println( u8_err );
}

void max11210_copro_set_conv_mode( uint8_t u8_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_CONV_MODE_REG );
  Wire.write( u8_value );
  Wire.endTransmission();
}

void max11210_copro_read_line_freq( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_LINE_FREQ_REG );
  Wire.endTransmission();
}

void max11210_copro_read_input_range( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_INPUT_RANGE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_clock_source( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_CLOCK_SOURCE_REG );
  Wire.endTransmission();
}

void max11210_copro_read_enable_ref_buf( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_ENABLE_REF_BUF_REG );
  Wire.endTransmission();
}

void max11210_copro_read_enable_sig_buf( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_ENABLE_SIG_BUF_REG );
  Wire.endTransmission();
}

void max11210_copro_read_format( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_FORMAT_REG );
  Wire.endTransmission();
}

void max11210_copro_read_conv_mode( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_CONV_MODE_REG );
  Wire.endTransmission();
}

uint8_t max11210_copro_get_line_freq( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_line_freq = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_LINE_FREQ_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_line_freq = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_line_freq;
}

uint8_t max11210_copro_get_input_range( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_input_range = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_INPUT_RANGE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_input_range = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_input_range;
}

uint8_t max11210_copro_get_clock_source( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_clock_source = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_CLOCK_SOURCE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_clock_source = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_clock_source;
}

bool max11210_copro_get_enable_ref_buf( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_enable_ref_buf = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_ENABLE_REF_BUF_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_enable_ref_buf = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_enable_ref_buf;
}

bool max11210_copro_get_enable_sig_buf( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_enable_sig_buf = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_ENABLE_SIG_BUF_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_enable_sig_buf = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_enable_sig_buf;
}

uint8_t max11210_copro_get_format( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_format = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_FORMAT_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_format = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_format;
}

uint8_t max11210_copro_get_conv_mode( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_conv_mode = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_CONV_MODE_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_conv_mode = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_conv_mode;
}

// CTRL3
void max11210_set_gain( uint8_t u8_gain, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_GAIN_REG );
  Wire.write( u8_gain );
  Wire.endTransmission();
}

void max11210_set_disable_sys_gain( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_DISABLE_SYS_GAIN_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_set_disable_sys_offset( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_DISABLE_SYS_OFFSET_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_set_disable_self_cal_gain( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_DISABLE_SELF_CAL_GAIN_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_set_disable_self_cal_offset( bool b_state, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_DISABLE_SELF_CAL_OFFSET_REG );
  Wire.write( ( uint8_t )b_state );
  Wire.endTransmission();
}

void max11210_read_gain( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_GAIN_REG );
  Wire.endTransmission();
}

void max11210_read_disable_sys_gain( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_DISABLE_SYS_GAIN_REG );
  Wire.endTransmission();
}

void max11210_read_disable_sys_offset( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_DISABLE_SYS_OFFSET_REG );
  Wire.endTransmission();
}

void max11210_read_disable_self_cal_gain( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_DISABLE_SELF_CAL_GAIN_REG );
  Wire.endTransmission();
}

void max11210_read_disable_self_cal_offset( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_DISABLE_SELF_CAL_OFFSET_REG );
  Wire.endTransmission();
}

uint8_t max11210_get_gain( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint8_t u8_max11210_gain = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_GAIN_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_gain = ( uint8_t )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_gain;
}

bool max11210_get_disable_sys_gain( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_sys_gain = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_DISABLE_SYS_GAIN_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_sys_gain = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_sys_gain;
}

bool max11210_get_disable_sys_offset( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_sys_offset = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_DISABLE_SYS_OFFSET_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_sys_offset = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_sys_offset;
}

bool max11210_get_disable_self_cal_gain( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_self_cal_gain = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_DISABLE_SELF_CAL_GAIN_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_self_cal_gain = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_self_cal_gain;
}

bool max11210_get_disable_self_cal_offset( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  bool u8_max11210_self_cal_offset = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_DISABLE_SELF_CAL_OFFSET_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 1 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u8_max11210_self_cal_offset = ( bool )( u8_copro_rx_buffer[0] );

  max11210_reset_buffers();

  return u8_max11210_self_cal_offset;
}

void max11210_set_sys_gain_cal( uint32_t u32_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  u8_copro_tx_buffer[0] = ( uint8_t )( u32_value >> 16 );
  u8_copro_tx_buffer[1] = ( uint8_t )( u32_value >> 8 );
  u8_copro_tx_buffer[2] = ( uint8_t )( u32_value >> 0 );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_SYS_GAIN_CAL_REG );
  Wire.write( u8_copro_tx_buffer[0] );
  Wire.write( u8_copro_tx_buffer[1] );
  Wire.write( u8_copro_tx_buffer[2] );
  Wire.endTransmission();

  max11210_reset_buffers();
}

void max11210_set_sys_offset_cal( uint32_t u32_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  u8_copro_tx_buffer[0] = ( uint8_t )( u32_value >> 16 );
  u8_copro_tx_buffer[1] = ( uint8_t )( u32_value >> 8 );
  u8_copro_tx_buffer[2] = ( uint8_t )( u32_value >> 0 );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_SYS_OFFSET_CAL_REG );
  Wire.write( u8_copro_tx_buffer[0] );
  Wire.write( u8_copro_tx_buffer[1] );
  Wire.write( u8_copro_tx_buffer[2] );
  Wire.endTransmission();

  max11210_reset_buffers();
}

void max11210_set_self_cal_gain( uint32_t u32_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  u8_copro_tx_buffer[0] = ( uint8_t )( u32_value >> 16 );
  u8_copro_tx_buffer[1] = ( uint8_t )( u32_value >> 8 );
  u8_copro_tx_buffer[2] = ( uint8_t )( u32_value >> 0 );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_SELF_CAL_GAIN_REG );
  Wire.write( u8_copro_tx_buffer[0] );
  Wire.write( u8_copro_tx_buffer[1] );
  Wire.write( u8_copro_tx_buffer[2] );
  Wire.endTransmission();

  max11210_reset_buffers();
}

void max11210_set_self_cal_offset( uint32_t u32_value, uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  u8_copro_tx_buffer[0] = ( uint8_t )( u32_value >> 16 );
  u8_copro_tx_buffer[1] = ( uint8_t )( u32_value >> 8 );
  u8_copro_tx_buffer[2] = ( uint8_t )( u32_value >> 0 );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_SET_SELF_CAL_OFFSET_REG );
  Wire.write( u8_copro_tx_buffer[0] );
  Wire.write( u8_copro_tx_buffer[1] );
  Wire.write( u8_copro_tx_buffer[2] );
  Wire.endTransmission();

  max11210_reset_buffers();
}

void max11210_read_sys_gain_cal( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_SYS_GAIN_CAL_REG );
  Wire.endTransmission();
}

void max11210_read_sys_offset_cal( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_SYS_OFFSET_CAL_REG );
  Wire.endTransmission();
}

void max11210_read_self_cal_gain( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_SELF_CAL_GAIN_REG );
  Wire.endTransmission();
}

void max11210_read_self_cal_offset( uint8_t u8_slave_addr )
{
  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_READ_SELF_CAL_OFFSET_REG );
  Wire.endTransmission();
}

uint32_t max11210_get_sys_gain_cal( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint32_t u32_sys_gain_cal = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_SYS_GAIN_CAL_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u32_sys_gain_cal = ( uint32_t )( ( u8_copro_rx_buffer[0] << 16 )	| ( u8_copro_rx_buffer[1] << 8 ) | ( u8_copro_rx_buffer[2] ) );
  
  max11210_reset_buffers();

  return u32_sys_gain_cal;
}

uint32_t max11210_get_sys_offset_cal( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint32_t u32_sys_offset_cal = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_SYS_OFFSET_CAL_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u32_sys_offset_cal = ( uint32_t )( ( u8_copro_rx_buffer[0] << 16 )	| ( u8_copro_rx_buffer[1] << 8 ) | ( u8_copro_rx_buffer[2] ) );
  
  max11210_reset_buffers();

  return u32_sys_offset_cal;
}

uint32_t max11210_get_self_cal_gain( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint32_t u32_self_cal_gain = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_SELF_CAL_GAIN_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u32_self_cal_gain = ( uint32_t )( ( u8_copro_rx_buffer[0] << 16 )	| ( u8_copro_rx_buffer[1] << 8 ) | ( u8_copro_rx_buffer[2] ) );
  
  max11210_reset_buffers();

  return u32_self_cal_gain;
}

uint32_t max11210_get_self_cal_offset( uint8_t u8_slave_addr )
{
  uint8_t u8_i = 0;
  uint32_t u32_self_cal_offset = 0;

  max11210_copro_wakeup( u8_slave_addr );

  Wire.beginTransmission( u8_slave_addr );
  Wire.write( I2C_MAX11210_GET_SELF_CAL_OFFSET_REG );
  Wire.endTransmission();

  Wire.requestFrom( u8_slave_addr, 3 );
  while( Wire.available() )
  {
    u8_copro_rx_buffer[u8_i] = Wire.read();
    u8_i++;
  }
  
  u32_self_cal_offset = ( uint32_t )( ( u8_copro_rx_buffer[0] << 16 )	| ( u8_copro_rx_buffer[1] << 8 ) | ( u8_copro_rx_buffer[2] ) );
  
  max11210_reset_buffers();

  return u32_self_cal_offset;
}
