
/****************************************************************************************/
/* Multi functional clock display with Franzis Prezelboard                              */
/* Version: 1.0                                                                         */
/* Date: 17.05.2016                                                                     */
/* Compiled with Arduino 1.6.5                                                          */
/*                                                                                      */
/*                                                                                      */
/* Pin asignments:                                                                      */
/* Soft Serial interface for Nextion LCD display:                                       */
/*    Transmit TX = Pin 2                                                               */
/*    Receive RX = Pin 3                                                                */
/* Soft Serial interface for onboard ESP8266 WLAN chip:                                 */
/*    Transmit TX = Pin 11                                                              */
/*    Receive RX = Pin 12                                                               */
/*                                                                                      */
/*                                                                                      */
/* Screen codes for screen 1:                                                           */
/*             p0 = FÜNF                                                                */
/*             p1 = ZEHN                                                                */
/*             P2 = VOR                                                                 */
/*             p3 = NACH                                                                */
/*             p4 = HALB                                                                */
/*             p5 = VIERTEL                                                             */
/*             p6 = VOR                                                                 */
/*             p7 = NACH                                                                */
/*             p8 = EIN                                                                 */
/*             p9 = EINS                                                                */
/*             p10 = ZWEI                                                               */
/*             p11 = DREI                                                               */
/*             p12 = VIER                                                               */
/*             p13 = FÜNF                                                               */
/*             p14 = SECHS                                                              */
/*             p15 = SIEBEN                                                             */
/*             p16 = ACHT                                                               */
/*             p17 = NEUN                                                               */
/*             p18 = ZEHN                                                               */
/*             p19 = ELF                                                                */
/*             p20 = ZWÖLF                                                              */
/*             p21 = UHR                                                                */
/*             p22 = PUNKT 1                                                            */
/*             p23 = PUNKT 2                                                            */
/*             p24 = PUNKT 3                                                            */
/*             p25 = PUNKT 4                                                            */
/*             p26 = WLAN Icon                                                          */
/*             p27 = NTP Icon                                                           */
/* Replaced Serial.print routine for less memory usage:                                 */
/*     http://www.utopiamechanicus.com/article/low-memory-serial-print/                 */
/*                                                                                      */
/****************************************************************************************/


/****************************************************************************************/
/* Libraries                                                                            */
/****************************************************************************************/
#include <Time.h>                                 /* Uses time library */
#include <SoftwareSerial.h>                       /* Uses softserial library */
#include <Nextion.h>                              /* Uses Nextion library */
#include <EEPROM.h>                               /* Include EEPROM read/write */

/****************************************************************************************/
/* General WLAN settings                                                                */
/****************************************************************************************/
#define SSID "xxx"                                /* SSID of WLAN network */
#define PASSWORD "xxx"                            /* Password of WLAN network */
#define DEBUG true                                /* True = Display degub informations, False = Do not dispaly debug informations */                               

/****************************************************************************************/
/* Constants                                                                            */
/****************************************************************************************/
const byte NTP_PACKET_SIZE = 48;                  /* NTP time is in the first 48 bytes of message payload */
const byte RECEIVE_DELAY = 500;                   /* 500ms receive delay in receiving NTP packets */

/****************************************************************************************/
/* Variables                                                                            */
/****************************************************************************************/
boolean WLAN_Connect;                             /* True = WLAN connected, False = WLAN not connected */
boolean NTP_Connect;                              /* True = NTP received OK, False = NTP not received OK */
boolean Daily_Update;                             /* True = Clock has to be daily updated, False = Clock was already updated */
boolean WLAN_Update;                              /* True = Check WLAN connection, False = Do not check WLAN connection */
boolean SLEEP_MODE;                               /* True = Device in sleep mode, False = Device not in sleep mode */
byte WLAN_Connect_Error;                          /* Reason why WLAN connection failed */
byte NTP_Connect_Error;                           /* Reason why NTP connection failed */
byte Loop_Counter;                                /* Loop counter */
byte Test_Pos_1;                                  /* Test position 1 for general calculation */
byte Test_Pos_2;                                  /* Test position 2 general calculation */
byte NTP_Time_Zone;                               /* Variable for time zone: 0 = UTC time, 1 = CET winter time, 2 = CET summer time */
byte TRIES;                                       /* Number of tries */
byte ACT_SCREEN;                                  /* Actual screen */
byte BEL_INT;                                     /* Light intensity 0-100 */ 
byte BEL_PERIOD;                                  /* Light timeout 0-11 */
int ACT_SEC;                                      /* Actual Second */
String Packet_Buffer = "";                        /* String to receive and send data */
String Temp_Buffer = "";                          /* String for string manipulation */
String Nextion_Message = "";                      /* Received string from Nextion display */

/****************************************************************************************/
/* Definitions                                                                          */
/****************************************************************************************/
SoftwareSerial nextion(2, 3);                     /* Define softserial interface for LCD display TX = Pin 2, RX = Pin 3 */
Nextion myNextion(nextion, 9600);                 /* Create a Nextion object named myNextion using the nextion serial port @ 9600bps */
SoftwareSerial esp8266(11, 12);                   /* Define softserial interface for ESP8266 WLAN chip TX = Pin 11, RX = Pin 12 */


/****************************************************************************************/
/* Program memory version of print - Copy of format string and result share a buffer    */
/* so as to avoid too much memory use                                                   */    
/****************************************************************************************/
void SUB_STREAM_PRINT_PROGMEM(Print &out,PGM_P format,...)
{
  char formatString[128], *ptr;
  
  strncpy_P( formatString, format, sizeof(formatString) );
  formatString[ sizeof(formatString)-2 ]='\0'; 
  ptr=&formatString[ strlen(formatString)+1 ];
  va_list args;
  va_start (args,format);
  vsnprintf(ptr, sizeof(formatString)-1-strlen(formatString), formatString, args );
  va_end (args);
  formatString[ sizeof(formatString)-1 ]='\0'; 
  out.print(ptr);
}

/* Definition for substituted Serial print routine for printing out of Flash */
#define SUB_SERIAL_PRINT(format, ...) SUB_STREAM_PRINT_PROGMEM(Serial,PSTR(format),##__VA_ARGS__)


/****************************************************************************************/
/* Setup procedure                                                                      */
/****************************************************************************************/
void setup() 
{  
  Packet_Buffer.reserve(NTP_PACKET_SIZE);         /* Reserve byte buffer for packet buffer string */  
  Temp_Buffer.reserve (20);                       /* Reserve byte buffer for temporary string */
  Nextion_Message.reserve (30);                   /* Reserve byte buffer for Nextion receive string */
  
  /* Init Nextion display */
  myNextion.init();                               /* Send the initialization commands for display of page 0 */
  SUB_Clock_Clear();                              /* Call subroutine to clear all clock display data */
    
  /* Init serial interface */  
  if (DEBUG)Serial.begin(19200);                  /* Start serial interface for terminal communication at 19200 Baud */
  /* Display welcome message with initialisation information */
  if (DEBUG)SUB_SERIAL_PRINT ("WLAN Client for NTP-watch with Nextion display\n\n");    


  /* Init WLAN chip ESP8266 */
  esp8266.begin(19200);                           /* Set the data rate for the software serial port to 19200 */
  nextion.begin(9600);                            /* Set the data rate for the software serial port to 9600 */
  
  /* Adjust brightness */
  BEL_INT = EEPROM.read(0);                       /* Read brightness out of EEPROM */
  if (BEL_INT > 100)                              /* Invalid value ? */
  {
    BEL_INT = 50;                                 /* Set BEL_INT = 50 */
    EEPROM.write(0, BEL_INT);                     /* Save brightness value in EEPROM */  
  }
  SUB_ADJUST_BRIGHT();                           /* Send actual brightness value to display */ 

  /* Adjust timeout */
  SUB_NEXTION_SEND (F("thup=1"));                 /* Init wake up on touch */
  BEL_PERIOD = EEPROM.read(1);                    /* Read timeout out of EEPROM */
  if (BEL_PERIOD > 11)                            /* Invalid value ? */
  {  
    BEL_PERIOD = 0;                               /* Set BEL_PERIOD = 0 */ 
    EEPROM.write(1, BEL_PERIOD);                  /* Save timeout value in EEPROM */    
  } 
  delay (5000);                                   /* 5 Seconds delay */
  SUB_NEXTION_SEND (F("page Uhr1"));              /* Display screen 1 */
    
  esp8266.listen();                               /* Listen to ESP chip */
  
  /* Connect to WLAN */
  do                                              /* Repeat until connected to WLAN --> WLAN_Connect = true */
  {
    SUB_WLAN_CONNECT();                           /* Call sub routine to connect to WLAN */
  }
  while (WLAN_Connect == false);                  /* Error received ? */

  /* Receive NTP time */
  do                                              /* Repeat until NTP time received --> NTP_Connect = true */
  {  
    SUB_NTP_Receive();                            /* Call subroutine to receive NTP time */  
  }
  while (NTP_Connect == false);                   /* Error received ? */

  SUB_CLOCK_1_DISPLAY();                          /* Call sub routine to update clock display on Nextion display */
  SUB_CLOCK_PRINT ();                             /* Call sub routine to send clock and date via serial interface to terminal */
  Daily_Update = true;                            /* Set daily flag false */
  WLAN_Update = true;                             /* Set WLAN update flag true */
  ACT_SCREEN = 0;                                 /* Actual screen = 0 */
  nextion.listen();                               /* Listen to Netion display */
  SUB_ADJUST_TIMEOUT ();                          /* Call Subroutine to adjust timeout */         
  SLEEP_MODE = false;                             /* Set sleep mode false */
}


/****************************************************************************************/
/* Main Loop                                                                            */
/****************************************************************************************/
void loop() 
{     

  /* Clear buffer */
  while (nextion.available() > 0)
  {
    Test_Pos_1 = nextion.read();                  /* Read next byte */
  }
  nextion.print("sendme");                        /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display to terminate string */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  delay (50);                                     /* Wait 50ms for answer */
  Nextion_Message = myNextion.listen();           /* Check for pressed buttons */
  if(Nextion_Message != "")                       /* Message received ? */
  {
    /* Screen 1 pressed and actual screen is 1 */
    if ((Nextion_Message == "2") && (ACT_SCREEN == 0))
    {
      SUB_CLOCK_2_DISPLAY ();                      /* Display clock 2 */
      ACT_SCREEN = 1;                              /* Set actual screen to 1 */
    }
    /* Screen 2 pressed and actual screen is 2 */
    if ((Nextion_Message == "1") && (ACT_SCREEN == 1))
    {
      SUB_CLOCK_1_DISPLAY ();                      /* Display clock 1 */       
      ACT_SCREEN = 0;                              /* Set actual screen to 0 */
    }
    /* Screen 2 pressed --> Config Menu */
    if (Nextion_Message == "3")
    {
      SUB_CONFIG_MENU ();                          /* Display config menu */  
      SUB_CLOCK_1_DISPLAY ();                      /* Display clock 1 */       
      ACT_SCREEN = 0;                              /* Set actual screen to 0 */
    }
    /* Wake up from sleep screen 0 */
    if ((Nextion_Message == "1") && (SLEEP_MODE == true))
    {
      SUB_Clock_Clear();                           /* Clear display 1 */
      SUB_CLOCK_1_DISPLAY ();                      /* Display clock 1 */
      SLEEP_MODE = false;                          /* Set sleep mode false */
    }
  } 
  else                                             /* No receiving function */
  {
    SLEEP_MODE = true;                             /* Set sleep mode true */
  }

  if (ACT_SEC != second())                         /* New second ? */
  {
    ACT_SEC = second ();                           /* Save second */
    if ((ACT_SEC == 0) && (ACT_SCREEN == 0))
    {
      SUB_CLOCK_1_DISPLAY();                       /* Call sub routine to update clock display on Nextion display */
      if (DEBUG)SUB_CLOCK_PRINT ();                /* Call sub routine to send clock and date via serial interface to terminal */
    }    
    if(ACT_SCREEN == 1)
    {
      SUB_CLOCK_2_DISPLAY();                       /* Call sub routine to update clock display on Nextion display */
      if (DEBUG)SUB_CLOCK_PRINT ();                /* Call sub routine to send clock and date via serial interface to terminal */
    }      
  }

  /* Update NTP Server at four o'clock in the morning = Hour equals four and minute equals two and not already updated ? */
  if ((hour() == 04) && (minute() == 02) && (Daily_Update == true))
  {
    esp8266.listen();                             /* Listen to ESP chip */
    TRIES = 0;                                    /* Set to frist try */
    do
    {
      SUB_NTP_Receive();                          /* Call subroutine to receive NTP time */
      TRIES++;                                    /* Increase tries */
    }
    /* Repreat until success or number of tries = 3 */
    while ((NTP_Connect == false) && (TRIES < 3));

    if (NTP_Connect == true)                      /* No error ? */
    {
      SUB_CLOCK_1_DISPLAY();                      /* Call sub routine to update clock display on Nextion display */
      SUB_CLOCK_PRINT ();                         /* Call sub routine to send clock and date via serial interface to terminal */      
    }

    Daily_Update = false;                         /* Set daily update false */ 
    nextion.listen();                             /* Listen to Netion display */
  } 
  /* Reset NTP Update flag for next update */
  if (minute() == 10)                             /* Reset clock update after 10 minutes */
  {
    Daily_Update = true;                          /* Set daily flag false */
  }  

  /* Check WLAN signal every 10 minutes */
  if ((minute () % 10 == 0) && (second() == 0) &&  (WLAN_Update == true))
  { 
    esp8266.listen();                             /* Listen to ESP chip */    
    /* Check for right WLAN connection */
    WLAN_Connect = SUB_ESP_SEND_CHECK("AT+CWJAP?", SSID);
    WLAN_Update = false;                          /* Set WLAN update flag true */
    if (WLAN_Connect == false)                    /* Error received ? */
    { 
      SUB_WLAN_CONNECT();                         /* Call sub routine to connect to WLAN */        
    }
    nextion.listen();                             /* Listen to Netion display */
  }
  /* Reset WLAN Update flag for next update */
  if (minute() % 10 == 1)                             /* Reset clock update after 10 minutes */
  {
    WLAN_Update = true;                              /* Set WLAN update  flag false */
  }  
 
}
 

/****************************************************************************************/
/****************************************************************************************/
/* Basic routines                                                                       */
/****************************************************************************************/
/****************************************************************************************/

/****************************************************************************************/
/* ESP8266 WLAN: Subroutine to send one AT command to the WLAN chip and awaits response */
/*               Returns true when no error received or false when error received       */
/* AT_Command: AT command to be transfered to the ESP8266 chip                          */
/* AT_Respond: Expected AT response from ESP8266 chip                                   */ 
/****************************************************************************************/
boolean SUB_ESP_SEND_CHECK (String AT_Command, char AT_Respond[])
{
  esp8266.println(AT_Command);                    /* Send AT command to WLAN chip */ 
  if (esp8266.findUntil(AT_Respond, "ERROR"))     /* Search for response, True = correct response received */
  {
    return true;                                  /* Return with no error */
  }
  else                                            /* Error received ? */
  {
    return false;                                 /* Return with no error */
  }
}

 
/****************************************************************************************/
/* ESP8266 WLAN: Subroutine to send one AT to the WLAN chip and return the              */
/*               response as string                                                     */
/* AT_Command: AT command to be transfered to the ESP8266 chip                          */
/****************************************************************************************/
String SUB_ESP_SEND_STRING (String AT_Command)
{
  esp8266.println(AT_Command);                    /* Send AT command to ESP chip */
  return esp8266.readString();                    /* Return string during return */
}


/****************************************************************************************/
/* NEXTION DISPLAY: Subroutine to send command to Nextion display out of Flash          */
/* Send_String: String to be sent to Nextion LCD display from Flash                     */
/****************************************************************************************/

void SUB_NEXTION_SEND (const __FlashStringHelper *Send_String)
{
  nextion.print(Send_String);                     /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display to terminate string */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
}


/****************************************************************************************/
/****************************************************************************************/
/* ESP8266 WLAN Routines                                                                */
/****************************************************************************************/
/****************************************************************************************/

/****************************************************************************************/
/* ESP8266 WLAN: Subroutine to connect to WLAN                                     */
/***********************************************************************************/
void SUB_WLAN_CONNECT (void)
{
  if (ACT_SCREEN == 0)                            /* First screen ? */
  {
    SUB_NEXTION_SEND (F("p26.pic=41"));           /* Switch WLAN signal to red */
  }
  if (ACT_SCREEN == 1)                            /* Second screen ? */
  {
    SUB_NEXTION_SEND (F("p58.pic=41"));           /* Switch WLAN signal to red */
  }
  if (DEBUG)SUB_SERIAL_PRINT ("Establish WLAN connection ...\n");
  esp8266.setTimeout(5000);                       /* Set receive timeout to 5 seconds */  
  /* Step 1: Reset chip */
  if (DEBUG)SUB_SERIAL_PRINT ("Step 1: Reset chip");
  WLAN_Connect = SUB_ESP_SEND_CHECK ("AT+RST", "ready"); 
  if (WLAN_Connect == false)                      /* Error received ? */
  {
    WLAN_Connect_Error = 1;                       /* Set WLAN connect error to 1 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    esp8266.setTimeout(1000);                     /* Set receive timeout to 1 seconds */              
    Packet_Buffer = SUB_ESP_SEND_STRING("AT+GMR");/* Receive version information */
    if (DEBUG)SUB_SERIAL_PRINT ("  FW Version: ");/* Extract and dispaly FW version */
    Test_Pos_1 = Packet_Buffer.indexOf (':', 0);  /* Search for double point */
    /* Search for ) */
    Test_Pos_2 = Packet_Buffer.indexOf (')', Test_Pos_1+1);    
    /* Extract FW version */
    Temp_Buffer = Packet_Buffer.substring(Test_Pos_1+1, Test_Pos_2+1);
    if (DEBUG)Serial.println (Temp_Buffer);                 /* Display FW version */
  
    /* Step 2: Adjust WLAN mode to WLAN client */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 2: Adjust WLAN mode to WLAN client");
    WLAN_Connect = SUB_ESP_SEND_CHECK("AT+CWMODE=1", "OK");
    if (WLAN_Connect == false)                    /* Error received ? */
    {
      WLAN_Connect_Error = 2;                     /* Set WLAN connect error to 2 */
      if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
    }
  }
  if (WLAN_Connect == true)                       /* No error received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    
    /* Step 3: Connect to WLAN */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 3: Connect to WLAN ");
    if (DEBUG)Serial.print (SSID);
    esp8266.setTimeout(20000);                    /* Set receive timeout to 20 seconds */
    Packet_Buffer ="AT+CWJAP=\"";                 /* Generate connect to WLAN command */
    Packet_Buffer += SSID;
    Packet_Buffer += "\",\"";
    Packet_Buffer += PASSWORD;
    Packet_Buffer += "\"";   
    WLAN_Connect = SUB_ESP_SEND_CHECK(Packet_Buffer, "OK");
    if (WLAN_Connect == false)                    /* Error received ? */
    {
      WLAN_Connect_Error = 3;                     /* Set WLAN connect error to 3 */
      if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
    }
    else                                          /* No error received ? */
    {
      if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
      esp8266.setTimeout(1000);                   /* Set receive timeout to 1 seconds */
      /* Check for right WLAN connection */
      WLAN_Connect = SUB_ESP_SEND_CHECK("AT+CWJAP?", SSID);
      if (WLAN_Connect == false)                  /* Error received ? */
      {
        WLAN_Connect_Error = 4;                   /* Set WLAN connect error to 4 */
      }    
      else                                        /* No error received ? */
      {  
        if (DEBUG)SUB_SERIAL_PRINT ("  Asigned IP-Adresss: ");
        Packet_Buffer = SUB_ESP_SEND_STRING("AT+CIFSR"); 
        /* Search for first " */
        Test_Pos_1 = Packet_Buffer.indexOf ('"', 3);
        /* Search second " */
        Test_Pos_2 = Packet_Buffer.indexOf ('"', Test_Pos_1+1);    
        /* Extract IP address of device */
        Temp_Buffer = Packet_Buffer.substring(Test_Pos_1+1, Test_Pos_2);
        if (DEBUG)Serial.println (Temp_Buffer);             /* Print IP address */
        if (DEBUG)SUB_SERIAL_PRINT ("  MAC Adresss: ");
        /* Search for third " */
        Test_Pos_1 = Packet_Buffer.indexOf ('"', Test_Pos_2+1);
        /* Search forth " */
        Test_Pos_2 = Packet_Buffer.indexOf ('"', Test_Pos_1+1);    
        /* Extract IP address of device */
        Temp_Buffer = Packet_Buffer.substring(Test_Pos_1+1, Test_Pos_2);
        if (DEBUG)Serial.println (Temp_Buffer);    /* Print MAC address */     
      } 
    }
  }

  if (WLAN_Connect == false)                      /* Error received ? */
  {
    if (WLAN_Connect_Error == 1)                  /* Error 1 ? */
    {
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: ESP8266 WLAN chip not ready\n");
    }
    if (WLAN_Connect_Error == 2)                  /* Error 2 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to configure ESP8266 chip!\n");
    }
    if (WLAN_Connect_Error == 3)                  /* Error 3 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to connect to WLAN ");
      if (DEBUG)Serial.print (SSID);
      if (DEBUG)SUB_SERIAL_PRINT ("!\n");
    } 
    if (WLAN_Connect_Error == 4)                  /* Error 4 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Connected to wrong WLAN!\n");
    }       
  }
  else                                            /* No errors received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT ("SUCCESS: Connected to WLAN ");
    if (DEBUG)Serial.print (SSID);
    if (DEBUG)SUB_SERIAL_PRINT ("!\n");
    if (ACT_SCREEN == 0)                          /* First screen ? */
    {
      SUB_NEXTION_SEND (F("p26.pic=42"));         /* Switch WLAN signal to green */
    }
    if (ACT_SCREEN == 1)                          /* Second screen ? */
    {
      SUB_NEXTION_SEND (F("p58.pic=42"));         /* Switch WLAN signal to green */
    }
  }
}


/****************************************************************************************/
/****************************************************************************************/
/* NTP routines                                                                         */
/****************************************************************************************/
/****************************************************************************************/

/****************************************************************************************/
/* NTP-Clock: Subroutine to receive NTP time                                            */
/****************************************************************************************/
void SUB_NTP_Receive()
{
  if (ACT_SCREEN == 0)                            /* First screen ? */
  {
    SUB_NEXTION_SEND (F("p27.pic=43"));           /* Switch watch signal to red */
  }
  if (ACT_SCREEN == 1)                            /* Second screen ? */
  {
    SUB_NEXTION_SEND (F("p27.pic=43"));           /* Switch watch signal to red */
  }
  if (DEBUG)SUB_SERIAL_PRINT ("\nReceive NTP time and date ...\n");
  if (DEBUG)SUB_SERIAL_PRINT ("Waiting for first NTP sync ...\n");
  NTP_Time_Zone = 0;                              /* Initialize time zone for UPC */
  /* Close NTP connections if open */
  NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPCLOSE" , "OK");    
  /* Clear buffer */
  while ( esp8266.available() > 0)
  {
    Test_Pos_1 = esp8266.read();                  /* Read next byte */
  }
  SUB_NTP_CONNECT ();                             /* Connect to NTP server and receive time */
  if (NTP_Connect == true)                        /* No error in first reception */
  {
    /* Display UTC time */
    if (DEBUG)SUB_SERIAL_PRINT ("UTC time received:");      
    SUB_CLOCK_PRINT ();                           /* Call sub routine to send clock and date via serial interface to terminal */ 
    /* Receive NTP information for CET */
    SUB_NTP_ADJUST_EUROPE();                      /* Determine adjustment factor for CET including summer and winter time */
    delay (1000);                                 /* Wait 1 second */
    if (DEBUG)SUB_SERIAL_PRINT ("\nWaiting for second NTP sync ...\n");
    SUB_NTP_CONNECT ();                           /* Connect to NTP server and receive time */
  }
  if (NTP_Connect == true)                        /* No error in second reception */
  {  
    if (DEBUG)SUB_SERIAL_PRINT ("CET time received:");      /* Display CET time */
    if (ACT_SCREEN == 0)                          /* First screen ? */
    {
      SUB_NEXTION_SEND (F("p27.pic=44"));         /* Switch watch signal to green */
    }
    if (ACT_SCREEN == 1)                          /* Second screen ? */
    {
      SUB_NEXTION_SEND (F("p27.pic=44"));           /* Switch watch signal to green */
    }
  }
}


/****************************************************************************************/
/* NTP-Clock: Subroutine to adjust received time to Europaen CET                        */
/****************************************************************************************/
void SUB_NTP_ADJUST_EUROPE()
{
  int Begin_DST_Date;                             /* Begin date */
  int Begin_DST_Month;                            /* Begin month */
  int End_DST_Date;                               /* End date */
  int End_DST_Month;                              /* End month */
  
  Begin_DST_Date=  (31 - (5* year() /4 + 4) % 7); /* Determine last Sunday of March */
  Begin_DST_Month=3;                              /* Begin month is March */
  End_DST_Date= (31 - (5 * year() /4 + 1) % 7);   /* Determine last Sunday of October */
  End_DST_Month=10;                               /* Begin month is October */
  
  /* Check for summer time */
  if (((month() > Begin_DST_Month) && (month() < End_DST_Month))
      || ((month() == Begin_DST_Month) && (day() >= Begin_DST_Date))
      || ((month() == End_DST_Month) && (day() <= End_DST_Date)))
  {
    NTP_Time_Zone = 2;                            /* Set time zone for CET = UTC +2 hour */
  }
  else                                            /* Winter time ? */
  {
    NTP_Time_Zone = 1;                            /* Set time zone for CET = UTC +1 hour */
  }
} 


/****************************************************************************************/
/* NTP-Clock: Subroutine to connect to NTP server and receive NTP time                  */
/****************************************************************************************/
void SUB_NTP_CONNECT (void)
{
  unsigned int NTP_Port;                          /* NTP port to receive UDP packets */
  unsigned int Local_Port;                        /* Local port to listen for UDP packets */
    
  esp8266.setTimeout(1000);                       /* Set receive timeout to 1 seconds */  
  /* Step 1: Set data mode to 0 = Transparent */
  if (DEBUG)SUB_SERIAL_PRINT ("Step 1: Set data mode to 0 = Transparent");
  NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPMODE=0", "OK");
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 1;                        /* Set NTP connect error to 1 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    
    /* Step 2: Disable more than one connection */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 2: Disable more than one connection");
    NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPMUX=0", "OK");   
  }
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 1;                        /* Set NTP connect error to 1 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    esp8266.setTimeout(5000);                     /* Set receive timeout to 5 seconds */ 
    
    /* Step 3: Connect to DNS Server to get IP address of NTP server */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 3: Connect to DNS Server to get IP address of NTP server");   
    Packet_Buffer ="AT+CIPSTART=\"UDP\",\"pool.ntp.org\",123";    
    NTP_Connect = SUB_ESP_SEND_CHECK (Packet_Buffer, "OK");
  }
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 2;                        /* Set NTP connect error to 2 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
     esp8266.setTimeout(1000);                    /* Set receive timeout to 1 seconds */ 
    Packet_Buffer = SUB_ESP_SEND_STRING ("AT+CIPSTATUS"); 
    Test_Pos_2 = Packet_Buffer.indexOf (',', 0);  /* Search for first semicolon */
    /* Search for second semicolon */
    Test_Pos_1 = Packet_Buffer.indexOf (',', Test_Pos_2+1);
    /* Search for third semicolon */
    Test_Pos_2 = Packet_Buffer.indexOf (',', Test_Pos_1+1);
    /* Search for forth semicolon */
    Test_Pos_1 = Packet_Buffer.indexOf (',', Test_Pos_2+1);
    /* Extract NTP Port */
    Temp_Buffer = Packet_Buffer.substring(Test_Pos_2+1, Test_Pos_1);
    NTP_Port = Temp_Buffer.toInt();               /* Covert string to integer */
    /* Search for fifth semicolon */
    Test_Pos_2 = Packet_Buffer.indexOf (',', Test_Pos_1+1);
    /* Extract local port to listen for UDP packets */
    Temp_Buffer = Packet_Buffer.substring(Test_Pos_1+1, Test_Pos_2);
    Local_Port = Temp_Buffer.toInt();             /* Covert string to integer */
    Test_Pos_2 = Packet_Buffer.indexOf (',', 0);  /* Search for first semicolon */
    /* Search for second semicolon */
    Test_Pos_1 = Packet_Buffer.indexOf (',', Test_Pos_2+1);
    /* Search for third semicolon */
    Test_Pos_2 = Packet_Buffer.indexOf (',', Test_Pos_1+1);    
    /* Extract IP address of NTP Server */
    Temp_Buffer = Packet_Buffer.substring(Test_Pos_1+2, Test_Pos_2-1);
    /* Print received information */
    if (DEBUG)SUB_SERIAL_PRINT ("  Received IP-Adresss of NTP server: ");
    if (DEBUG)Serial.println (Temp_Buffer); 
    if (DEBUG)SUB_SERIAL_PRINT ("  Received NTP port: ");
    if (DEBUG)Serial.println (NTP_Port);
    if (DEBUG)SUB_SERIAL_PRINT ("  Received local UDP listen port: ");
    if (DEBUG)Serial.println (Local_Port);        
     
    /* Step 4: Close connection to DNS server */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 4: Close connection to DNS Server");    
    NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPCLOSE" , "OK");    
  }
  
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 1;                        /* Set NTP connect error to 1 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error received ? */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");

    /* Step 5: Connect to NTP Server */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 5: Connect to NTP Server");    
    esp8266.setTimeout(5000);                     /* Set receive timeout to 10 seconds */ 
    Packet_Buffer ="AT+CIPSTART=\"UDP\",\"";      /* Generate sending message */
    Packet_Buffer += Temp_Buffer;
    Packet_Buffer += "\",";
    Packet_Buffer += NTP_Port;
    Packet_Buffer += ",";
    Packet_Buffer += Local_Port;
    Packet_Buffer += ",0";
    NTP_Connect = SUB_ESP_SEND_CHECK (Packet_Buffer, "OK");

    esp8266.setTimeout(1000);                     /* Set receive timeout to 1 seconds */ 
  }
  
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 3;                        /* Set NTP connect error to 3 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    SUB_GET_NTP_TIME ();                          /* Sync internal time with NTP subroutine */
  }

  /* Step 8: Close connection to NTP Server */
  if (DEBUG)SUB_SERIAL_PRINT ("Step 8: Close connection to NTP Server"); 
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPCLOSE" , "OK"); 
    if (NTP_Connect == false)                     /* Error received ? */
    {
      NTP_Connect_Error = 1;                      /* Set NTP connect error to 1 */
      if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
    }
    else                                          /* No error received */
    {
      if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    }
    NTP_Connect = false;                          /* Set to error even the connection could be closed */  
  } 
  else
  {
    NTP_Connect = SUB_ESP_SEND_CHECK ("AT+CIPCLOSE" , "OK"); 
    if (NTP_Connect == false)                     /* Error received ? */
    {
      NTP_Connect_Error = 1;                      /* Set NTP connect error to 1 */
      if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
    }
    else                                          /* No error received */
    {
      if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    }  
  }
}


/****************************************************************************************/
/* NTP-Clock: Subroutine to get NTP time and sync with internal time                    */
/****************************************************************************************/
void SUB_GET_NTP_TIME(void)
{
  unsigned long Secs_Since_1900;                  /* Seconds since 1970 */
  unsigned int Receive_Timeout;                   /* Timeout waiting for received characters */

  /* Step 6: Send NTP Request */ 
  if (DEBUG)SUB_SERIAL_PRINT ("Step 6: Send NTP Request");
  Packet_Buffer ="AT+CIPSEND=";                   /* Generate sending message */
  Packet_Buffer += NTP_PACKET_SIZE;
  NTP_Connect = SUB_ESP_SEND_CHECK (Packet_Buffer, ">");
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 4;                        /* Set NTP connect error to 4 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
  }
  else                                            /* No error received */
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");
    /* Send NTP request */
    esp8266.print (char(B11100011));              /* 0: LI, Version, Mode */
    esp8266.print (char(0));                      /* 1: Stratum, or type of clock */
    esp8266.print (char(6));                      /* 2: Polling interval */
    esp8266.print (char(0xEC));                   /* 3: Peer clock precision */
    /* 8 bytes of zero for Root delay & Root dispersion */
    esp8266.print (char(0));                      /* 4 */
    esp8266.print (char(0));                      /* 5 */
    esp8266.print (char(0));                      /* 6 */
    esp8266.print (char(0));                      /* 7 */
    esp8266.print (char(0));                      /* 8 */
    esp8266.print (char(0));                      /* 9 */
    esp8266.print (char(0));                      /* 10*/
    esp8266.print (char(0));                      /* 11 */
    esp8266.print (char(49));                     /* 12 */    
    esp8266.print (char(0x4E));                   /* 13 */   
    esp8266.print (char(49));                     /* 14 */      
    esp8266.print (char(52));                     /* 15 */        
    for (Loop_Counter = 16; Loop_Counter < NTP_PACKET_SIZE; Loop_Counter++)
    {
      esp8266.print (char(0));                    /* 16-47 */    
    } 
       
    /* Step 7: Receive NTP Response */
    if (DEBUG)SUB_SERIAL_PRINT ("Step 7: Receive NTP Response");
    esp8266.setTimeout(1000);                     /* Set receive timeout to 1 seconds */
    NTP_Connect = esp8266.find("+IPD,48:");       /* Search for right string */
  }
  
  if (NTP_Connect == false)                       /* Error received ? --> No starting string found ? */
  {
    NTP_Connect_Error = 5;                        /* Set NTP connect error to 5 */
  }
  else                                            /* No error received ? --> Starting string found ? */
  {
    Loop_Counter = 0;                             /* Reset loop counter */
    Receive_Timeout = 0;                          /* Reset receiver timeout */
    NTP_Connect = false;                          /* Set NTP_Connect false */               
    while (Receive_Timeout < 5000)                /* Repeat until timeout of 500ms */
    {
      /* At least 48 bytes received ? */
      if ( esp8266.available() >= NTP_PACKET_SIZE )
      {
        NTP_Connect = true;                       /* Set NTP_Connect true */
        /* Loop for receiving bytes */
        for (Loop_Counter = 0; Loop_Counter < NTP_PACKET_SIZE; Loop_Counter++)
        {
          Test_Pos_1 = esp8266.read();            /* Read next byte */
          /* Calculate NTP time by converting four bytes starting at location 40 to a long integer */
          if (Loop_Counter == 40)                 /* First byte received ? */
          {
            Secs_Since_1900 =  (unsigned long)Test_Pos_1 << 24;
          }
          if (Loop_Counter == 41)                 /* Second byte received ? */
          {
            Secs_Since_1900 |= (unsigned long)Test_Pos_1 << 16;
          }
          if (Loop_Counter == 42)                 /* Third byte received ? */
          {
            Secs_Since_1900 |= (unsigned long)Test_Pos_1 << 8;
          }
          if (Loop_Counter == 43)                 /* Forth byte received ? */
          {
            Secs_Since_1900 |= (unsigned long)Test_Pos_1;
          }          
        } 
        if (esp8266.find( "/r/n" ))               /* Wait for end of received string to clear buffer */
        {
        }
        Receive_Timeout = 5000;                   /* Break loop */
      }
      delayMicroseconds (100);                    /* 100us delay */
      Receive_Timeout ++;                         /* Increase receive timeout */
    }
  }
  
  if (NTP_Connect == false)                       /* Error received ? */
  {
    NTP_Connect_Error = 5;                        /* Set NTP connect error to 5 */
    if (DEBUG)SUB_SERIAL_PRINT (" - Error\n");
    if (NTP_Connect_Error == 1)                   /* Error 1 ? */
    {
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to configure ESP8266 chip!\n");
    }
    if (NTP_Connect_Error == 2)                   /* Error 2 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to conect to DNS server!\n");
    }
    if (NTP_Connect_Error == 3)                   /* Error 3 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to conect to NTP server!\n");
    }
    if (NTP_Connect_Error == 4)                   /* Error 4 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Unable to send NTP request!\n");
    }
    if (NTP_Connect_Error == 5)                   /* Error 5 ? */
    {    
      if (DEBUG)SUB_SERIAL_PRINT ("ERROR: Wrong NTP response received!\n");
    }  
  }
  else
  {
    if (DEBUG)SUB_SERIAL_PRINT (" - OK\n");  
    if (DEBUG)SUB_SERIAL_PRINT ("SUCCESS: NTP data received !\n");
    /* Return time information to set time */
    setTime (Secs_Since_1900 - 2208988800UL + NTP_Time_Zone * SECS_PER_HOUR);
  }
}


/****************************************************************************************/
/****************************************************************************************/
/* Display routines                                                                     */
/****************************************************************************************/
/****************************************************************************************/

/****************************************************************************************/
/* Terminal: Subroutine to send clock and date via serial interface to terminal         */
/****************************************************************************************/
void SUB_CLOCK_PRINT()                             
{
  /* Display time */
  SUB_SERIAL_PRINT ("\nTime: ");                  /* Display time header */
  Serial.print(hour());                           /* Display hour information */
  SUB_SERIAL_PRINT (":");
  if (minute()<10)                                /* Display minute information with leading zero */
  {
    SUB_SERIAL_PRINT ("0");
  }
  Serial.print(minute());
  SUB_SERIAL_PRINT (":");
  if (second()<10)                                /* Display second information with leading zero */
  {
    SUB_SERIAL_PRINT ("0");
  }
  Serial.print(second());
  
  /* Display date */
  SUB_SERIAL_PRINT (", Date: ");                  /* Display date header */
  if (day()<10)                                   /* Display day information with leading zero */
  {
    if (DEBUG)SUB_SERIAL_PRINT ("0");
  }
  Serial.print(day());                            
  SUB_SERIAL_PRINT(".");
  if (month()<10)                                 /* Display month information with leading zero */
  {
    SUB_SERIAL_PRINT ("0");
  }
  Serial.print(month());
  SUB_SERIAL_PRINT(".");
  Serial.print(year());                           /* Display year information */
}


/****************************************************************************************/
/* Nextion display: Subroutine to clear all clock segments                              */
/****************************************************************************************/
void SUB_Clock_Clear()
{
  SUB_NEXTION_SEND (F("p0.pic=1"));               /* Disable Fünf_1 */
  SUB_NEXTION_SEND (F("p1.pic=3"));               /* Disable Zehn_1 */
  SUB_NEXTION_SEND (F("p2.pic=5"));               /* Disable Vor_1 */
  SUB_NEXTION_SEND (F("p3.pic=7"));               /* Disable Nach_1 */
  SUB_NEXTION_SEND (F("p4.pic=9"));               /* Disable Halb */
  SUB_NEXTION_SEND (F("p5.pic=11"));              /* Disable Viertel */
  SUB_NEXTION_SEND (F("p6.pic=5"));               /* Disable Vor_2 */
  SUB_NEXTION_SEND (F("p7.pic=7"));               /* Disable Nach_2 */
  SUB_NEXTION_SEND (F("p8.pic=13"));              /* Disable Ein */
  SUB_NEXTION_SEND (F("p9.pic=15"));              /* Disable Eins */
  SUB_NEXTION_SEND (F("p10.pic=17"));             /* Disable Zwei */
  SUB_NEXTION_SEND (F("p11.pic=19"));             /* Disable Drei */
  SUB_NEXTION_SEND (F("p12.pic=21"));             /* Disable Vier */
  SUB_NEXTION_SEND (F("p13.pic=23"));             /* Disable Fünf */
  SUB_NEXTION_SEND (F("p14.pic=25"));             /* Disable Sechs */
  SUB_NEXTION_SEND (F("p15.pic=27"));             /* Disable Sieben */
  SUB_NEXTION_SEND (F("p16.pic=29"));             /* Disable Acht */
  SUB_NEXTION_SEND (F("p17.pic=31"));             /* Disable Neun */
  SUB_NEXTION_SEND (F("p18.pic=3"));              /* Disable Zehn */
  SUB_NEXTION_SEND (F("p19.pic=33"));             /* Disable Elf */
  SUB_NEXTION_SEND (F("p20.pic=35"));             /* Disable Zwölf */ 
  SUB_NEXTION_SEND (F("p21.pic=37"));             /* Disable Uhr */
  SUB_NEXTION_SEND (F("p22.pic=39"));             /* Disable Punkt_1 */   
  SUB_NEXTION_SEND (F("p23.pic=39"));             /* Disable Punkt_2 */
  SUB_NEXTION_SEND (F("p24.pic=39"));             /* Disable Punkt_3 */   
  SUB_NEXTION_SEND (F("p25.pic=39"));             /* Disable Punkt_4 */
  SUB_NEXTION_SEND (F("p26.pic=41"));             /* Switch WLAN signal to red */
  SUB_NEXTION_SEND (F("p27.pic=43"));             /* Disable watch symbol */
}


/****************************************************************************************/
/* Nextion display: Subroutine to adjust the brightness of the display                  */
/****************************************************************************************/
void SUB_ADJUST_TIMEOUT (void)
{
  switch (BEL_PERIOD)                      /* Call sub menues after choosing a menu item */
  {
    case 0:                                /* 0 = Always On chosen */
      SUB_NEXTION_SEND (F("thsp=0"));      /* Set to zero for next start */ 
      break;
    case 1:                                /* 1 = 10 Seconds */
      SUB_NEXTION_SEND (F("thsp=10"));     /* Set to 10 seconds */ 
      break; 
    case 2:                                /* 2 = 20 Seconds */
      SUB_NEXTION_SEND (F("thsp=20"));     /* Set to 20 seconds */ 
      break;   
    case 3:                                /* 3 = 30 Seconds */
      SUB_NEXTION_SEND (F("thsp=30"));     /* Set to 30 seconds */ 
      break;      
    case 4:                                /* 4 = 1 Minute */
      SUB_NEXTION_SEND (F("thsp=60"));     /* Set to 1 minute */ 
      break;     
    case 5:                                /* 5 = 2 Minutes */
      SUB_NEXTION_SEND (F("thsp=120"));    /* Set to 2 minutes */ 
      break;  
    case 6:                                /* 6 = 5 Minutes */
      SUB_NEXTION_SEND (F("thsp=300"));    /* Set to 5 minutes */ 
      break;  
    case 7:                                /* 7 = 10 Minutes */
      SUB_NEXTION_SEND (F("thsp=600"));    /* Set to 10 minutes */ 
      break;  
    case 8:                                /* 8 = 20 Minutes */
      SUB_NEXTION_SEND (F("thsp=1200"));   /* Set to 20 minutes */ 
      break;  
    case 9:                                /* 9 = 30 Minutes */
      SUB_NEXTION_SEND (F("thsp=1800"));   /* Set to 30 minutes */ 
      break;  
    case 10:                               /* 10 = 1 Hour */
      SUB_NEXTION_SEND (F("thsp=6000"));   /* Set to 1 hour */ 
      break;  
    case 11:                               /* 11 = 4 Hour */
      SUB_NEXTION_SEND (F("thsp=24000"));  /* Set to 4 hour */ 
      break; 
  }                     
}


/****************************************************************************************/
/* Nextion display: Subroutine to adjust the brightness of the display                  */
/****************************************************************************************/
void SUB_ADJUST_BRIGHT(void)
{
  /* Set actual dim value */
  Packet_Buffer ="dim=";
  Packet_Buffer += BEL_INT;
  nextion.print(Packet_Buffer);                  /* Send string message to Nextion display */
  nextion.print(char (0xFF));                    /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));  
  /* Set startup dim value */
  Packet_Buffer ="dims=";
  Packet_Buffer += BEL_INT;
  nextion.print(Packet_Buffer);                  /* Send string message to Nextion display */
  nextion.print(char (0xFF));                    /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF)); 
}


/****************************************************************************************/
/* Nextion display: Subroutine to update time and date on Nextion display 1             */
/****************************************************************************************/
void SUB_CLOCK_1_DISPLAY()
{
  int Check_Time;                                 /* Used for checks on time */
  int Check_Date;                                 /* Used for checks on date */

  
  /* Update minute information */
  Check_Time = minute() % 5;                      /* Modulo 5 check */
  if (Check_Time == 0)                            /* Full 5 minutes ? */
  {
    SUB_NEXTION_SEND (F("p22.pic=39"));           /* Disable Punkt_1 */   
    SUB_NEXTION_SEND (F("p23.pic=39"));           /* Disable Punkt_2 */
    SUB_NEXTION_SEND (F("p24.pic=39"));           /* Disable Punkt_3 */   
    SUB_NEXTION_SEND (F("p25.pic=39"));           /* Disable Punkt_4 */ 
  }
  if (Check_Time == 1)                            /* Minute 1 or 6 */
  {
    SUB_NEXTION_SEND (F("p22.pic=40"));           /* Enable Punkt_1 */   
    SUB_NEXTION_SEND (F("p23.pic=39"));           /* Disable Punkt_2 */
    SUB_NEXTION_SEND (F("p24.pic=39"));           /* Disable Punkt_3 */   
    SUB_NEXTION_SEND (F("p25.pic=39"));           /* Disable Punkt_4 */ 
  }
  if (Check_Time == 2)                            /* Minute 2 or 7 */
  {
    SUB_NEXTION_SEND (F("p22.pic=40"));           /* Enable Punkt_1 */   
    SUB_NEXTION_SEND (F("p23.pic=40"));           /* Enable Punkt_2 */
    SUB_NEXTION_SEND (F("p24.pic=39"));           /* Disable Punkt_3 */   
    SUB_NEXTION_SEND (F("p25.pic=39"));           /* Disable Punkt_4 */ 
  }  
  if (Check_Time == 3)                            /* Minute 3 or 8 */
  {
    SUB_NEXTION_SEND (F("p22.pic=40"));           /* Enable Punkt_1 */   
    SUB_NEXTION_SEND (F("p23.pic=40"));           /* Enable Punkt_2 */
    SUB_NEXTION_SEND (F("p24.pic=39"));           /* Enable Punkt_3 */   
    SUB_NEXTION_SEND (F("p25.pic=40"));           /* Disable Punkt_4 */
  }    
  if (Check_Time == 4)                            /* Minute 4 or 9 */
  {
    SUB_NEXTION_SEND (F("p22.pic=40"));           /* Enable Punkt_1 */   
    SUB_NEXTION_SEND (F("p23.pic=40"));           /* Enable Punkt_2 */
    SUB_NEXTION_SEND (F("p24.pic=40"));           /* Enable Punkt_3 */   
    SUB_NEXTION_SEND (F("p25.pic=40"));           /* Enable Punkt_4 */
  }
  
  
  /* Update 5 minute information */
  Check_Time = minute();                          /* Save actual minute information */
  if ((Check_Time >= 0) && (Check_Time < 5))      /* Minutes 0-4 ? */
  {
    SUB_NEXTION_SEND (F("p0.pic=1"));             /* Disable Fünf_1 */
    SUB_NEXTION_SEND (F("p2.pic=5"));             /* Disable Vor_1 */ 
    SUB_NEXTION_SEND (F("p21.pic=38"));           /* Ensable Uhr */
  }
  if ((Check_Time >= 5) && (Check_Time < 10))     /* Minutes 5-9 ? */
  {
    SUB_NEXTION_SEND (F("p21.pic=37"));           /* Disable Uhr */
    SUB_NEXTION_SEND (F("p0.pic=2"));             /* Enable Fünf_1 */
    SUB_NEXTION_SEND (F("p3.pic=8"));             /* Enable Nach_1 */
  }
  if ((Check_Time >= 10) && (Check_Time < 15))    /* Minutes 10-14 ? */
  {
    SUB_NEXTION_SEND (F("p0.pic=1"));             /* Disable Fünf_1 */
    SUB_NEXTION_SEND (F("p1.pic=4"));             /* Enable Zehn_1 */
    SUB_NEXTION_SEND (F("p3.pic=8"));             /* Enable Nach_1 */
  }  
  if ((Check_Time >= 15) && (Check_Time < 20))    /* Minutes 15-19 ? */
  {
    SUB_NEXTION_SEND (F("p1.pic=3"));             /* Disable Zehn_1 */
    SUB_NEXTION_SEND (F("p3.pic=7"));             /* Disable Nach_1 */
    SUB_NEXTION_SEND (F("p5.pic=12"));            /* Enable Viertel */
    SUB_NEXTION_SEND (F("p7.pic=8"));             /* Enable Nach_2 */
  }  
  if ((Check_Time >= 20) && (Check_Time < 25))    /* Minutes 20-24 ? */
  {
    SUB_NEXTION_SEND (F("p5.pic=11"));            /* Disable Viertel */
    SUB_NEXTION_SEND (F("p7.pic=7"));             /* Disable Nach_2 */
    SUB_NEXTION_SEND (F("p1.pic=4"));             /* Enable Zehn_1 */
    SUB_NEXTION_SEND (F("p2.pic=6"));             /* Enable Vor_1 */
    SUB_NEXTION_SEND (F("p4.pic=10"));            /* Enable Halb */  
  } 
  if ((Check_Time >= 25) && (Check_Time < 30))    /* Minutes 25-29 ? */
  {
    SUB_NEXTION_SEND (F("p1.pic=3"));             /* Disable Zehn_1 */
    SUB_NEXTION_SEND (F("p0.pic=2"));             /* Enable Fünf_1 */
    SUB_NEXTION_SEND (F("p2.pic=6"));             /* Enable Vor_1 */
    SUB_NEXTION_SEND (F("p4.pic=10"));            /* Enable Halb */    
  } 
  if ((Check_Time >= 30) && (Check_Time < 35))    /* Minutes 30-34 ? */
  {
    SUB_NEXTION_SEND (F("p2.pic=5"));             /* Disable Vor_1 */  
    SUB_NEXTION_SEND (F("p0.pic=1"));             /* Disable Fünf_1 */
    SUB_NEXTION_SEND (F("p4.pic=10"));            /* Enable Halb */     
  } 
  if ((Check_Time >= 35) && (Check_Time < 40))    /* Minutes 35-39 ? */
  {
    SUB_NEXTION_SEND (F("p0.pic=2"));             /* Enable Fünf_1 */ 
    SUB_NEXTION_SEND (F("p3.pic=8"));             /* Enable Nach_1 */
    SUB_NEXTION_SEND (F("p4.pic=10"));            /* Enable Halb */
  }     
  if ((Check_Time >= 40) && (Check_Time < 45))    /* Minutes 40-44 ? */
  {
    SUB_NEXTION_SEND (F("p0.pic=1"));             /* Disable Fünf_1 */
    SUB_NEXTION_SEND (F("p1.pic=4"));             /* Enable Zehn_1 */
    SUB_NEXTION_SEND (F("p3.pic=8"));             /* Enable Nach_1 */
    SUB_NEXTION_SEND (F("p4.pic=10"));            /* Enable Halb */
  }   
  if ((Check_Time >= 45) && (Check_Time < 50))    /* Minutes 45-49 ? */
  {
    SUB_NEXTION_SEND (F("p1.pic=3"));             /* Disable Zehn_1 */
    SUB_NEXTION_SEND (F("p3.pic=7"));             /* Disable Nach_1 */
    SUB_NEXTION_SEND (F("p4.pic=9"));             /* Disable Halb */
    SUB_NEXTION_SEND (F("p5.pic=12"));            /* Enable Viertel */
    SUB_NEXTION_SEND (F("p6.pic=6"));             /* Enable Vor_2 */
  }     
  if ((Check_Time >= 50) && (Check_Time < 55))    /* Minutes 50-54 ? */
  {    
    SUB_NEXTION_SEND (F("p5.pic=11"));            /* Disable Viertel */
    SUB_NEXTION_SEND (F("p6.pic=5"));             /* Disable Vor_2 */
    SUB_NEXTION_SEND (F("p1.pic=4"));             /* Enable Zehn_1 */
    SUB_NEXTION_SEND (F("p2.pic=6"));             /* Enable Vor_1 */
  }   
  if ((Check_Time >= 55) && (Check_Time <= 59))   /* Minutes 55-59 ? */
  {    
    SUB_NEXTION_SEND (F("p1.pic=3"));             /* Disable Zehn_1 */
    SUB_NEXTION_SEND (F("p0.pic=2"));             /* Enable Fünf_1 */
    SUB_NEXTION_SEND (F("p2.pic=6"));             /* Enable Vor_1 */
  }  

  /* Update hour information */
  Check_Time = hour();                            /* Save actual hour information */
  if (minute() < 20)                              /* Display of actual hour ? */
  {
    if ((Check_Time == 0) || (Check_Time == 12))  /* 12 o'clock */
    {
      SUB_NEXTION_SEND (F("p19.pic=33"));         /* Disable Elf */
      SUB_NEXTION_SEND (F("p20.pic=36"));         /* Enable Zwölf */
    }
    if ((Check_Time == 1) || (Check_Time == 13))  /* 1 o'clock */
    {
      SUB_NEXTION_SEND (F("p20.pic=35"));         /* Disable Zwölf */
      if ((minute() >= 0) && (minute() < 5))      /* Minutes 0-4 ? */
      {
        SUB_NEXTION_SEND (F("p8.pic=14"));        /* Enable Ein */
      }
      else                                        /* Minutes > 5 */
      {
        SUB_NEXTION_SEND (F("p8.pic=14"));        /* Enable Ein */
        SUB_NEXTION_SEND (F("p9.pic=16"));        /* Enable Eins */
      }
    }
    if ((Check_Time == 2) || (Check_Time == 14))  /* 2 o'clock */
    {
      SUB_NEXTION_SEND (F("p8.pic=13"));          /* Disable Ein */
      SUB_NEXTION_SEND (F("p9.pic=15"));          /* Disable Eins */
      SUB_NEXTION_SEND (F("p10.pic=18"));         /* Enable Zwei */
    }    
    if ((Check_Time == 3) || (Check_Time == 15))  /* 3 o'clock */
    {
      SUB_NEXTION_SEND (F("p10.pic=17"));         /* Disable Zwei */
      SUB_NEXTION_SEND (F("p11.pic=20"));         /* Enable Drei */
    }    
    if ((Check_Time == 4) || (Check_Time == 16))  /* 4 o'clock */
    {
      SUB_NEXTION_SEND (F("p11.pic=19"));         /* Disable Drei */
      SUB_NEXTION_SEND (F("p12.pic=22"));         /* Enable Vier */
    }  
    if ((Check_Time == 5) || (Check_Time == 17))  /* 5 o'clock */
    {
      SUB_NEXTION_SEND (F("p12.pic=21"));         /* Disable Vier */
      SUB_NEXTION_SEND (F("p13.pic=24"));         /* Enable Fünf */
    }
    if ((Check_Time == 6) || (Check_Time == 18))  /* 6 o'clock */
    {
      SUB_NEXTION_SEND (F("p13.pic=23"));         /* Disable Fünf */
      SUB_NEXTION_SEND (F("p14.pic=26"));         /* Enable Sechs */
    }
    if ((Check_Time == 7) || (Check_Time == 19))  /* 7 o'clock */
    {
      SUB_NEXTION_SEND (F("p14.pic=25"));         /* Disable Sechs */
      SUB_NEXTION_SEND (F("p15.pic=28"));         /* Enable Sieben */
    }
    if ((Check_Time == 8) || (Check_Time == 20))  /* 8 o'clock */
    {
      SUB_NEXTION_SEND (F("p15.pic=27"));         /* Disable Sieben */
      SUB_NEXTION_SEND (F("p16.pic=30"));         /* Enable Acht */
    }        
    if ((Check_Time == 9) || (Check_Time == 21))  /* 9 o'clock */
    {
      SUB_NEXTION_SEND (F("p16.pic=29"));         /* Disable Acht */  
      SUB_NEXTION_SEND (F("p17.pic=32"));         /* Enable Neun */
    }
    if ((Check_Time == 10) || (Check_Time == 22)) /* 10 o'clock */
    {
      SUB_NEXTION_SEND (F("p17.pic=31"));         /* Disable Neun */
      SUB_NEXTION_SEND (F("p18.pic=4"));          /* Enable Zehn */
    }
    if ((Check_Time == 11) || (Check_Time == 23)) /* 11 o'clock */
    {
      SUB_NEXTION_SEND (F("p18.pic=3"));          /* Disable Zehn */
      SUB_NEXTION_SEND (F("p19.pic=34"));         /* Enable Elf */
    }
  }   
  else                                            /* Display of next hour ? */
  {
    if ((Check_Time == 0) || (Check_Time == 12))  /* 12 o'clock */
    {
      SUB_NEXTION_SEND (F("p20.pic=35"));         /* Disable Zwölf */
      SUB_NEXTION_SEND (F("p8.pic=14"));          /* Enable Ein */
      SUB_NEXTION_SEND (F("p9.pic=16"));          /* Enable Eins */
    }
    if ((Check_Time == 1) || (Check_Time == 13))  /* 1 o'clock */
    {
      SUB_NEXTION_SEND (F("p8.pic=13"));          /* Disable Ein */
      SUB_NEXTION_SEND (F("p9.pic=15"));          /* Disable Eins */
      SUB_NEXTION_SEND (F("p10.pic=18"));         /* Enable Zwei */
    }
    if ((Check_Time == 2) || (Check_Time == 14))  /* 2 o'clock */
    {
      SUB_NEXTION_SEND (F("p10.pic=17"));         /* Disable Zwei */
      SUB_NEXTION_SEND (F("p11.pic=20"));         /* Enable Drei */
    }    
    if ((Check_Time == 3) || (Check_Time == 15))  /* 3 o'clock */
    {
      SUB_NEXTION_SEND (F("p11.pic=19"));         /* Disable Drei */
      SUB_NEXTION_SEND (F("p12.pic=22"));         /* Enable Vier */
    }    
    if ((Check_Time == 4) || (Check_Time == 16))  /* 4 o'clock */
    {
      SUB_NEXTION_SEND (F("p12.pic=21"));         /* Disable Vier */
      SUB_NEXTION_SEND (F("p13.pic=24"));         /* Enable Fünf */
    }  
    if ((Check_Time == 5) || (Check_Time == 17))  /* 5 o'clock */
    {
      SUB_NEXTION_SEND (F("p13.pic=23"));         /* Disable Fünf */
      SUB_NEXTION_SEND (F("p14.pic=26"));         /* Enable Sechs */
    }
    if ((Check_Time == 6) || (Check_Time == 18))  /* 6 o'clock */
    {
      SUB_NEXTION_SEND (F("p14.pic=25"));         /* Disable Sechs */
      SUB_NEXTION_SEND (F("p15.pic=28"));         /* Enable Sieben */
    }
    if ((Check_Time == 7) || (Check_Time == 19))  /* 7 o'clock */
    {
      SUB_NEXTION_SEND (F("p15.pic=27"));         /* Disable Sieben */
      SUB_NEXTION_SEND (F("p16.pic=30"));         /* Enable Acht */
    }
    if ((Check_Time == 8) || (Check_Time == 20))  /* 8 o'clock */
    {
      SUB_NEXTION_SEND (F("p16.pic=29"));         /* Disable Acht */  
      SUB_NEXTION_SEND (F("p17.pic=32"));         /* Enable Neun */
    }        
    if ((Check_Time == 9) || (Check_Time == 21))  /* 9 o'clock */
    {
      SUB_NEXTION_SEND (F("p17.pic=31"));         /* Disable Neun */
      SUB_NEXTION_SEND (F("p18.pic=4"));          /* Enable Zehn */
    }
    if ((Check_Time == 10) || (Check_Time == 22)) /* 10 o'clock */
    {
      SUB_NEXTION_SEND (F("p18.pic=3"));          /* Disable Zehn */
      SUB_NEXTION_SEND (F("p19.pic=34"));         /* Enable Elf */
    }
    if ((Check_Time == 11) || (Check_Time == 23)) /* 11 o'clock */
    {
      SUB_NEXTION_SEND (F("p19.pic=33"));         /* Disable Elf */
      SUB_NEXTION_SEND (F("p20.pic=36"));         /* Enable Zwölf */
    }
  }

  /* Update date information */
  Check_Time = day();                             /* Save day information in variable */
  Check_Date = 45 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p28.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 45 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p29.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  
  Check_Time = month();                           /* Save month information in variable */
  Check_Date = 45 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p31.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 45 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p32.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  Check_Time = year();                            /* Save year information in variable */
  Check_Date = 45 + (Check_Time / 1000);          /* Determine correct offset for thousand digit */
  Packet_Buffer ="p34.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 45) * 1000;          /* Calculate correct offset */
  Check_Time = Check_Time - Check_Date;           /* Calculate new value */
  Check_Date = 45 + (Check_Time / 100);           /* Determine correct offset for hundret digit */
  Packet_Buffer ="p35.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 45) * 100;           /* Calculate correct offset */
  Check_Time = Check_Time -Check_Date;           /* Calculate new value */
  Check_Date = 45 + (Check_Time / 10);            /* Determine correct offset for tens digit */
  Packet_Buffer ="p36.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 45) * 10;            /* Calculate correct offset */
  Check_Time = Check_Time - Check_Date;           /* Calculate new value */
  Check_Date = 45 + Check_Time;                   /* Determine correct offset for one digit */
  Packet_Buffer ="p37.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  if (WLAN_Connect == true)                       /* WLAN connected ? */
  {
    SUB_NEXTION_SEND (F("p26.pic=42"));           /* Switch WLAN signal to green */
  }
  else 
  {
    SUB_NEXTION_SEND (F("p26.pic=41"));           /* Switch WLAN signal to red */
  }  
  if (NTP_Connect == true)                        /* No error in second reception */
  {  
    SUB_NEXTION_SEND (F("p27.pic=44"));           /* Switch watch signal to green */
  } 
  else
  {  
    SUB_NEXTION_SEND (F("p27.pic=43"));           /* Switch watch signal to red */
  }    
}


/****************************************************************************************/
/* Nextion display: Subroutine to update time and date on Nextion display 2             */
/****************************************************************************************/
void SUB_CLOCK_2_DISPLAY()
{
  int Check_Time;                                 /* Used for checks on time */
  int Check_Date;                                 /* Used for checks on date */

  
  /* Update hour information */
  Check_Time = hour();                            /* Save actual hour information */
  Check_Date = 56 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p40.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 56 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p41.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  /* Update minute information */
  Check_Time = minute();                          /* Save actual minute information */
  Check_Date = 56 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p43.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 56 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p44.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  /* Update second information */
  Check_Time = second();                          /* Save actual second information */
  Check_Date = 56 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p46.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 56 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p47.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  /* Update date information */
  Check_Time = day();                             /* Save day information in variable */
  Check_Date = 67 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p48.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 67 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p49.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  
  Check_Time = month();                           /* Save month information in variable */
  Check_Date = 67 + ((Check_Time / 10) % 10);     /* Determine correct offset for tens digit */
  Packet_Buffer ="p51.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = 67 + (Check_Time % 10);            /* Determine correct offset for one digit */
  Packet_Buffer ="p52.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  Check_Time = year();                            /* Save year information in variable */
  Check_Date = 67 + (Check_Time / 1000);          /* Determine correct offset for thousand digit */
  Packet_Buffer ="p54.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 67) * 1000;          /* Calculate correct offset */
  Check_Time = Check_Time - Check_Date;           /* Calculate new value */
  Check_Date = 67 + (Check_Time / 100);           /* Determine correct offset for hundret digit */
  Packet_Buffer ="p55.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 67) * 100;           /* Calculate correct offset */
  Check_Time = Check_Time -Check_Date;            /* Calculate new value */
  Check_Date = 67 + (Check_Time / 10);            /* Determine correct offset for tens digit */
  Packet_Buffer ="p56.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  Check_Date = (Check_Date - 67) * 10;            /* Calculate correct offset */
  Check_Time = Check_Time - Check_Date;           /* Calculate new value */
  Check_Date = 67 + Check_Time;                   /* Determine correct offset for one digit */
  Packet_Buffer ="p57.pic=";
  Packet_Buffer += Check_Date;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
  if (WLAN_Connect == true)                       /* WLAN connected ? */
  {
    SUB_NEXTION_SEND (F("p58.pic=42"));           /* Switch WLAN signal to green */
  }
  else 
  {
    SUB_NEXTION_SEND (F("p58.pic=41"));           /* Switch WLAN signal to red */
  }  
  if (NTP_Connect == true)                        /* No error in second reception */
  {  
    SUB_NEXTION_SEND (F("p59.pic=44"));           /* Switch watch signal to green */
  } 
  else
  {  
    SUB_NEXTION_SEND (F("p59.pic=43"));           /* Switch watch signal to red */
  }   
}


/****************************************************************************************/
/* Nextion display: Subroutine to config lightening and lightening time                 */
/****************************************************************************************/
void SUB_CONFIG_MENU()
{
  String Receive_Message;                         /* Buffer for received string */
  byte Update_Value;                              /* Value to correctly update the dosplay light timept */
  boolean Break_Flag;                             /* Break flag, false = Home not pressed, true = Home pressed */

  Break_Flag = false;                             /* Set break flag false */

  /* Update light progess bar */
  Packet_Buffer ="Regler_1.val=";
  Packet_Buffer += BEL_INT;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));

  /* Update light timeout bar */
  Packet_Buffer ="Anzeige_1.pic=";
  Update_Value = BEL_PERIOD + 87;                 /* Calculate right offset */
  Packet_Buffer += Update_Value;
  nextion.print(Packet_Buffer);                   /* Send string message to Nextion display */
  nextion.print(char (0xFF));                     /* Send final three FF to Nextion display */
  nextion.print(char (0xFF));
  nextion.print(char (0xFF));
   
  do                                              /* Repeat until Home key pressed */
  {
    Receive_Message = myNextion.listen();         /* Receive messages for key pressed */
    /* Home key pressed ? */
    if (Receive_Message == "65 3 1 0 ffff ffff ffff")     
    {
      Break_Flag = true;                          /* Set break flag true */
    }
    /* Light down key pressed ? */
    if (Receive_Message == "65 3 3 0 ffff ffff ffff")   
    {
      if (BEL_INT >= 5)                           /* Not lower limit reached ? */
      {
        BEL_INT -= 5;   
      }
      /* Update light progess bar */
      Packet_Buffer ="Regler_1.val=";
      Packet_Buffer += BEL_INT;
      nextion.print(Packet_Buffer);               /* Send string message to Nextion display */
      nextion.print(char (0xFF));                 /* Send final three FF to Nextion display */
      nextion.print(char (0xFF));
      nextion.print(char (0xFF));

      SUB_ADJUST_BRIGHT();                        /* Send actual brightness value to display */

    }
    /* Light up key pressed ? */
    if (Receive_Message == "65 3 2 0 ffff ffff ffff")   
    {
      if (BEL_INT <= 95)                          /* Not upper limit reached ? */
      {
        BEL_INT += 5;   
      }
      /* Update light progess bar */
      Packet_Buffer ="Regler_1.val=";
      Packet_Buffer += BEL_INT;
      nextion.print(Packet_Buffer);               /* Send string message to Nextion display */
      nextion.print(char (0xFF));                 /* Send final three FF to Nextion display */
      nextion.print(char (0xFF));
      nextion.print(char (0xFF));

      SUB_ADJUST_BRIGHT ();                       /* Send actual brightness value to display */      
    }
    
    /* Timeout down key pressed ? */
    if (Receive_Message == "65 3 4 0 ffff ffff ffff")   
    {
      if (BEL_PERIOD >= 1)                        /* Not lower limit reached ? */
      {
        BEL_PERIOD -= 1;   
      }
      /* Update light timeout bar */
      Packet_Buffer ="Anzeige_1.pic=";
      Update_Value = BEL_PERIOD + 87;             /* Calculate right offset */
      Packet_Buffer += Update_Value;
      nextion.print(Packet_Buffer);               /* Send string message to Nextion display */
      nextion.print(char (0xFF));                 /* Send final three FF to Nextion display */
      nextion.print(char (0xFF));
      nextion.print(char (0xFF));

      SUB_ADJUST_TIMEOUT ();                      /* Call Subroutine to adjust timeout */
    }

    /* Timeout up key pressed ? */
    if (Receive_Message == "65 3 5 0 ffff ffff ffff")   
    {
      if (BEL_PERIOD <= 10)                       /* Not lower limit reached ? */
      {
        BEL_PERIOD += 1;   
      }
      /* Update picture for timeout value */
      Packet_Buffer ="Anzeige_1.pic=";
      Update_Value = BEL_PERIOD + 87;             /* Calculate right offset */
      Packet_Buffer += Update_Value;
      nextion.print(Packet_Buffer);               /* Send string message to Nextion display */
      nextion.print(char (0xFF));                 /* Send final three FF to Nextion display */
      nextion.print(char (0xFF));
      nextion.print(char (0xFF));
      
      SUB_ADJUST_TIMEOUT ();                      /* Call Subroutine to adjust timeout */
    }
    
  }
  while (Break_Flag == false);
  EEPROM.write(0, BEL_INT);                       /* Save brightness value in EEPROM */  
  EEPROM.write(1, BEL_PERIOD);                    /* Save timeout value in EEPROM */  
  SUB_NEXTION_SEND (F("page Uhr1"));              /* Display screen 1 */
  delay (300);                                    /* 300 ms delay to update screen */
}
